# -*- coding: utf-8 -*-

import os
import tarfile
from datetime import datetime, timedelta

from sandbox import sdk2

import sandbox.projects.resource_types as rt
import sandbox.projects.geosearch.resource_types as geotypes
import sandbox.common.types.resource as ctr
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.geobase.Geodata4BinStable.resource import GEODATA4BIN_STABLE


class BuildMapsGeoStaticFactors(sdk2.Task):
    """ Generate yt table with geosearch static factors """

    class Requirements(sdk2.Requirements):
        environments = (PipEnvironment("yandex-yt", use_wheel=True), )

    class Parameters(sdk2.Parameters):
        with sdk2.parameters.Group("Stable resources") as resource_block:
            binary_id = sdk2.parameters.Resource(
                'gen static factors binary',
                resource_type=rt.GEOSEARCH_STATIC_FACTORS_EXECUTABLE,
                state=(ctr.State.READY, ),
            )
            geobase_id = sdk2.parameters.Resource(
                'geodata4.bin',
                resource_type=GEODATA4BIN_STABLE,
                state=(ctr.State.READY, ),
            )
            canonizer_dump = sdk2.parameters.Resource(
                'canonizer dump',
                resource_type=geotypes.GEOSEARCH_CANONIZER_DUMP,
                state=(ctr.State.READY, ),
            )
        with sdk2.parameters.Group("Options") as options_block:
            yt_server = sdk2.parameters.String(
                'Web annotations YT server',
                default='hahn'
            )
            yt_table_input = sdk2.parameters.String(
                'YT input table',
                default='//home/sprav/altay/prod/export/exported/company'
            )
            yt_table_output = sdk2.parameters.String(
                'YT output table',
                default='//tmp/count_factors/output'
            )
            yt_output_path = sdk2.parameters.String(
                'YT older versions path'
            )
            ttl_days = sdk2.parameters.Integer(
                'Build artifacts TTL in days',
                default_value=7
            )
            yt_table_ratings = sdk2.parameters.String(
                'YT ratings table',
                default="//home/geosearch/user_sessions_features/yandex_search/biz_clicks_org1_static.{latest_state}"
            )
            yt_latest_state_path = sdk2.parameters.String(
                'Latest state attr',
                default='//home/geosearch/user_sessions_features/yandex_search/@latest_state'
            )

    def get_geobase(self):
        geobase = sdk2.ResourceData(self.Parameters.geobase_id)
        with tarfile.open(str(geobase.path), 'r') as tar:
            tar.extractall()
        return os.path.abspath('geodata4.bin')

    def init_rating(self):
        if self.Parameters.yt_table_ratings.find('{latest_state}') == -1:
            return
        from yt.wrapper import YtClient
        yt_token = sdk2.Vault.data('GEOMETA-SEARCH', 'yt-token')
        yt_client = YtClient(proxy=self.Parameters.yt_server, token=yt_token)
        ratings_meta = yt_client.get(self.Parameters.yt_latest_state_path)
        self.table_ratings = self.Parameters.yt_table_ratings.format(latest_state=ratings_meta)

    def on_execute(self):
        bin_res = sdk2.ResourceData(self.Parameters.binary_id)
        geodata_path = self.get_geobase()
        self.init_rating()
        canonizer = sdk2.ResourceData(self.Parameters.canonizer_dump).path

        yt_token = sdk2.Vault.data('GEOMETA-SEARCH', 'yt-token')
        os.environ['YT_TOKEN'] = yt_token

        if self.Parameters.yt_output_path:
            output_table_name = '{out_path}/count_factors_{task_id}'.format(out_path=self.Parameters.yt_output_path, task_id=self.id)
        else:
            output_table_name = self.Parameters.yt_table_output

        cmd = [
            str(bin_res.path),
            '-s', self.Parameters.yt_server,
            '-i', self.Parameters.yt_table_input,
            '-o', output_table_name,
            '-r', self.table_ratings,
            '-g', geodata_path,
            '-c', str(canonizer),
        ]

        with sdk2.helpers.ProcessLog(self, logger="count_factors") as l:
            sp.check_call(cmd, stdout=l.stdout, stderr=l.stderr)

        if self.Parameters.yt_output_path:
            import yt.wrapper as yt
            yt.config['token'] = yt_token
            yt.config['proxy']['url'] = 'hahn.yt.yandex.net'
            if self.Parameters.ttl_days is not None:
                ts = datetime.now()
                ts += timedelta(days=self.Parameters.ttl_days)
                yt.set_attribute(output_table_name, 'expiration_time', ts.isoformat())
            yt.link(output_table_name, self.Parameters.yt_table_output, force=True)
