# -*- coding: utf-8 -*-

from sandbox.projects import resource_types
from sandbox.projects.common.geosearch.indexing import BuildHelper
from sandbox.projects.common.utils import get_or_default

from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.parameters import SandboxStringParameter, LastReleasedResource
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.task import SandboxTask

INDEXING_OPTIONS_GROUP = 'Indexing Options'
DATA_SOURCES_GROUP = 'Data Sources'


class BusinessUrlsParameter(LastReleasedResource):
    name = 'business_urls_resource_id'
    description = 'Business urls: '
    resource_type = resource_types.MAPS_WEB_URLS_BUSINESS
    group = DATA_SOURCES_GROUP


class PoiUrlsParameter(LastReleasedResource):
    name = 'poi_urls_resource_id'
    description = 'Poi urls: '
    resource_type = resource_types.MAPS_WEB_URLS_POI
    group = DATA_SOURCES_GROUP


class IndexerParameter(LastReleasedResource):
    name = 'indexer_resource_id'
    description = 'Indexer executable: '
    resource_type = resource_types.GEOWEBANNINDEXER_EXECUTABLE
    group = INDEXING_OPTIONS_GROUP


class MrFilterParameter(LastReleasedResource):
    name = 'mr_filter_resource_id'
    description = 'MrFilter executable: '
    resource_type = resource_types.GEOWEBANNMRFILTER_EXECUTABLE
    group = INDEXING_OPTIONS_GROUP


class YtServer(SandboxStringParameter):
    name = 'yt_server'
    description = 'Web annotations YT server'
    group = INDEXING_OPTIONS_GROUP
    default_value = "banach"


class YtTable(SandboxStringParameter):
    name = 'yt_table'
    description = 'Web annotations YT table. Format: //path/to/{variable}/table'
    group = INDEXING_OPTIONS_GROUP
    default_value = "//home/jupiter/delivery_snapshot/{production_state}/UserData/IndexAnn"


class BuildMapsWebIndexAnn(SandboxTask):
    """
        Build maps web indexann
    """
    type = 'BUILD_MAPS_WEB_INDEXANN'

    input_parameters = (BusinessUrlsParameter,
                        PoiUrlsParameter,
                        IndexerParameter,
                        MrFilterParameter,
                        YtServer,
                        YtTable)

    environment = (environments.PipEnvironment('yandex-yt'), )

    def on_execute(self):
        # MR filter
        yt_server, annotations_table = self.ctx[YtServer.name], self.ctx[YtTable.name]
        if len(yt_server) == 0 or len(annotations_table) == 0:
            raise SandboxTaskFailureError('Invalid table name')

        # import available only on execute
        from yt.wrapper import YtClient
        yt_client = YtClient(proxy=yt_server)
        jupiter_meta = yt_client.get('//home/jupiter/@jupiter_meta')
        self.ctx[YtTable.name] = annotations_table = annotations_table.format(**jupiter_meta)

        business_urls_file = self.sync_resource(get_or_default(self.ctx, BusinessUrlsParameter))
        wiki_urls_file = self.sync_resource(get_or_default(self.ctx, PoiUrlsParameter))
        mr_filter = self.sync_resource(get_or_default(self.ctx, MrFilterParameter))

        temp_file = 'IndexAnn.zz'
        yt_token = self.get_vault_data('GEOMETA-SEARCH', 'yt-token')
        yt_tokenfile = 'yt_token.txt'
        with open(yt_tokenfile, 'w') as output:
            output.write(yt_token)

        run_process([
            mr_filter,
            '-s', yt_server,
            '-u', business_urls_file,
            '-u', wiki_urls_file,
            '-i', annotations_table,
            '-o', temp_file,
            '-t', yt_tokenfile],
            log_prefix='mr_filter')

        # Indexing
        index_dir = 'index'
        indexer = self.sync_resource(
            get_or_default(self.ctx, IndexerParameter))
        run_process([indexer, '-i', temp_file, '-o', index_dir], log_prefix='indexer')
        BuildHelper.create_index_resource(self,
                                          primary_source_resource_id=self.ctx.get(BusinessUrlsParameter.name, None),
                                          index_directory=index_dir,
                                          index_type=resource_types.MAPS_WEB_INDEXANN)


__Task__ = BuildMapsWebIndexAnn
