# -*- coding: utf-8 -*-

import requests
import os
import subprocess

from sandbox.projects import resource_types
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxGitUrlParameter
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk import environments
from sandbox.projects.common import environments as common_environments
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError


class RepoUrlParameter(SandboxGitUrlParameter):
    name = 'repo_url'
    description = 'Repo url:'
    default_value = 'https://github.yandex-team.ru/greyevil/mediaportraits.git'
    required = True


class StaticVersionParameter(SandboxStringParameter):
    name = 'static_version'
    description = 'Static version:'
    default_value = 'latest'
    required = True


class BuildMediaportraitsFrontend(SandboxTask, object):
    """
        Фронтенд Медиапортретов на nodejs
    """
    type = 'BUILD_MEDIAPORTRAITS_FRONTEND'

    environment = (
        environments.SvnEnvironment(),
        common_environments.SandboxNodeNpmEnvironment('0.10.26'),
    )

    input_parameters = (RepoUrlParameter, StaticVersionParameter, )

    RESOURCE_TYPE = resource_types.MEDIAPORTRAITS_FRONTEND_PACKAGE  # тип собираемого ресурса

    BUILD_SCRIPT_SVN = 'arcadia:/arc/trunk/arcadia/yweb/freshness/scripts/media_portraits/build_mediap_server.sh'

    CONDUCTOR_LATEST_STABLE_STATIC_PACKAGE_URL \
        = 'https://c.yandex-team.ru/api/package_version/yandex-mediaportraits-static?branch=stable'

    def _build_server_archive(self):
        build_script_name = os.path.basename(self.BUILD_SCRIPT_SVN)

        Arcadia.export(self.BUILD_SCRIPT_SVN, "./" + build_script_name)

        build_script_path = channel.task.abs_path(build_script_name)

        if not os.path.exists(build_script_path):
            raise SandboxTaskFailureError("Can't find %s in task dir" % build_script_path)

        process = run_process(
            [build_script_path,
                self.ctx['repo_url'],
                'mediap-server-%s-' % self.ctx['static_version'],
                self.ctx['static_version']],
            shell=True,
            log_prefix=build_script_name,
            stdout=subprocess.PIPE,
            outputs_to_one_file=False
            # environment=env
        )

        archive_file = process.stdout.readline().rstrip()

        if not archive_file or not os.path.exists(archive_file):
            raise RuntimeError("Bad result from building script")

        return archive_file

    def _determine_latest_static_version(self):
        resp = requests.get(self.CONDUCTOR_LATEST_STABLE_STATIC_PACKAGE_URL, timeout=60)
        return resp.json()['yandex-mediaportraits-static']['stable']['version']

    def on_execute(self):
        if self.ctx['static_version'] == 'latest':
            self.ctx['static_version'] = self._determine_latest_static_version()

        archive_filename = self._build_server_archive()

        self.create_resource(
            description=self.descr,
            resource_path=archive_filename,
            resource_type=self.RESOURCE_TYPE,
            arch='any'
        )

    def arcadia_info(self):
        return '', None, 'version-XXX'


__Task__ = BuildMediaportraitsFrontend
