# coding: utf-8

import logging
import os

import sandbox.common.types.client as ctc

from sandbox.projects import resource_types
from sandbox import sandboxsdk
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxBoolParameter
from sandbox.projects.common.build.BuildForAllTask import VersionParameter, DoNotRemoveResourcesParameter
from sandbox.projects.common.nanny import nanny


class MemcachedVersion(VersionParameter):
    name = 'memcached_version'
    description = 'Memcached version'
    default_value = '1.4.25'


class MemcachedSourceURL(SandboxStringParameter):
    name = 'memcached_url'
    description = 'Memcached git URL'
    default_value = 'https://github.com/memcached/memcached.git'


class LibeventVersion(VersionParameter):
    name = 'libevent_version'
    description = 'Libevent version'
    default_value = 'release-2.0.21-stable'


class LibeventSourceUrl(SandboxStringParameter):
    name = 'libevent_url'
    description = 'Libevent git URL'
    default_value = 'https://github.com/libevent/libevent.git'


class BuildStatic(SandboxBoolParameter):
    name = 'build_static'
    description = 'Build static executable'
    default_value = True


class BuildMemcached(SandboxTask, nanny.ReleaseToNannyTask):
    type = 'BUILD_MEMCACHED'
    client_tags = ctc.Tag.LINUX_PRECISE

    input_parameters = [MemcachedVersion, MemcachedSourceURL, LibeventSourceUrl, LibeventVersion,
                        DoNotRemoveResourcesParameter, BuildStatic]
    execution_space = 4 * 1024

    def checkout(self, module):

        logging.info('Checking out {} from {}, tag: {}'.format(module['name'], module['url'], module['tag']))
        sandboxsdk.paths.make_folder(module['name'], delete_content=True)

        git_commands = (
            'git clone %s .' % module['url'],
            'git fetch --all',
            'git checkout %s' % module['tag']
        )
        git_commands = map(lambda x: x.split(), git_commands)
        for cmd in git_commands:
            run_process(cmd, log_prefix='checkout', work_dir=module['name'], shell=True)

    def autogen(self, module):
        run_process('./autogen.sh', log_prefix='build', work_dir=module['name'], shell=True)

    def configure(sellf, module):
        run_process('./configure {0}'.format(module['configure_opts']), log_prefix='build', work_dir=module['name'],
                     environment=module['configure_env'], shell=True)

    def make(self, module):
        run_process('make', log_prefix='build', work_dir=module['name'], shell=True)

    def install(self, module):
        run_process('make install', log_prefix='build', work_dir=module['name'], shell=True)

    def build(self, module):

        logging.info('Building {}...'.format(module['name']))
        for method in module['methods']:
            method(module)

    def on_execute(self):

        modules = [{'name': 'libevent',
                      'url': self.ctx['libevent_url'],
                      'tag': self.ctx['libevent_version'],
                      'methods': [self.autogen, self.configure, self.make, self.install],
                      'configure_env': {},
                      'configure_opts': '--enable-shared=no --prefix={}/libevent-install'.format(os.getcwd()),
                    },
                    {'name': 'memcached',
                      'url': self.ctx['memcached_url'],
                      'tag': self.ctx['memcached_version'],
                      'methods': [self.autogen, self.configure, self.make],
                      'configure_env': {'CC': 'gcc {}'.format('-static' if self.ctx['build_static'] else ''),
                                         'PATH': '/usr/local/sbin:/usr/local/bin:/usr/sbin:/usr/bin:/sbin:/bin'},
                      'configure_opts': '--with-libevent={}/libevent-install'.format(os.getcwd()),
                    }]
        with self.current_action('Downloading sources'):
            for mod in modules:
                self.checkout(mod)
        with self.current_action('Building'):
            for mod in modules:
                self.build(mod)

        res = self.create_resource(
            description='Memcached {} {}executable'.format(self.ctx['memcached_version'],
                                                           'static ' if self.ctx['build_static'] else ''),
            resource_path='memcached/memcached',
            resource_type=resource_types.MEMCACHED_BINARY,
            arch='linux'
        )
        self.ctx['report_resource_id'] = res.id

        self.set_info('Build memcached process completed successfully.')

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)
        SandboxTask.on_release(self, additional_parameters)


__Task__ = BuildMemcached
