import logging
import tarfile
from os import path, walk

from sandbox.sandboxsdk.process import run_process

from sandbox.projects.resource_types import PYTHON_BUNDLE
from sandbox.projects.BuildPythonBundle2 import BuildPythonBundle2


class BuildMikey(BuildPythonBundle2):
    type = 'BUILD_MIKEY'

    @property
    def release_template(self):
        logging.info('==> saved changelog:\n{}'.format(self.exec_context.get('changelog')))
        return self.ReleaseTemplate(
            subject='Mikey v.{} from {}'.format(self.ctx['mikey_version'], self.ctx['revision']),
            message=self.ctx.get('changelog')
        )

    def pre_install(self):
        mikey_src = self.exec_context['target_src']
        mikey_python_bin = self.exec_context['target_python_bin']
        mikey_lib_path = self.exec_context['target_lib_path']
        mikey_manage_py = path.join(mikey_src, 'src', 'manage.py')

        # Get version.
        with open(path.join(mikey_src, 'version.txt'), 'r') as f:
            mikey_version = f.read()
            self.ctx['mikey_version'] = mikey_version

        # Sandbox task id.
        self.ctx['mikey_sandbox_task_id'] = str(self.id)

        # Collect static files.
        # TODO: get settings module from input_parameters
        logging.info('==> collecting static files')
        run_process(
            [
                mikey_python_bin, mikey_manage_py, 'collectstatic',
                '--noinput',
                '--settings', 'mikey.settings.build'
            ],
            log_prefix='mikey_collect_static', check=True, wait=True,
            environment={
                'PYTHONPATH': mikey_lib_path,
                'LD_LIBRARY_PATH': mikey_lib_path,
            }
        )

        # Find static root.
        out, err = run_process(
            [mikey_python_bin, mikey_manage_py, 'diffsettings', '--settings', 'mikey.settings.build'],
            check=True, wait=True, outs_to_pipe=True,
            environment={
                'PYTHONPATH': mikey_lib_path,
                'LD_LIBRARY_PATH': mikey_lib_path,
            }
        ).communicate()
        out, err = out.decode(), err.decode()
        logging.debug('==> ERR:\n{}'.format(err))
        logging.debug('==> OUT:\n{}'.format(out))
        # Only get upper case settings.
        mikey_settings = {}
        for line in out.split('\n'):
            splitted = line.split(' = ')
            if len(splitted) == 1:
                mikey_settings[line] = None
            else:
                key = splitted[0].strip()
                value = ' = '.join(splitted[1:])

                # Remove quotes.
                value = value.strip()
                if value.startswith("'") and value.endswith("'"):
                    value = value[1:-1]

                mikey_settings[key] = value

        logging.debug('==> read settings: {}'.format(mikey_settings))
        mikey_static_root = mikey_settings.get('STATIC_ROOT', path.join(mikey_src, 'collected_static'))
        logging.debug(list(walk(mikey_static_root)))

        self.exec_context['staticfiles_path'] = mikey_static_root

    def do_create_resource(self):
        target_env = self.exec_context['target_path']
        target_path = self.path('bundle.tar')

        staticfiles_path = self.exec_context['staticfiles_path']

        # Add version to main resource.
        version_path = self.path('VERSION')
        logging.info('==> writing version ({}) to {}'.format(
            self.ctx['mikey_version'],
            version_path
        ))
        with open(version_path, 'w') as f:
            f.write(self.ctx['mikey_version'])

        # Add sandbox task id to main resource.
        sandbox_task_id_path = self.path('sandbox_task_id.txt')
        logging.info('==> writing sandbox task id ({}) to {}'.format(
            self.ctx['mikey_sandbox_task_id'],
            sandbox_task_id_path
        ))
        with open(sandbox_task_id_path, 'w') as f:
            f.write(self.ctx['mikey_sandbox_task_id'])

        # Add commit info to main resource.
        revision_path = self.path('revision.txt')
        logging.info('==> writing revision ({}) to {}'.format(
            self.ctx['revision'],
            revision_path
        ))
        with open(revision_path, 'w') as f:
            f.write(self.ctx['revision'])

        logging.info('-> compressing virtualenv')
        with tarfile.TarFile(target_path, 'w') as t:
            # FIXME no more custom names
            t.add(revision_path, 'revision.txt')
            t.add(sandbox_task_id_path, 'sandbox_task_id.txt')
            t.add(staticfiles_path, 'collected_static')
            t.add(target_env, 'ENV')
            t.add(version_path, 'VERSION')

        logging.info('-> creating resource: {}'.format('target.tar'))
        self.create_resource(
            description=self.ctx.get(
                'description', 'Python bundle built from {}, rev.{}'.format(self.ctx['git_url'], self.ctx['revision'])
            ),
            resource_path=target_path,
            resource_type=PYTHON_BUNDLE,
            attributes={
                'branch': self.ctx['git_branch'],
                'commit': self.ctx['last_commit'],
                'version': self.ctx['mikey_version'],
                'build': self.ctx['build']
            }
        )


__Task__ = BuildMikey
