# coding: utf-8

import os
import tarfile

from sandbox.common.types.misc import DnsType
from sandbox.common.types.client import Tag
from sandbox.projects import resource_types
from sandbox.projects.nanny.common.environments import GoDistEnvironment
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.task import SandboxTask


class ToolsVersion(parameters.SandboxStringParameter):
    name = 'version'
    description = 'Mongo-tools version'
    default_value = 'master'
    required = True


class BuildMongoToolsPackage(SandboxTask):
    type = 'BUILD_MONGO_TOOLS_PACKAGE'

    dns = DnsType.DNS64  # github doesn't support ipv6 at the moment
    client_tags = Tag.LINUX_PRECISE

    input_parameters = [ToolsVersion]
    environment = (
        GoDistEnvironment('1.7.3'),
    )

    GIT_URL = 'https://github.com/mongodb/mongo-tools'
    TGZ_NAME = 'mongo-tools.tgz'

    CHECKOUT_DIR = 'mongo-tools'

    def on_execute(self):
        run_process(
            ['git', 'clone', self.GIT_URL, self.CHECKOUT_DIR],
            log_prefix='checkout'
        )
        run_process(
            ['git', 'checkout', self.ctx.get(ToolsVersion.name)],
            log_prefix='tag',
            work_dir='mongo-tools'
        )
        run_process(
            ['./build.sh'],
            log_prefix='build',
            work_dir='mongo-tools'
        )

        with tarfile.open(self.path(self.TGZ_NAME), 'w:gz') as tar:
            for entry in os.listdir(os.path.join(self.path(), self.CHECKOUT_DIR, 'bin')):
                tar.add(os.path.join(os.path.join(self.path(), self.CHECKOUT_DIR, 'bin'), entry), entry)

        self.create_resource(
            description='Mongo tools archive, version {}'.format(self.ctx.get(ToolsVersion.name)),
            resource_path=self.TGZ_NAME,
            resource_type=resource_types.MONGO_TOOLS_ARCHIVE,
            arch='linux'
        )


__Task__ = BuildMongoToolsPackage
