# -*- coding: utf-8 -*-

import logging
import shutil
import tarfile
from os import path, environ, listdir

from sandbox.projects.BuildPythonBundle import BuildPythonBundle, GitUrl, GitBranch, TargetRequirements, TargetVirtualenvName, \
    PythonSource, ExternalLibraries, BuildId
from sandbox.projects.resource_types import AbstractResource, search_priemka_releasers
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.parameters import SandboxStringParameter


class CommitId(SandboxStringParameter):
    name = 'commit_id'
    description = 'Commit id:'
    default_value = 'HEAD'
    required = False


class MONSYS_DC_BUNDLE(AbstractResource):
    """
        Archived relocatable virtualenv with monsys-dc code.
    """
    any_arch = False
    auto_backup = False
    releasable = True
    releasers = search_priemka_releasers


class BuildMonsysDcWithVenv(BuildPythonBundle):
    type = 'BUILD_MONSYS_DC_WITH_VENV'

    input_parameters = (
        GitUrl,
        GitBranch,
        CommitId,
        TargetRequirements,
        TargetVirtualenvName,
        PythonSource,
        ExternalLibraries,
        BuildId
    )

    def on_execute(self):
        logging.info('==> checking out target source from {}'.format(self.ctx['git_url']))
        target_src = self.path('target_src')
        self.exec_context['target_src'] = target_src
        run_process(
            ['git', 'clone', self.ctx['git_url'].format(
                oauth=self.get_vault_data(self.owner, 'robot_maestro_stash_oauth')
            ), target_src],
            log_prefix='git', check=True, wait=True, environment={}
        )

        if self.ctx['git_branch'].startswith('refs/heads'):
            # Need to fetch ref first.
            actual_branch_name = self.ctx['git_branch'][11:]
            run_process(
                ['git', 'checkout', actual_branch_name],
                log_prefix='git', check=True, wait=True, work_dir=target_src, environment={}
            )
            run_process(
                ['git', 'fetch'],
                log_prefix='git', check=True, wait=True, work_dir=target_src, environment={}
            )

        if self.ctx['commit_id'] == 'HEAD':
            run_process(
                ['git', 'checkout', self.ctx['git_branch']],
                log_prefix='git', check=True, wait=True, work_dir=target_src, environment={}
            )
        else:
            run_process(
                ['git', 'checkout', self.ctx['commit_id']],
                log_prefix='git', check=True, wait=True, work_dir=target_src, environment={}
            )

        logging.info('==> looking for last commit')
        self.exec_context['release_commits'] = []

        out, _ = run_process(
            ['git', 'log', '-1'],
            check=True, wait=True, work_dir=target_src, environment={}, outs_to_pipe=True
        ).communicate()
        self.ctx['last_commit'] = out.decode('utf-8').split('\n')[0].split()[1]
        if self.ctx['commit_id'] == 'HEAD':
            self.ctx['revision'] = '{}@{}'.format(self.ctx['git_branch'], self.ctx['last_commit'])
        else:
            self.ctx['revision'] = '{}@{}'.format(self.ctx['git_branch'], self.ctx['commit_id'])

        logging.info('==> last commit is {}'.format(self.ctx['last_commit']))

        logging.info('==> BUILDING PYTHON FROM SOURCE')
        logging.info('==> downloading python source')
        compressed_python_src = self.sync_resource(self.ctx['python_src'])

        python_src = self.path('python-source')
        logging.info('==> decompressing python source to {}'.format(python_src))
        with tarfile.open(compressed_python_src) as t:
            t.extractall(path=python_src)
            source_prefix = t.getnames()[0]
            python_version = source_prefix.split('-')[1][:3]
            python_src = path.join(python_src, source_prefix)

        ext_lib_path = None
        if self.ctx['python_ext_lib']:
            # Download and unpack external libraries.
            logging.info('==> downloading external libraries')
            compressed_ext_lib = self.sync_resource(self.ctx['python_ext_lib'])

            ext_lib_path = self.path('ext_lib')
            logging.info('==> unpacking external libraries')
            with tarfile.open(compressed_ext_lib) as t:
                t.extractall(path=ext_lib_path)

            # Dirty hack for external libsqlite3.
            with open(path.join(python_src, 'setup.py'), 'r') as f:
                setup_py = f.read().split('\n')

            last_include_lineno = setup_py.index([l for l in setup_py if '/usr/local/include/sqlite3' in l][0])
            new_setup_py = setup_py[:last_include_lineno]
            spaces_before_line = setup_py[last_include_lineno].count(' ')
            new_setup_py.append("{}'{}',".format(
                ' ' * spaces_before_line,
                path.join(ext_lib_path, 'include')
            ))
            new_setup_py += setup_py[last_include_lineno + 1:]

            with open(path.join(python_src, 'setup.py'), 'w') as f:
                f.write('\n'.join(new_setup_py))

        python_dist = self.path('python-dist')
        logging.info('==> configuring python')
        run_process(
            [
                './configure',
                '--enable-shared',
                '--with-ensurepip=install',
                '--prefix={}'.format(python_dist)
            ],
            log_prefix='python_configure', check=True, wait=True, work_dir=python_src, environment={}
        )

        logging.info('==> building python')
        run_process(
            ['make'],
            log_prefix='python_make', check=True, wait=True, work_dir=python_src, environment={}
        )

        logging.info('==> installing python to {}'.format(python_dist))
        run_process(
            ['make', 'install'],
            log_prefix='python_install', check=True, wait=True, work_dir=python_src, environment={}
        )
        python_bin = path.join(python_dist, 'bin', 'python{}'.format(python_version))
        assert path.isfile(python_bin)

        source_lib_content = listdir(path.join(python_dist, 'lib'))
        logging.debug('==> lib from source:\n{}'.format(
            '\n'.join(source_lib_content)
        ))

        logging.info('==> installing virtualenv')
        run_process(
            [
                python_bin, '-m', 'pip', 'install',
                '--only-binary', 'all', '-f', path.join(target_src, 'wheel'), 'virtualenv'
            ],
            log_prefix='pip_install_virtualenv', check=True, wait=True, environment={
                'PATH': '{}:{}'.format(path.join(python_dist), environ['PATH']),
                'LD_LIBRARY_PATH': path.join(python_dist, 'lib')
            }
        )
        virtualenv_bin = path.join(python_dist, 'bin', 'virtualenv')

        target_env = self.path(self.ctx['target_virtualenv_name'])
        self.exec_context['target_env'] = target_env
        python_bin = path.join(python_dist, 'bin', 'python{}'.format(python_version))

        logging.info('==> using python at {}'.format(python_bin))
        assert path.isfile(python_bin)

        logging.info('==> using virtualenv at {}'.format(virtualenv_bin))
        assert path.isfile(virtualenv_bin)

        logging.info('==> creating new virtualenv at {}'.format(target_env))
        run_process(
            [
                virtualenv_bin, '--always-copy', '-p', python_bin, target_env
            ],
            log_prefix='virtualenv_create_new', check=True, wait=True,
            environment={
                'PATH': '{}:{}'.format(path.join(python_dist), environ['PATH']),
                'LD_LIBRARY_PATH': path.join(python_dist, 'lib')
            }
        )

        # Even with --always-copy virtualenv may leave some symlinks.
        existing_lib = path.join(target_env, 'lib')
        logging.info('==> deleting existing lib: {}'.format(existing_lib))
        shutil.rmtree(existing_lib)

        logging.info('==> copying lib from {}'.format(python_dist, 'lib'))
        shutil.copytree(
            path.join(python_dist, 'lib'),
            path.join(target_env, 'lib')
        )

        if self.ctx['python_ext_lib']:
            logging.info('==> copying external libraries from {}'.format(ext_lib_path))
            for i in listdir(path.join(ext_lib_path, 'lib')):
                current_lib_path = path.join(ext_lib_path, 'lib', i)
                if path.isfile(current_lib_path):
                    logging.info('\t-> {}'.format(i))
                    shutil.copy2(current_lib_path, path.join(target_env, 'lib'))
                else:
                    logging.info('\t-> {} (skipped)'.format(i))

        logging.info('==> BUILDING TARGET')

        target_python_bin = path.join(target_env, 'bin', 'python')
        self.exec_context['target_python_bin'] = target_python_bin
        assert path.isfile(target_python_bin)

        target_lib_path = path.join(target_env, 'lib')
        self.exec_context['target_lib_path'] = target_lib_path
        assert path.isdir(target_lib_path)

        logging.info('==> upgrading pip')
        run_process(
            [
                target_python_bin, '-m', 'pip', 'install',
                '--index-url=https://pypi.yandex-team.ru/simple',
                '--find-links', path.join(target_src, 'wheel'),
                '--verbose',
                '--upgrade',
                'pip'
            ],
            log_prefix='pip_upgrade', check=True, wait=True,
            environment={
                'PYTHONPATH': target_lib_path,
                'LD_LIBRARY_PATH': target_lib_path,
            }
        )

        logging.info('==> installing target requirements')
        run_process(
            [
                target_python_bin, '-m', 'pip', 'install',
                '--index-url=https://pypi.yandex-team.ru/simple',
                '--find-links', path.join(target_src, 'wheel'),
                '--verbose',
                '--prefix', target_env,
                '-r', path.join(target_src, self.ctx['target_requirements'])
            ],
            log_prefix='pip_install_target', check=True, wait=True,
            environment={
                'PYTHONPATH': target_lib_path,
                'LD_LIBRARY_PATH': target_lib_path,
            }
        )

        self.pre_install()

        # fix for logging
        run_process(
            [
                'mkdir', 'logs'
            ]
        )

        # Collect static files.
        logging.info('==> collecting static files')
        manage_py = path.join(target_src, 'avtomatika', 'manage.py')
        run_process(
            [
                target_python_bin, manage_py, 'collectstatic',
                '--noinput',
                '--settings', 'avtomatika.settings.web'
            ],
            log_prefix='collect_static', check=True, wait=True,
            environment={
                'PYTHONPATH': '{}:{}'.format(target_lib_path, target_src),
                'LD_LIBRARY_PATH': target_lib_path,
            }
        )

        run_process(
            ['rm', path.join(target_src, 'avtomatika/settings/generated.py')]
        )

        logging.info('==> making virtualenv relocatable')
        run_process(
            [target_python_bin, '-m', 'virtualenv', '--relocatable', target_env],
            log_prefix='virtualenv_make_relocatable', check=True, wait=True,
            environment={
                'PYTHONPATH': target_lib_path,
                'LD_LIBRARY_PATH': target_lib_path,
            }
        )

        self.exec_context['target_path'] = target_env

        self.do_create_resource()
        self.post_execute()

        logging.info('==> SUCCESS!')

    def do_create_resource(self):
        target_src = self.path('target_src')
        staticfiles_path = self.path('collected_static')

        # Add sandbox task id to main resource.
        sandbox_task_id_path = self.path('sandbox_task_id.txt')
        logging.info('==> writing sandbox task id ({}) to {}'.format(
            self.ctx['build'],
            sandbox_task_id_path
        ))
        with open(sandbox_task_id_path, 'w') as f:
            f.write(str(self.id))

        # Add commit info to main resource.
        revision_path = self.path('revision.txt')
        logging.info('==> writing revision ({}) to {}'.format(
            self.ctx['revision'],
            revision_path
        ))
        with open(revision_path, 'w') as f:
            f.write(self.ctx['revision'])

        tar = self.path('bundle.tar')
        with tarfile.TarFile(tar, 'w') as t:
            t.add(self.exec_context['target_path'], self.ctx['target_virtualenv_name'])
            t.add(target_src, 'monsys-dc')
            t.add(sandbox_task_id_path, 'sandbox_task_id.txt')
            t.add(staticfiles_path, 'collected_static')
            t.add(revision_path, 'revision.txt')

        # Create bundle resource.
        self.create_resource(
            description='compressed virtualenv',
            resource_path=tar,
            resource_type=MONSYS_DC_BUNDLE,
            attributes={
                'branch': self.ctx['git_branch'],
                'commit': self.ctx['last_commit'] if self.ctx['commit_id'] == 'HEAD' else self.ctx['commit_id'],
                'build': self.ctx['build'],

            }
        )


__Task__ = BuildMonsysDcWithVenv
