# coding: utf-8
import hashlib
import json
import logging
import os
import tarfile
import time

import sandbox.common.types.misc as ctm
from sandbox.common.types.client import Tag
from sandbox.projects import resource_types
from sandbox.projects.common import environments
from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxBoolParameter
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.yasandbox import manager

from sandbox.projects.common.nanny import nanny


def list_files(root):
    rv = []
    for d, _, names in os.walk(root):
        for n in names:
            p = os.path.join(d, n)
            if not os.path.islink(p) and os.path.splitext(p)[1] != '.pyc':
                rv.append(p)
    return rv


def calc_checksum(files, read_chunk_size):
    h = hashlib.md5()
    for f in sorted(files):
        with open(f, 'rb') as fd:
            for chunk in fd.read(read_chunk_size):
                h.update(chunk)
    return h.hexdigest()


class GitRefIdParameter(SandboxStringParameter):
    name = 'ref_id'
    description = 'Git ref id'
    default_value = 'master'
    required = True


class GitRefShaParameter(SandboxStringParameter):
    name = 'ref_sha'
    description = 'Git ref SHA'
    default_value = ''
    required = False


class ReleaseParameter(SandboxBoolParameter):
    name = 'release'
    description = 'Create resource after tests'
    default_value = False


class BuildNanny(nanny.ReleaseToNannyTask, SandboxTask):
    """
        Задача тестирования и сборки Nanny.
        При сборке создаётся пустой venv, в него устанавливается пакет nanny и
        все его зависимости (sepelib, Flask, etc.).
    """
    type = 'BUILD_NANNY'

    # DNS64 is required to be able to connect to npm.yandex-team.ru
    dns = ctm.DnsType.DNS64

    client_tags = Tag.LINUX_XENIAL

    cores = 1

    input_parameters = [
        GitRefIdParameter,
        GitRefShaParameter,
        ReleaseParameter,
        nanny.StartrekTicketIdsParameter
    ]

    environment = [
        environments.SandboxGitEnvironment('2.19.0'),
        environments.SwatMongoDbEnvironment('3.4.2'),
        environments.SwatZookeeperEnvironment('3.4.6'),
        environments.SandboxJavaJdkEnvironment('1.8.0'),
    ]

    GIT_URL = 'https://{username}:{password}@bb.yandex-team.ru/scm/nanny/nanny.git'
    CHECKOUT_PATH = 'nanny'
    TGZ_PATH = 'nanny.tar.gz'
    PIP_TGZ = 'pip-20.3.4.tar.gz'
    CHECKSUM_FILE = 'CHECKSUM'

    NODE_VERSIONS = {
        'v10.20.1': 1530173875,  # https://sandbox.yandex-team.ru/resource/1530173875/view
        'v16.13.1': 2642699237,  # https://sandbox.yandex-team.ru/resource/2642699237/view
    }

    def _get_nanny_src_dir(self):
        return os.path.join(self.CHECKOUT_PATH, 'nanny')

    def _get_checkout_url(self):
        oauth_token = self.get_vault_data('GBG', 'nanny_robot_bb_oauth_token')
        return self.GIT_URL.format(username='x-oauth-token', password=oauth_token)

    def _get_venv_path(self):
        return self.path('nanny_build')

    def _get_venv_bin_path(self):
        return os.path.join(self._get_venv_path(), 'bin')

    def _get_venv_python_path(self):
        return os.path.join(self._get_venv_bin_path(), 'python')

    def _run_pip_install(self, args, log_prefix=None):
        return run_process([self._get_venv_python_path(), '-m', 'pip', 'install'] + args,
                           log_prefix=log_prefix,
                           work_dir=self._get_nanny_src_dir())

    @staticmethod
    def _run_git_command(command, args, work_dir=None):
        run_process(
            ['git', command] + args,
            log_prefix='git_%s' % command,
            work_dir=work_dir,
            shell=True
        )

    def _checkout(self, url, dest, ref=None):
        if ref is None:
            self._run_git_command('clone', [url, dest, '--depth', '1'])
        else:
            self._run_git_command('init', [dest])
            self._run_git_command('remote', ['add', 'origin', url], work_dir=dest)
            self._run_git_command('fetch', ['origin', ref, '--depth', '1'], work_dir=dest)
            self._run_git_command('reset', ['--hard', 'FETCH_HEAD'], work_dir=dest)

    def _install_packages(self):
        # update pip to be able to install wheels
        self._run_pip_install(['-U', self.PIP_TGZ])

        # install dependencies from local dir, fail if some are not satisfied
        self._run_pip_install(
            ['--no-index', '--find-links=./wheels', '-r', 'pip-requirements.txt'],
            log_prefix='nanny_reqs_pip_install')
        # now install nanny itself
        self._run_pip_install(
            ['--no-index', '.'],
            log_prefix='nanny_pip_install')

    def _make_resource(self, virtualenv_path, description):
        logging.info('creating tgz file')

        # calc checksum for sources and save it.
        checksum = calc_checksum(files=list_files(virtualenv_path),
                                 read_chunk_size=1024 * 1024)
        checksum_path = os.path.join(virtualenv_path, self.CHECKSUM_FILE)
        with open(checksum_path, 'wb') as f:
            f.write(checksum)

        with tarfile.open(self.TGZ_PATH, 'w:gz') as tar:
            for entry in os.listdir(virtualenv_path):
                tar.add(os.path.join(virtualenv_path, entry), entry)
            tar.add(checksum_path, self.CHECKSUM_FILE)

        self.create_resource(
            description=description,
            resource_path=self.TGZ_PATH,
            resource_type=resource_types.NANNY,
            arch='linux'
        )

    def _set_nanny_version(self):
        nanny_src_dir = self._get_nanny_src_dir()
        version_file_path = os.path.join('src', 'nanny', 'version.py')
        run_process(['sed', '-i',
                     '"s/development_version/$(/skynet/python/bin/python setup.py --version)/"', version_file_path],
                    shell=True,
                    work_dir=nanny_src_dir)
        now = str(int(time.time()))
        run_process(['sed', '-i', '"s/TIMESTAMP = -1/TIMESTAMP = {}/"'.format(now), version_file_path],
                    shell=True,
                    work_dir=nanny_src_dir)

    def _build_static_files(self):
        nanny_src_dir = self._get_nanny_src_dir()
        work_dir = os.path.join(nanny_src_dir, 'src', 'ui')

        nvmrc = os.path.join(work_dir, '.nvmrc')
        with open(nvmrc) as f:
            node_version = f.read().strip()
        if node_version not in self.NODE_VERSIONS:
            raise RuntimeError('{}: unknown node version "{}", supported versions are "{}"'.format(nvmrc, node_version, '", "'.join(sorted(self.NODE_VERSIONS))))
        node_resource = self.NODE_VERSIONS[node_version]

        path = self.sync_resource(node_resource)
        node_dir = make_folder('node')  # absolute
        run_process(['tar', 'xf', path, '-C', node_dir], shell=True)

        bin_path = os.path.join(node_dir, 'bin') + ':' + os.environ.get('PATH', '')
        bin_path = bin_path.rstrip(':')

        bin_path = os.path.join(self.path(work_dir), 'node_modules', '.bin') + ':' + bin_path
        grunt_path = self.path(os.path.join(work_dir, 'node_modules', '.bin', 'grunt'))
        react_path = self.path(os.path.join(work_dir, 'reactComponents'))
        integrator_path = self.path(os.path.join(work_dir, 'reactComponents', 'integrator'))
        if os.path.isdir(react_path):
            run_process(['npm', 'ci'],
                        log_prefix='react_install',
                        work_dir=react_path,
                        shell=True,
                        environment={'PATH': bin_path})
            run_process(['npm', 'run', 'build'],
                        log_prefix='react_build',
                        work_dir=react_path,
                        shell=True,
                        environment={'PATH': bin_path})
            run_process(['npm', 'ci'],
                        log_prefix='integrator_install',
                        work_dir=integrator_path,
                        shell=True,
                        environment={'PATH': bin_path})
            run_process(['npm', 'run', 'build'],
                        log_prefix='integrator_build',
                        work_dir=integrator_path,
                        shell=True,
                        environment={'PATH': bin_path})
        run_process([grunt_path, '--force'],  # build & compile
                    log_prefix='grunt_compile',
                    work_dir=work_dir,
                    shell=True,
                    environment={'PATH': bin_path})

    def arcadia_info(self):
        """
        Hacky way to allow this task to be released: provide tag, other fields are not checked.
        """
        return None, self.ctx.get('tag'), None

    def _run_tests(self):
        venv_bin_path = self._get_venv_bin_path()
        nanny_src_dir = self._get_nanny_src_dir()

        self._run_pip_install(
            ['--no-index', '--find-links=./wheels', '-r', 'pip-dev-requirements.txt'],
            log_prefix='nanny_dev_reqs_pip_install')

        env = dict(os.environ)
        path_parts = [venv_bin_path]
        if 'PATH' in env:
            path_parts.append(env['PATH'])
        env_path = ':'.join(path_parts)
        env.update({'PATH': env_path})

        old_do_not_restart = self.ctx['do_not_restart']
        self.ctx['do_not_restart'] = True
        run_process(
            [
                './test.sh',
                '--zookeeper={}'.format(os.environ['ZOOKEEPER_DIR']),
            ],
            environment=env,
            log_prefix='test_sh',
            work_dir=nanny_src_dir
        )
        self.ctx['do_not_restart'] = old_do_not_restart

    def on_execute(self):
        logging.info('checking out the source code')
        ref_id = self.ctx['ref_id']
        ref_sha = self.ctx.get('ref_sha')
        checkout_url = self._get_checkout_url()
        self._checkout(checkout_url, self.CHECKOUT_PATH, ref=ref_sha or ref_id)

        if self.ctx.get('release'):
            self._build_static_files()

        venv_path = self._get_venv_path()
        # create virtualenv using *skynet* python
        run_process(['/skynet/python/bin/virtualenv', venv_path])

        self._set_nanny_version()
        self._install_packages()

        logging.info('run tests')
        self._run_tests()

        if self.ctx.get('release'):
            run_process(['/skynet/python/bin/virtualenv', '--relocatable', venv_path])
            if ref_id.startswith('refs/tags/'):
                tag = ref_id[len('refs/tags/'):]
            else:
                tag = 'master'
            description = 'nanny virtualenv tgz {0}'.format(tag)
            self._make_resource(venv_path, description)

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)

        data = self.get_nanny_release_info(additional_parameters)
        logging.info('Release payload: %s', json.dumps(data, indent=4))
        logging.info('Sending release of task %s to adm-nanny', self.id)
        adm_nanny_oauth_token = self.get_vault_data('GBG', 'nanny_robot_adm_nanny_oauth_token')
        adm_nanny_client = nanny.NannyClient(api_url='http://adm-nanny.yandex-team.ru/',
                                             oauth_token=adm_nanny_oauth_token)
        adm_nanny_client.create_release2(data)
        logging.info('Release of task %s has been sent to adm-nanny', self.id)
        self.set_logs_inf_ttl()

    def set_logs_inf_ttl(self):
        resources = [_ for _ in self.list_resources()]
        for resource in resources:
            if resource.type != 'TASK_LOGS':
                continue
            resource.attrs["ttl"] = 'inf'
            resource.attrs.setdefault("backup_task", True)
            manager.resource_manager.update(resource)
        return resources


__Task__ = BuildNanny
