import glob
import os
import textwrap
import time

from sandbox.sandboxsdk import ssh
from sandbox.sandboxsdk.parameters import SandboxStringParameter, Container
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.task import SandboxTask


class ProjectParameter(SandboxStringParameter):
    name = "project"
    description = "Project name"
    required = True


class GitTagParameter(SandboxStringParameter):
    name = "tag"
    description = "Git tag"
    required = True


class LoginParameter(SandboxStringParameter):
    name = "login"
    description = "Robot login"
    required = True


class VaultOwnerParameter(SandboxStringParameter):
    name = "vault_owner"
    description = "Vault owner"
    required = True


class SshKeyParameter(SandboxStringParameter):
    name = "ssh_key"
    description = "SSH key vault name"
    default_value = "ssh-key"


class GpgPublicKeyParameter(SandboxStringParameter):
    name = "gpg_public_key"
    description = "GPG public key"
    default_value = "gpg-public-key"


class GpgPrivateKeyParameter(SandboxStringParameter):
    name = "gpg_private_key"
    description = "GPG private key"
    default_value = "gpg-private-key"


class LxcContainer(Container):
    required = True
    description = "LXC container for task"
    alias = "linux_ubuntu_12.04_precise"
    resource_type = "LXC_CONTAINER"
    default_value = "528578338"


class BuildNannyDeb(SandboxTask):
    """Builds a deb package from Nanny's git repository."""

    type = "BUILD_NANNY_DEB"

    input_parameters = [
        ProjectParameter,
        GitTagParameter,
        LoginParameter,
        VaultOwnerParameter,
        SshKeyParameter,
        GpgPublicKeyParameter,
        GpgPrivateKeyParameter,
        LxcContainer,
    ]

    execution_space = 100

    def on_execute(self):
        login = self.ctx["login"]
        environment, ssh_key_path = self.__prepare_environment()

        with ssh.Key(self, self.ctx["vault_owner"], self.ctx["ssh_key"]):
            src_dir = self.__checkout(self.ctx["project"], self.ctx["tag"])

        run_process(["debuild", "--no-tgz-check"], work_dir=src_dir, environment=environment, log_prefix="debuild")

        with open("dupload.conf", "w") as dupload_conf:
            dupload_conf.write(textwrap.dedent("""\
                package config;
                $default_host = "common";

                $cfg{{"common"}} = {{
                    fqdn => "common.dupload.dist.yandex.ru",
                    method => "scpb",
                    login => "{login}",
                    options => "-i {ssh_key_path}",
                    incoming => "/repo/common/mini-dinstall/incoming/",
                    dinstall_runs => 0,
                }};
            """.format(login=login, ssh_key_path=ssh_key_path)))

        uploaded_packages = []

        for changes_file in glob.glob("*.changes"):
            package_full_name = changes_file[:-len(".changes")]
            package_name, package_version, _package_arch = package_full_name.split("_")

            run_process(["dupload", "-c", "-d", "--nomail", "--to", "common", changes_file],
                        environment=environment, log_prefix="dupload-" + package_full_name)

            uploaded_packages.append((package_name, package_version, package_full_name))

        # The packages won't be available immediately and ibiryulin@ said that we shouldn't try to move them until they
        # will be indexed because in this case we can get an undefined behaviour. ibiryulin@ said that 2 minutes should
        # be enough to wait for packages to reindex.
        time.sleep(2 * 60)

        for package_name, package_version, package_full_name in uploaded_packages:
            run_process(["ssh", "dupload.dist.yandex.ru", "-l", login, "-i", ssh_key_path, "--",
                         "sudo", "dmove", "common", "stable", package_name, package_version, "unstable"],
                        log_prefix="dmove-" + package_full_name)

    def __prepare_environment(self):
        environment = os.environ.copy()
        vault_owner = self.ctx["vault_owner"]

        with open("ssh.key", "w") as ssh_key:
            os.fchmod(ssh_key.fileno(), 0o600)
            ssh_key.write(self.get_vault_data(vault_owner, self.ctx["ssh_key"]))

        with open("gpg.public.key", "w") as gpg_public_key:
            os.fchmod(gpg_public_key.fileno(), 0o600)
            gpg_public_key.write(self.get_vault_data(vault_owner, self.ctx["gpg_public_key"]))

        with open("gpg.private.key", "w") as gpg_private_key:
            os.fchmod(gpg_private_key.fileno(), 0o600)
            gpg_private_key.write(self.get_vault_data(vault_owner, self.ctx["gpg_private_key"]))

        gpg_dir = self.path("gnupg")
        os.mkdir(gpg_dir, 0o700)
        environment["GNUPGHOME"] = gpg_dir

        run_process(["gpg", "--import", gpg_public_key.name], environment=environment, log_prefix="gpg-import-public")
        run_process(["gpg", "--import", gpg_private_key.name], environment=environment, log_prefix="gpg-import-private")

        return environment, ssh_key.name

    def __checkout(self, project_name, tag):
        src_dir = self.path("sources")
        git_url = "ssh://git@bb.yandex-team.ru/nanny/{name}.git".format(name=project_name)

        run_process(["git", "clone", git_url, src_dir], log_prefix="git_clone")
        run_process(["git", "checkout", tag], work_dir=src_dir, log_prefix="git_checkout")

        return src_dir


__Task__ = BuildNannyDeb
