# -*- coding: utf-8 -*-

import os
import logging
import tarfile
import shutil
import copy

from sandbox.sandboxsdk import parameters
import sandbox.projects.common.build.parameters as build_params
import sandbox.projects.common.constants as consts
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.svn import Arcadia

from sandbox.projects.common.build.YaMake import YaMakeTask, ya_make_build_params

from sandbox.projects.BuildNews.common import add_file_to_tar
from sandbox.projects.BuildDockerImageV6 import BuildDockerImageV6

from sandbox.projects import resource_types


class TargetInfo:
    def __init__(self, svnpath, destpath=None, binname=None):
        if not destpath:
            destpath = os.path.basename(svnpath)
        self.svnpath = svnpath
        self.destpath = destpath
        self.binname = binname


GROUP_NAME = 'News build options'


class BuildNews(YaMakeTask):
    """
        Сборка бинарников из аркадии для новостей. Список берётся из target_list_file - должен быть
        либо полный svn url, либо путь относительно checkout_arcadia_from_url
    """

    type = 'BUILD_NEWS'

    execution_space = 200 << 10

    class Params(object):
        class TargetList(parameters.SandboxStringParameter):
            name = 'target_list'
            multiline = True
            description = 'Target list in form "arcadia_path result_path build_art_name"'
            group = GROUP_NAME

        class TargetListFile(parameters.SandboxArcadiaUrlParameter):
            name = 'target_list_file'
            description = 'File in repository with build target list (overrides previous field)'
            group = GROUP_NAME
            default_value = ''
            required = False

        class ResourceType(parameters.SandboxSelectParameter):
            name = 'resource_type'
            description = 'Result resource type'
            group = GROUP_NAME

            choices = sorted(((rt.name, rt.name) for rt in resource_types.AbstractResource))
            default_value = resource_types.OTHER_RESOURCE.name

            @classmethod
            def cast(cls, value):
                if not value:
                    return None if cls.required else ''

                for rt in resource_types.AbstractResource:
                    if rt == resource_types.UNKNOWN_RESOURCE:
                        continue
                    if value == rt.name:
                        break
                else:
                    raise ValueError('Resource type %s does not exist' % value)

                return value

        class NoTar(parameters.SandboxBoolParameter):
            name = 'no_tar'
            description = 'Do not tar result'
            group = GROUP_NAME

        class BuildDockerImage(parameters.SandboxBoolParameter):
            name = 'build_docker_image'
            description = '...And build Docker image'
            group = GROUP_NAME

        def create_docker_fields(self):
            blacklist = frozenset(['docker_file_url', 'packaged_resource_id', 'docker_setup_script', 'build_type',
                                   BuildDockerImageV6.ContainerParameter.name])

            docker_params = [p for p in BuildDockerImageV6.input_parameters if p.name not in blacklist]

            class BuildDockerImageGroup(self.BuildDockerImage):
                sub_fields = {
                    'true': [p.name for p in docker_params]
                }

            return [BuildDockerImageGroup] + docker_params

        def __init__(self):
            self.input_parameters = [self.TargetList, self.TargetListFile, self.ResourceType, self.NoTar] + self.create_docker_fields() + build_params.get_arcadia_params() + ya_make_build_params()

    p = Params()

    input_parameters = p.input_parameters

    RESULT_ARCHIVE = 'result.tar'

    def create_result_archve_resource(self, path, arch, attrs):
        """
            Создаёт ресурс архива бинарников
            @path: относительный путь для ресурса
            @arch: архитектура ресурса
            @return: объект созданного ресурса
        """
        resource_description = '%s %s' % (self.descr, path)
        resource = self.create_resource(description=resource_description,
                                        resource_path=path,
                                        resource_type=self.ctx['resource_type'],
                                        arch=arch,
                                        attributes=attrs)
        return resource

    @staticmethod
    def exclude_broken_symlinks(path):
        if os.path.lexists(path):
            return not os.path.exists(path)
        else:
            return False

    def make_result_resource(self, src_dir, build_dir, target_list, attrs={}):
        """
            Создаёт ресурс в виде tar-файла
            @return: объект созданного ресурса
        """

        class FakeTar(object):
            def __init__(self, dirname):
                self.dirname = dirname
                if not os.path.exists(self.dirname):
                    os.makedirs(self.dirname)

            def __enter__(self):
                return self

            def __exit__(*args):
                return False

            def add(self, src, dest, exclude=lambda p: False):
                def ignore(d, names):
                    return [n for n in names if exclude(os.path.join(d, n))]

                dpath = os.path.join(self.dirname, dest)
                if os.path.isdir(src):
                    if dest == '.':
                        dpath = self.dirname
                        os.rmdir(dpath)
                    shutil.copytree(src, dpath, symlinks=False, ignore=ignore)
                elif not exclude(src):
                    shutil.copy(src, dpath)

            def addfile(self, tarinfo, stream):
                with open(os.path.join(self.dirname, tarinfo.name), 'w') as fd:
                    fd.write(stream.read(tarinfo.size))

        resource = self.create_result_archve_resource(self.RESULT_ARCHIVE, self.arch, attrs)

        def open_result():
            if self.ctx.get(self.p.NoTar.name):
                logging.info('Creating directory with binaries')
                return FakeTar(resource.path)
            else:
                logging.info('Create the tar-file with binaries.')
                return tarfile.open(resource.path, 'w', dereference=True)

        with open_result() as f:
            def add_files(src, dest):
                logging.info('Add %s => %s' % (src, dest))
                f.add(src, dest, exclude=self.exclude_broken_symlinks)

            for target in target_list:
                def tgtname(srcname):
                    bn = os.path.basename(srcname)
                    return os.path.join(target.destpath, bn) if target.destpath[-1] == '/' else target.destpath

                if target.binname:
                    add_files(os.path.join(build_dir, target.svnpath, target.binname), tgtname(target.binname))
                else:
                    add_files(os.path.join(src_dir, target.svnpath), tgtname(target.svnpath))

            svndirs = []
            svn_root = Arcadia.normalize_url(self.ctx[consts.ARCADIA_URL_KEY])
            for t in target_list:
                if (not t.binname) and os.path.isdir(os.path.join(src_dir, t.svnpath)):
                    svndirs.append("%s\t%s" % (t.destpath, Arcadia.append(svn_root, t.svnpath)))
            if len(svndirs):
                s = "\n".join(svndirs) + "\n"
                add_file_to_tar(f, 'svndirs', s)
        self.mark_resource_ready(resource)
        return resource

    def read_target_list(self):
        '''
        Парсит target_list_file напрямую из svn.
        '''
        listfile = self.ctx.get(self.p.TargetListFile.name)

        if listfile and ':/' in listfile:
            rev = Arcadia.info(self.ctx['target_list_file'])['entry_revision']
            listurl = Arcadia.replace(self.ctx['target_list_file'], revision=rev)
            data = Arcadia.cat(listurl)
        else:
            url = self.ctx[consts.ARCADIA_URL_KEY]
            parsed_url = Arcadia.parse_url(url)
            rev = parsed_url.revision
            if not rev:
                rev = Arcadia.info(url)['entry_revision']
            if listfile:
                listurl = Arcadia.replace(
                    url,
                    path=os.path.join(parsed_url.path, listfile),
                    revision=rev
                )
                data = Arcadia.cat(listurl)
            else:
                data = self.ctx[self.p.TargetList.name]

        self.ctx['svn_rev'] = rev

        res = []
        for line in data.splitlines():
            fields = line.split()
            if len(fields):
                res.append(TargetInfo(*fields))
        return res

    def get_targets(self):
        return [x.svnpath for x in self.target_list if x.binname is not None]

    def get_arts(self):
        return []

    def get_arts_source(self):
        return []

    def get_resources(self):
        return {}

    def on_enqueue(self):
        YaMakeTask.on_enqueue(self)
        decreased_space_map = {
            resource_types.NEWS_DATA.name: 48 << 10,
            resource_types.NEWS_SCRIPTS.name: 10 << 10,
        }
        decreased_space = decreased_space_map.get(self.ctx['resource_type'], None)
        if decreased_space and (decreased_space < self.execution_space):
            logging.debug("Decrease build execution space: %s -> %s" % (self.execution_space, decreased_space))
            self.execution_space = decreased_space  # Decrease scripts/data build execution space

    def do_execute(self):
        self.ctx['use_aapi_fuse'] = True
        if self.ctx.get(self.p.BuildDockerImage.name):
            logging.info('Docker image enabled => no tar')
            self.ctx[self.p.NoTar.name] = True

        self.target_list = self.read_target_list()
        self.svn_url = self.ctx[build_params.ArcadiaUrl.name]
        self.source_revision = Arcadia.info(self.svn_url)['commit_revision']
        logging.info('Source revision: %s' % self.source_revision)

        if any([x.binname is not None for x in self.target_list]):
            return YaMakeTask.do_execute(self)
        elif any([x.binname is None for x in self.target_list]):
            # только скрипты - делаем выборочный svn export
            src_dir = self.abs_path('scripts')
            make_folder(src_dir)
            path = Arcadia.parse_url(self.svn_url).path
            for target in self.target_list:
                tpath = os.path.join(src_dir, target.svnpath)
                parent = os.path.dirname(tpath)
                if parent != tpath and not os.path.exists(parent):
                    os.makedirs(parent)
                url = Arcadia.replace(self.svn_url, path=os.path.join(path, target.svnpath))
                self.remote_copy(url, tpath)

            self.post_build(src_dir, None, None)

    def fix_tag(self, repo):
        if repo.find(':') >= 0:
            return repo

        pu = Arcadia.parse_url(self.svn_url)

        if pu.branch:
            tag = pu.branch + '@' + str(pu.revision)
        elif pu.tag:
            tag = pu.tag
        else:
            # trunk
            tag = 'r' + str(pu.revision)

        return repo + ':' + tag

    def post_build(self, source_dir, build_dir, pack_dir):
        logging.info('create resource tar-file with all binaries')
        res = self.make_result_resource(source_dir, build_dir, self.target_list,
                                        attrs={'svn_url': self.svn_url,
                                               'svn_rev': self.source_revision})
        self.set_info('Build is completed successfully.')

        if self.ctx.get(self.p.BuildDockerImage.name):
            ctx = copy.deepcopy(self.ctx)
            tags = map(self.fix_tag, ctx[BuildDockerImageV6.RegistryTags.name])
            ctx.update({
                BuildDockerImageV6.PackagedResource.name: res.id,
                BuildDockerImageV6.RegistryTags.name: tags,
            })

            st = self.create_subtask(task_type=BuildDockerImageV6.type, input_parameters=ctx, description=self.descr, inherit_notifications=True)
            self.set_info('Created BuildDockerImage task <a href="/task/{id}">{id}</a>'.format(id=st.id), do_escape=False)


__Task__ = BuildNews
