# -*- coding: utf-8 -*-

import time
import os.path
import shutil

from sandbox.projects import resource_types
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.parameters import LastReleasedResource
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.process import run_process
from sandbox.projects.common.utils import get_bsconfig


class IssShardsPackageParameter(LastReleasedResource):
    name = 'iss_shards_package'
    description = 'iss_shards utility'
    resource_type = resource_types.ISS_SHARDS


class BuildNewsArchiveShard(SandboxTask):
    type = 'BUILD_NEWS_ARCHIVE_SHARD'

    class ShardName(parameters.SandboxStringParameter):
        name = "shard_name"
        description = "Shard name"

    class NewsPackage(parameters.ResourceSelector):
        name = "news_package"
        description = "News package"
        resource_type = resource_types.NEWS_INDEXER_PACKAGE
        required = True

    class ShardConf(parameters.SandboxArcadiaUrlParameter):
        name = "shard_conf"
        description = "Arcadia url for shard conf"
        default_value = "arcadia:/arc/trunk/arcadia/yweb/news/shard_scripts"
        required = False

    input_parameters = [ShardName, NewsPackage, ShardConf, IssShardsPackageParameter]

    def initCtx(self):
        self.ctx['shard_name'] = 'arcnews_robot-000-%d' % int(time.time())

    def on_execute(self):
        shard_name = self.ctx['shard_name']
        shard_dir = self.abs_path(shard_name)
        make_folder(shard_dir)
        Arcadia.export(url=self.ctx['shard_conf'], path=shard_dir)

        iss_shards_tool = self.sync_resource(self.ctx[IssShardsPackageParameter.name])
        run_process(['chmod', '+x', iss_shards_tool])

        pkg_path = self.sync_resource(self.ctx['news_package'])
        shutil.copy(pkg_path, os.path.join(shard_dir, 'pkg.tar'))

        self._subprocess(
            "%s shard_init --torrent --logfile %s %s" % (
                get_bsconfig(), self.log_path('bsconfig.log'), shard_name
            ),
            check=True,
            wait=True,
            log_prefix="bs_config"
        )
        run_process([iss_shards_tool, 'register', './'],
                    log_prefix="iss_shards_register",
                    work_dir=shard_dir
                    )
        self.create_resource(self.descr, shard_dir, 'NEWS_ARCHIVE_WORKER_SHARD')


__Task__ = BuildNewsArchiveShard
