# -*- coding: utf-8 -*-

import os
import re
import tarfile
import logging
from contextlib import closing
import copy
import types
from hashlib import md5

import sandbox.sandboxsdk.parameters as sb_params

from sandbox.projects import resource_types
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.process import run_process

from sandbox.projects.common.build.YaMake import get_arcadia_project_build_params

from sandbox.projects.BuildNews.common import fix_svn_rev as _fix_svn_rev
from sandbox.projects.BuildNews.common import add_file_to_tar

from sandbox.projects.common import apihelpers
from sandbox.projects.common.nanny import nanny
import sandbox.projects.common.build.parameters as build_params

import sandbox.common.rest

NEWS_RESTYPE = resource_types.NEWS_BINS
NEWS_SCRIPTS_RESTYPE = resource_types.NEWS_SCRIPTS
BASESEARCH_RESTYPE = resource_types.NEWS_BASESEARCH_EXECUTABLE
NEWS_DATA_RESTYPE = resource_types.NEWS_DATA
PERL_MODULES_RESTYPE = resource_types.NEWS_PERL_MODULES
CUSTOM_BINS_RESTYPE = resource_types.NEWS_CUSTOM_BINS

ARC_TEST_DATA = 'arcadia:/arc/trunk/arcadia_tests_data'

RES_ID = 'resource_id'


class Md5Calc(object):
    def __init__(self, fileobj):
        self.fileobj = fileobj
        self.md5 = md5()

    def read(self, size=None):
        res = self.fileobj.read(size)
        self.md5.update(res)
        return res

    def hexdigest(self):
        return self.md5.hexdigest()


def format_digests(digests):
    result = ''
    for (name, digest) in digests.iteritems():
        result += '%s *%s\n' % (digest, name)
    return result


def fix_svn_rev(url):
    return url and _fix_svn_rev(url)


def make_build_entries(target_name, title,
                       svn=True, def_flags=True, build=True):
    restype = globals().get(target_name.upper() + '_RESTYPE', resource_types.BUILD_NEWS_RESULT_ARCHIVE)
    def_svn_url = svn if isinstance(svn, types.StringTypes) else None

    def make_build_param(parent, required=False, default=None):
        is_required = required

        class BuildParam(parent):
            group = title
            name = '_'.join((target_name, parent.name))
            required = is_required
            default_value = default if default is not None else parent.default_value
        return BuildParam

    class SvnEntries(object):
        SvnUrl = make_build_param(build_params.ArcadiaUrl, default=def_svn_url)

        DefinitionFlags = make_build_param(build_params.DefinitionFlags)

        params = (
            ([SvnUrl] if svn else []) +
            ([DefinitionFlags] if def_flags else [])
        )

    class PrebuiltEntries(object):
        class PreBuilt(sb_params.LastReleasedResource):
            name = '_'.join((target_name, RES_ID))
            group = title
            description = 'Pre-built resource id'
            resource_type = restype
            required = False

        params = [PreBuilt]

    _title = title

    class SubtaskBuildEntries(SvnEntries, PrebuiltEntries):
        name = target_name
        title = _title

        class BuildFromSvn(sb_params.SandboxStringParameter):
            name = 'do_build_' + target_name
            group = title
            description = 'What to do'
            default_value = 'build'
            choices = [
                ('Build from svn', 'build'),
                ('Use pre-built resource', ''),
            ]
            sub_fields = {
                'build': [x.name for x in SvnEntries.params],
                '': [x.name for x in PrebuiltEntries.params]
            }

        params = [BuildFromSvn] + SvnEntries.params + PrebuiltEntries.params if build else PrebuiltEntries.params

    return SubtaskBuildEntries


class Unpacked(sb_params.SandboxBoolParameter):
    name = 'produce_unpacked'
    description = 'Produce unpacked resources'


class BuildNewsPackage(nanny.ReleaseToNannyTask, SandboxTask):
    '''
    Сборка пакета для новостного индексатора. Запускает задачи по сборке отдельных компонент,
    затем упаковывает их результаты в единый тарбол.
    '''

    type = 'BUILD_NEWS_PACKAGE'

    FILEOWNER = 'root'
    FILEGROUP = 'ynews'
    MODEMASK = 0755
    LISTS_PATH = 'yweb/news/pkg'

    BASESEARCH_TARGET = 'bin/basesearch'

    ST_RES_ID = 'subtask_res_id'

    news_params = make_build_entries('news', 'News binaries', svn=False)
    scripts_params = make_build_entries('news_scripts', 'News scripts', svn=False, def_flags=False)
    basesearch_params = make_build_entries('basesearch', 'Basesearch binary')
    news_data_params = make_build_entries('news_data', 'News data', def_flags=False, svn=ARC_TEST_DATA)
    perl_modules_params = make_build_entries('perl_modules', 'Pre-built perl modules', build=False)
    custom_bins_params = make_build_entries('custom_bins', 'Custom pre-built binaries', build=False)

    custom_params = [
        news_params,
        scripts_params,
        basesearch_params,
        news_data_params,
        perl_modules_params,
        custom_bins_params,
    ]

    execution_space = 60000

    arcadia_url = build_params.ArcadiaUrl

    input_parameters = (
        [arcadia_url, Unpacked, build_params.UseArcadiaApiFuse] +
        get_arcadia_project_build_params() +
        reduce(lambda a, b: a + b, (p.params for p in custom_params))
    )

    def __init__(self, *args, **kwargs):
        SandboxTask.__init__(self, *args, **kwargs)

        self.svndirs = ''
        self.digests = dict()

    def get_svn_branch_or_tag(self, svn_url):
        path = Arcadia.parse_url(svn_url).path
        match = re.match('^arc/((branches|tags)/news/.*)/arcadia', path)
        if match:
            return match.group(1)
        else:
            match = re.match('^arc/(.*)/arcadia', path)
            if match:
                return match.group(1)
            else:
                return svn_url

    def create_subtask_ctx(self, resid, svn_url, def_flags):
        sub_ctx = copy.deepcopy(self.ctx)
        sub_ctx.update({'notify_via': '',
                        build_params.ArcadiaUrl.name: svn_url,
                        self.ST_RES_ID: resid})

        if def_flags:
            sub_ctx.update({
                build_params.DefinitionFlags.name: def_flags,
            })

        if '_log_resource' in sub_ctx:
            del sub_ctx['_log_resource']
        return sub_ctx

    def create_generic_subtask(self, task_type, descr, ctx, tags=None):
        subtask = self.create_subtask(task_type=task_type, input_parameters=ctx, description=descr, arch=self.arch, tags=tags)
        return subtask.id

    @staticmethod
    def make_build_subtask_tags(params):
        new_tag = '-'.join(params.title.split())
        return [new_tag]

    def create_build_subtask(self, params, svn_url, build_list_file, def_flags):
        sub_ctx = self.create_subtask_ctx(params.PreBuilt.name, svn_url, def_flags)
        sub_ctx.update({'target_list_file': build_list_file,
                        'resource_type': str(globals().get(params.name.upper() + '_RESTYPE', resource_types.BUILD_NEWS_RESULT_ARCHIVE))})

        return self.create_generic_subtask('BUILD_NEWS',
                                           "%s from %s" % (params.title, self.get_svn_branch_or_tag(svn_url) or svn_url),
                                           sub_ctx, tags=self.make_build_subtask_tags(params))

    def create_subtask_if_need(self, params):
        if self.ctx[params.BuildFromSvn.name]:
            logging.info('Creating subtask for ' + params.title)
            def_flags_name = params.DefinitionFlags.name
            def_flags = self.ctx[def_flags_name] if def_flags_name in self.ctx else None
            task = self.create_build_subtask(params, self.ctx[params.SvnUrl.name],
                                             Arcadia.append(self.ctx[self.news_params.SvnUrl.name], os.path.join(self.LISTS_PATH, params.name + '.lst')),
                                             def_flags)
            return task

    def create_basesearch_subtask_if_need(self):
        if self.ctx[self.basesearch_params.BuildFromSvn.name]:
            logging.info('Creating subtask for basesearch')
            svn_url = self.ctx[self.basesearch_params.SvnUrl.name]
            sub_ctx = self.create_subtask_ctx(self.basesearch_params.PreBuilt.name, svn_url, self.ctx[self.basesearch_params.DefinitionFlags.name])
            sub_ctx.update({'build_news_basesearch': True,
                            'build_bundle': False
                            })

            return self.create_generic_subtask('BUILD_SEARCH',
                                               'basesearch from %s' % (self.get_svn_branch_or_tag(svn_url) or svn_url),
                                               sub_ctx)

    def tar_cp(self, src, dest, destname=None, prefix=None, white_list_path=None):
        '''
        @src: имя архива-источника или tarfile-объект для него
        @dest: tarfile-объект куда пишется результат
        @destname: если указано - src кладётся как единый файл с этим именем
        @white_list_path: если указано - путь в аркадии до white list'а со списком перепаковываемых файлов
        '''
        whitelist = set()
        if white_list_path:
            for line in Arcadia.cat(white_list_path).splitlines():
                logging.info("Read from whitelist: " + line)
                whitelist.add(line)
            whitelist = frozenset(whitelist)

        if not destname and type(src) != tarfile.TarFile:
            src = tarfile.open(src)

        def process_info(info):
            newinfo = copy.deepcopy(info)
            newinfo.mode &= self.MODEMASK
            newinfo.uname = self.FILEOWNER
            newinfo.gname = self.FILEGROUP
            if prefix:
                newinfo.name = os.path.join(prefix, info.name)
            logging.info('Add %s%s%s' % (info.name,
                                         (' [l] => %s' % info.linkname if info.issym() else ''),
                                         (' as %s' % newinfo.name if prefix else '')))
            if info.isfile():
                with closing(open(src, 'r') if destname else src.extractfile(info)) as data:
                    mcalc = Md5Calc(data)
                    dest.addfile(newinfo, mcalc)
                    self.digests[newinfo.name] = mcalc.hexdigest()
            else:
                dest.addfile(newinfo)

        if not destname:
            info = src.next()
            while info:
                if len(whitelist) > 0 and info.name not in whitelist:
                    info = src.next()
                    continue
                if os.path.basename(info.name) == '.symlinks':
                    logging.info('Creating symlinks from description %s' % info.name)
                    wd = os.path.dirname(info.name)
                    with closing(src.extractfile(info)) as input:
                        for line in input:
                            lsrc, ltgt = line.split()
                            newinfo = tarfile.TarInfo(os.path.join(wd, ltgt))
                            newinfo.type = tarfile.SYMTYPE
                            newinfo.linkname = lsrc
                            process_info(newinfo)
                elif info.name == 'svndirs':
                    self.svndirs += src.extractfile(info).read()
                else:
                    process_info(info)
                info = src.next()
        else:
            info = dest.gettarinfo(src, destname)
            process_info(info)

    def repack_subpackage(self, dest, params, prefix=None, absent_ok=False, apply_yt_whitelist=False):
        logging.info('Repacking ' + params.title)
        name_up = params.name.upper()
        resid = self.ctx[params.PreBuilt.name]
        if absent_ok and not resid and not self.ctx.get(params.BuildFromSvn.name):
            return
        respath = self.sync_resource(resid)
        target = getattr(self, name_up + '_TARGET', None)

        whitelist = Arcadia.append(self.ctx[self.news_params.SvnUrl.name], os.path.join(self.LISTS_PATH, params.name + '_yt.lst')) if apply_yt_whitelist else None
        self.tar_cp(respath, dest, destname=target, prefix=prefix, white_list_path=whitelist)

    def store_subtask_resource_id(self, st, restype):
        if not st.is_finished():
            raise SandboxTaskFailureError('Subtask %s has failed (status=%s)' % (st.descr, repr(st.status)))
        resid_name = st.ctx[self.ST_RES_ID]
        resources = apihelpers.list_task_resources(task_id=st.id, resource_type=restype, arch=self.client_info['arch'])
        if len(resources) != 1:
            raise SandboxTaskFailureError("Subtask %s doesn't have result resource %s (or has more than one)"
                                          % (st.descr, restype))

        self.ctx[resid_name] = resources[0].id

    @staticmethod
    def get_svn_tag_branch(path):
        match = re.match('^arc/(branches|tags)/(.*)/arcadia', path)
        if match:
            name = match.group(2)
            if match.group(1) == 'branches':
                return None, name
            else:
                return name, None
        else:
            return None, None

    def arcadia_info_path_rev(self):
        if self.scripts_params.PreBuilt.name in self.ctx:
            res = channel.sandbox.get_resource(self.ctx[self.scripts_params.PreBuilt.name])
            if res is not None:
                svn_path = Arcadia.parse_url(res.attributes['svn_url']).path
                svn_rev = res.attributes['svn_rev']
                return svn_path, svn_rev
        parsed_url = Arcadia.parse_url(self.ctx[self.news_params.SvnUrl.name])
        svn_path = parsed_url.path
        svn_rev = parsed_url.revision
        return svn_path, svn_rev

    def arcadia_info(self):
        svn_path, svn_rev = self.arcadia_info_path_rev()
        tag, branch = self.get_svn_tag_branch(svn_path)

        return svn_rev, tag, branch

    def get_version_str(self):
        res = channel.sandbox.get_resource(self.ctx[self.scripts_params.PreBuilt.name])
        path = Arcadia.parse_url(res.attributes['svn_url']).path
        ver = None
        needrev = True
        tag, branch = self.get_svn_tag_branch(path)
        if tag:
            needrev = False
            tag = re.sub('^news/', '', tag)
            match = re.match('^releases/([^/]+/)?(\d+\w*([-/]\d+)*)', tag)
            if match:
                ver = match.group(2)
        elif branch:
            branch = re.sub('^news/', '', branch)
            match = re.match('^releases/release-(.*)', branch)
            if match:
                ver = match.group(1)

        if ver:
            ver = re.sub('[-/]', '.', ver)
        else:
            match = re.match('^arc/trunk', path)
            if match:
                ver = 'trunk'
            else:
                ver = tag or branch or 'unknown'
                ver = re.sub('/', '-', ver)
                needrev = True

        if needrev:
            rev = 0
            for params in self.custom_params:
                res = channel.sandbox.get_resource(self.ctx[params.PreBuilt.name])
                if res and 'svn_rev' in res.attributes:
                    rev = max(rev, int(res.attributes['svn_rev']))
            ver += '-r%d' % rev
        return ver

    def produce_unpacked(self, tar_res, restype):
        result = self.create_resource(description=tar_res.description + ' - unpacked',
                                      resource_path='.'.join(tar_res.path.split('.')[:-1]),
                                      resource_type=restype,
                                      arch=tar_res.arch)
        make_folder(result.path, delete_content=True)
        logging.info('Unpacking %s => %s' % (tar_res.path, result.path))
        run_process(['tar', '-xvf', tar_res.path], work_dir=result.path)
        for (path, dirs, files) in os.walk(result.path):
            for f in files + dirs:
                fp = os.path.join(path, f)
                if os.path.islink(fp):
                    target = os.readlink(fp)
                    if target.startswith('/') and not target.startswith(result.path):
                        logging.info('Found bad symlink, removing: %s => %s' % (fp, target))
                        os.unlink(fp)
        self.mark_resource_ready(result.id)

    def add_version_hint(self, version):
        if version:
            try:
                version = str(version)
                client = sandbox.common.rest.Client()
                hints = [version]
                match = re.match('^(\d+)\.(\d+)$', version)
                if match:
                    hints.append(match.group(1))
                client.task[self.id].hints(hints)
            except Exception as e:
                logging.debug("Unable to add version hint via rest api: %s", str(e))

    def on_execute(self):
        if 'BUILD_ENQUEUED' not in self.ctx:
            svn_name = self.arcadia_url.name
            self.ctx['news_' + svn_name] = fix_svn_rev(self.ctx[svn_name])
            self.ctx['news_scripts_' + svn_name] = self.ctx['news_' + svn_name]
            newsdata_svn = 'news_data_' + svn_name
            if newsdata_svn in self.ctx:
                self.ctx[newsdata_svn] = fix_svn_rev(self.ctx[newsdata_svn])

            self.create_subtask_if_need(self.news_params)
            self.create_subtask_if_need(self.scripts_params)
            self.create_basesearch_subtask_if_need()
            self.create_subtask_if_need(self.news_data_params)

            subtasks = self.list_subtasks(load=False)
            if len(subtasks):
                self.ctx['BUILD_ENQUEUED'] = True
                self.wait_tasks(subtasks, self.Status.Group.SUCCEED + self.Status.Group.SCHEDULER_FAILURE, wait_all=True, state='Waiting for build tasks to complete')

        if 'BUILD_ENQUEUED' in self.ctx:
            for st in self.list_subtasks(load=True):
                if st.type == 'BUILD_NEWS':
                    self.store_subtask_resource_id(st, st.ctx['resource_type'])
                elif st.type == 'BUILD_SEARCH':
                    self.store_subtask_resource_id(st, resource_types.NEWS_BASESEARCH_EXECUTABLE)

        ver = self.get_version_str()
        self.add_version_hint(ver)

        resource = self.create_resource(description='News indexer package - ' + self.descr,
                                        resource_path='news-indexer-%s.tar' % ver,
                                        resource_type=resource_types.NEWS_INDEXER_PACKAGE,
                                        arch=self.arch)

        with closing(tarfile.open(resource.path, 'w')) as f:
            add_file_to_tar(f, 'version', ver)
            self.repack_subpackage(f, self.news_params)
            self.repack_subpackage(f, self.scripts_params)
            self.repack_subpackage(f, self.basesearch_params)
            self.repack_subpackage(f, self.perl_modules_params)
            self.repack_subpackage(f, self.news_data_params)
            self.repack_subpackage(f, self.custom_bins_params, prefix='custom', absent_ok=True)
            add_file_to_tar(f, 'svndirs', self.svndirs)
            add_file_to_tar(f, 'MD5SUMS', format_digests(self.digests))

        yt_resource = self.create_resource(description='YT News builder package - ' + self.descr,
                                           resource_path='news-builder-%s.tar.gz' % ver,
                                           resource_type=resource_types.NEWS_INDEXER_YT_PACKAGE,
                                           arch=self.arch)

        with closing(tarfile.open(yt_resource.path, 'w:gz')) as f:
            add_file_to_tar(f, 'version', ver)
            self.repack_subpackage(f, self.news_params, apply_yt_whitelist=True)
            self.repack_subpackage(f, self.scripts_params)
            self.repack_subpackage(f, self.perl_modules_params)
            self.repack_subpackage(f, self.news_data_params, apply_yt_whitelist=True)
            add_file_to_tar(f, 'MD5SUMS', format_digests(self.digests))

        if self.ctx.get(Unpacked.name):
            logging.info("Producing unpacked resources")
            self.produce_unpacked(resource, resource_types.NEWS_INDEXER_PACKAGE_UNPACK)
            self.produce_unpacked(yt_resource, resource_types.NEWS_INDEXER_YT_PACKAGE_UNPACK)

        self.set_info('Build is completed successfully.')


__Task__ = BuildNewsPackage
