# -*- coding: utf-8 -*-

from sandbox.projects import resource_types

from sandbox.projects.common.news.svn import checkout_file

from sandbox.projects.common import utils
from sandbox.projects.common import apihelpers
from sandbox.sandboxsdk import sandboxapi
from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import task as st
from sandbox.sandboxsdk.channel import channel

import os
import logging
import shutil
import json


class YtProxy(sp.SandboxStringParameter):
    name = 'proxy'
    description = 'YT proxy'
    required = True
    default_value = "hahn.yt.yandex.net"


class Builder(sp.ResourceSelector):
    name = 'builder_resource'
    description = 'Builds news search shard'
    required = True
    resource_type = resource_types.NEWS_SEARCH_SHARD_BUILDER


class VaultItemOwner(sp.SandboxStringParameter):
    name = 'vault_item_owner'
    description = 'YT token owner in vault'
    required = True
    default_value = "NEWS"


class VaultItemName(sp.SandboxStringParameter):
    name = 'vault_item_name'
    description = 'YT resource key in vault'
    required = True
    default_value = 'yt_token'


class ShardNumber(sp.SandboxIntegerParameter):
    name = 'shard_number'
    description = 'Shard number'
    required = True
    default_value = 0


class ShardName(sp.SandboxStringParameter):
    name = 'shard_name'
    description = 'Shard name'
    required = True
    default_value = 'index'


class CypressPath(sp.SandboxStringParameter):
    name = 'cypress_path'
    description = 'Path to the Cypress node where index tables are located'
    required = True
    default_value = '//home/news-prod/archive/index/shards/latest'


class GrattrConf(sp.SandboxStringParameter):
    name = 'grattr_conf'
    description = 'Path in svn to the grattr config'
    required = True
    default_value = 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/yweb/news/config/archive/clatrbase_config'


class Ttl(sp.SandboxIntegerParameter):
    name = 'ttl'
    description = 'TTL of shard resource'
    required = True
    default_value = 14


class Restore(sp.SandboxBoolParameter):
    name = 'restore'
    description = 'Restore this shard'
    required = True
    default_value = False


class BuildNewsSearchShard(st.SandboxTask):
    '''
    Build search index files for one shard from the specified YT tables
    '''

    type = 'BUILD_NEWS_SEARCH_SHARD'

    input_parameters = (
        YtProxy,
        Builder,
        VaultItemOwner,
        VaultItemName,
        ShardNumber,
        ShardName,
        CypressPath,
        GrattrConf,
        Ttl,
        Restore
    )

    cores = 1
    required_ram = 10240  # 10GB

    def _get_iss_shards_tool(self):
        try:
            tool_id = utils.get_and_check_last_released_resource_id(
                resource_types.ISS_SHARDS,
                arch=sandboxapi.ARCH_ANY
            )
            return self.sync_resource(tool_id)
        except Exception as e:
            logging.error("Cannot get latest stable ISS_SHARDS tool: %s", e)

    def calc_shard_md5(self, directory):
        md5 = {}
        p = process.run_process(
            "md5sum index* *.d2c *.c2n",
            shell=True,
            work_dir=directory,
            log_prefix='md5sum',
            outputs_to_one_file=False,
        )
        with open(p.stdout_path) as fd:
            for line in fd:
                line = line.rstrip()
                (md5sum, filename) = line.split("  ")
                md5[filename] = md5sum
        return md5

    def on_execute(self):
        from sandbox.yasandbox.api.xmlrpc.resource import touch_resource

        restore = self.ctx[Restore.name]

        builder_exec = os.path.join(self.sync_resource(self.ctx[Builder.name]), "shard_builder")
        os.chmod(builder_exec, 0775)
        grattr_conf_file = checkout_file(self.ctx[GrattrConf.name], self.path("config"))

        shard_dir = self.path('shard_data')
        if os.path.exists(shard_dir):
            shutil.rmtree(shard_dir)
        os.makedirs(shard_dir)

        env = os.environ.copy()
        env['YT_TOKEN'] = self.get_vault_data(self.ctx[VaultItemOwner.name], self.ctx[VaultItemName.name])

        cmd = [
            builder_exec,
            '--proxy', self.ctx[YtProxy.name],
            '-c', self.ctx[CypressPath.name],
            '--number', str(self.ctx[ShardNumber.name]),
            '-g', grattr_conf_file,
        ]

        process.run_process(cmd, work_dir=shard_dir, environment=env, log_prefix='shard_builder')

        shard_md5 = self.calc_shard_md5(shard_dir)

        last_shard = apihelpers.get_last_resource_with_attribute(
            resource_type=resource_types.NEWS_SEARCH_SHARD,
            attribute_name=ShardNumber.name,
            attribute_value=str(self.ctx[ShardNumber.name]),
            status='READY',
        )
        if last_shard is not None and not restore:
            md5_json_str = channel.sandbox.get_resource_attribute(last_shard.id, attribute_name="shard_md5")
            last_shard_md5 = json.loads(md5_json_str)
            if cmp(shard_md5, last_shard_md5) == 0:
                touch_resource(last_shard.id)
                return

        iss_shards = self._get_iss_shards_tool()
        process.run_process(
            [
                iss_shards,
                'configure',
                shard_dir,
                '--id',
                self.ctx[ShardName.name]
            ],
            work_dir=self.path(),
            log_prefix='iss_shards_configure'
        )

        process.run_process(
            [
                iss_shards,
                'register',
                shard_dir
            ],
            work_dir=self.path(),
            log_prefix='iss_shards_register',
            check=not restore
        )

        resource = self.create_resource(description='News search shard - ' + self.ctx[ShardName.name],
                                        resource_path=shard_dir,
                                        resource_type=resource_types.NEWS_SEARCH_SHARD,
                                        arch='any',
                                        attributes={
            'ttl': self.ctx[Ttl.name],
            'shard_name': self.ctx[ShardName.name],
            'shard_number': self.ctx[ShardNumber.name],
            'shard_md5': json.dumps(shard_md5, ensure_ascii=False).encode('utf-8'),
        })

        self.mark_resource_ready(resource.id)


__Task__ = BuildNewsSearchShard
