# coding: utf-8
import os
import shutil
import multiprocessing as mp

import sandbox.sandboxsdk.process as sdk_process
from sandbox.projects import resource_types
from sandbox.projects.common import environments
from sandbox.projects.common.juggler import base
from sandbox.common.types.misc import DnsType
from sandbox.common.types.client import Tag
from sandbox.sandboxsdk import ssh


class GitTagParameter(base.GitTagParameter):
    default_value = 'yandex-stable-1.8'


class BuildNginxBinary(base.BaseBuildJugglerTask):
    """
    Build Nginx statically from sources.
    """

    type = "BUILD_NGINX"
    dns = DnsType.DNS64
    client_tags = Tag.LINUX_PRECISE

    REPO_NAME = "nginx"
    GIT_URL_TEMPLATE = "https://github.yandex-team.ru/InfraComponents/{name}.git"

    PCRE_RESOURCE_ID = 109325093
    LIBOPENSSL_RESOURCE_ID = 195130964

    VAULT_OWNER = "JUGGLER"
    SSH_KEY_VAULT_NAME = "robot-juggling-ssh"

    environment = base.BaseBuildJugglerTask.environment + [
        environments.SandboxLibOpensslEnvironment(),
    ]
    input_parameters = [GitTagParameter]

    def _get_version(self):
        with open(os.path.join(self._checkout_path, 'src', 'core', 'nginx.h')) as header:
            for s in header.read().split('\n'):
                if len(s.split()) > 1 and s.split()[1] == "NGINX_VERSION":
                    return s.split()[-1].strip('"')

    def _get_sandbox(self, name, resource_id, srcpath):
        with self.current_action("Downloading {0} from sandbox".format(name)):
            resource_path = self.sync_resource(resource_id)
        with self.current_action("Extracting sources to '{0}'".format(srcpath)):
            if os.path.exists(srcpath):
                shutil.rmtree(srcpath)
            os.mkdir(srcpath)
            sdk_process.run_process(
                ["tar", "xf", resource_path, "--strip-components=1", "--directory", srcpath],
                log_prefix="extract_{0}".format(name)
            )

    def _get_git(self, name, url, srcpath):
        with self.current_action("Cloning git repo for {0}".format(name)):
            if os.path.exists(srcpath):
                shutil.rmtree(srcpath)
            os.mkdir(srcpath)
            sdk_process.run_process(
                ["git", "clone", url, srcpath],
                log_prefix="clone_{0}".format(name)
            )

    def on_execute(self):
        self._checkout()

        lua_path = self.abs_path("lua")
        with ssh.Key(self, self.VAULT_OWNER, self.SSH_KEY_VAULT_NAME):
            self._get_git("lua", "ssh://robot-juggling@bb.yandex-team.ru/juggler/lua.git", lua_path)

        pcre_path = self.abs_path("pcre")
        self._get_sandbox("pcre", self.PCRE_RESOURCE_ID, pcre_path)

        openssl_path = self.abs_path("openssl")
        self._get_sandbox("pcre", self.LIBOPENSSL_RESOURCE_ID, openssl_path)

        with self._compiler_environ():
            with self.current_action("Building lua"):
                env = os.environ.copy()
                env["MYCFLAGS"] = "-ULUA_USE_DLOPEN"
                sdk_process.run_process(
                    ["make", "all", "-j", str(mp.cpu_count()), "-C", os.path.join(lua_path, "src"), "MYCFLAGS=-ULUA_USE_DLOPEN"],
                    work_dir=lua_path, environment=env, log_prefix="build_lua"
                )

            with self.current_action("Configuring nginx"):
                env = os.environ.copy()
                env["LUA_INC"] = os.path.join(lua_path, 'src')
                env["LUA_LIB"] = os.path.join(lua_path, 'src')
                sdk_process.run_process(
                    ["auto/configure",
                    "--prefix=.",
                    "--with-cc-opt=-O2",
                    "--with-cc-opt=-DNGX_LUA_ABORT_AT_PANIC",
                    "--with-ld-opt='-L{0} -Wl,-Bstatic -llua -lcontrib-newssl-openssl -Wl,-Bdynamic'".format(os.path.join(lua_path, "src")),
                    "--with-debug",
                    "--with-http_stub_status_module",
                    "--with-http_ssl_module",
                    "--with-http_realip_module",
                    "--with-ipv6",
                    "--with-threads",
                    "--with-file-aio",
                    "--with-yandex-newssl",
                    "--with-yandex-newssl-dir={}".format(openssl_path),
                    "--add-module=debian/modules/nginx-echo",
                    "--add-module=debian/modules/nginx-lua",
                    "--with-pcre={}".format(pcre_path),
                    ], work_dir=self._checkout_path, environment=env, log_prefix="configure_nginx"
                )

            with self.current_action("Building nginx"):
                sdk_process.run_process(
                    ["make", "-j", str(mp.cpu_count()), "-C", self._checkout_path],
                    log_prefix="make_nginx", work_dir=self._checkout_path
                )

        version = self._get_version()
        self._make_resource(
            path=os.path.join(self._checkout_path, "objs", "nginx"),
            resource_type=resource_types.NGINX_BINARY,
            description='nginx binary {0}, {1}'.format(self.ctx[GitTagParameter.name], version),
            version=version)


__Task__ = BuildNginxBinary
