# coding: utf-8

import sandbox.common.types.misc as ctm
from sandbox.common.types.client import Tag
from sandbox.projects.common.nanny import nanny
from sandbox.sandboxsdk.parameters import SandboxStringParameter
import os
import sandbox.sandboxsdk.process as sdk_process
import sandbox.sandboxsdk.task as sdk_task


class NGINXSourceVersion(SandboxStringParameter):
    name = 'nginx_source_version'
    description = 'Version of Nginx'
    default_value = '1.13.3'


class LibreSSLSourceVersion(SandboxStringParameter):
    name = 'libressl_source_version'
    description = 'Version of LibreSSL'
    default_value = '2.5.5'


class PCRESourceVersion(SandboxStringParameter):
    name = 'pcre_source_version'
    description = 'Version of PCRE'
    default_value = '8.41'


class ZLIBSourceVersion(SandboxStringParameter):
    name = 'zlib_source_version'
    description = 'Version of ZLIB'
    default_value = '1.2.11'


class BrotliSourceVersion(SandboxStringParameter):
    name = 'brotli_source_version'
    description = 'Version of NgxBrotli'
    default_value = 'master'


class LibBrotliSourceVersion(SandboxStringParameter):
    name = 'libbrotli_source_version'
    description = 'Version of LibBrotli'
    default_value = '1.0.7'


class BuildHAProxy(nanny.ReleaseToNannyTask, sdk_task.SandboxTask):
    '''Build mainline version of Nginx statically from sources'''

    type = 'BUILD_NGINX_CPLB'
    client_tags = Tag.LINUX_PRECISE
    dns = ctm.DnsType.DNS64
    input_parameters = [
        NGINXSourceVersion,
        LibreSSLSourceVersion,
        PCRESourceVersion,
        ZLIBSourceVersion,
        LibBrotliSourceVersion,
        BrotliSourceVersion,
    ]

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)

    def on_execute(self):
        nginx_source_url = 'http://nginx.org/download/nginx-%s.tar.gz'
        nginx_source_ver = self.ctx['nginx_source_version']

        libressl_source_url = 'http://ftp.openbsd.org/pub/OpenBSD/LibreSSL/libressl-%s.tar.gz'
        libressl_source_ver = self.ctx['libressl_source_version']

        pcre_source_url = 'https://ftp.pcre.org/pub/pcre/pcre-%s.tar.gz'
        pcre_source_ver = self.ctx['pcre_source_version']

        zlib_source_url = 'http://zlib.net/zlib-%s.tar.gz'
        zlib_source_ver = self.ctx['zlib_source_version']

        brotli_source_url = 'https://codeload.github.com/google/ngx_brotli/zip/%s'
        brotli_source_ver = self.ctx['brotli_source_version']

        libbrotli_source_url = 'https://github.com/google/brotli/archive/v%s.tar.gz'
        libbrotli_source_ver = self.ctx['libbrotli_source_version']

        current_dir = os.path.abspath('./')
        libressl_dir = os.path.join(current_dir, 'libressl-%s' % libressl_source_ver)
        nginx_dir = os.path.join(current_dir, 'nginx-%s' % nginx_source_ver)
        pcre_dir = os.path.join(current_dir, 'pcre-%s' % pcre_source_ver)
        zlib_dir = os.path.join(current_dir, 'zlib-%s' % zlib_source_ver)
        brotli_dir = os.path.join(current_dir, 'ngx_brotli-%s' % brotli_source_ver)
        libbrotli_dir = os.path.join(current_dir, 'brotli-%s' % libbrotli_source_ver)

        sdk_process.run_process(
            ['wget', '-q', libressl_source_url % libressl_source_ver],
            log_prefix='get_libressl'
        )

        sdk_process.run_process(
            ['tar', '-xf', 'libressl-%s.tar.gz' % libressl_source_ver],
            log_prefix='tar_libressl'
        )

        sdk_process.run_process(
            ['wget', '-q', pcre_source_url % pcre_source_ver],
            log_prefix='get_pcre'
        )

        sdk_process.run_process(
            ['tar', '-xf', 'pcre-%s.tar.gz' % pcre_source_ver],
            log_prefix='tar_pcre'
        )

        sdk_process.run_process(
            ['wget', '-q', zlib_source_url % zlib_source_ver],
            log_prefix='get_zlib'
        )

        sdk_process.run_process(
            ['tar', '-xf', 'zlib-%s.tar.gz' % zlib_source_ver],
            log_prefix='tar_zlib'
        )

        sdk_process.run_process(
            ['wget', '-q', '-O', 'ngx_brotli-%s.zip' % brotli_source_ver,  brotli_source_url % brotli_source_ver],
            log_prefix='get_brotli'
        )

        sdk_process.run_process(
            ['unzip', 'ngx_brotli-%s.zip' % brotli_source_ver],
            log_prefix='unzip_brotli'
        )

        sdk_process.run_process(
            ['wget', '-q', '-O', 'brotli-%s.tar.gz' % libbrotli_source_ver,  libbrotli_source_url % libbrotli_source_ver],
            log_prefix='get_libbrotli'
        )

        sdk_process.run_process(
            ['tar', '-xf', 'brotli-%s.tar.gz' % libbrotli_source_ver],
            log_prefix='tar_libbrotli'
        )

        sdk_process.run_process(
            ['wget', '-q', nginx_source_url % nginx_source_ver],
            log_prefix='get_nginx'
        )

        sdk_process.run_process(
            ['tar', '-xf', 'nginx-%s.tar.gz' % nginx_source_ver],
            log_prefix='tar_nginx'
        )

        os.chdir(libressl_dir)

        sdk_process.run_process(
            ['./configure', 'LDFLAGS=-lrt', '--prefix=%s/.openssl/' % libressl_dir],
            work_dir=libressl_dir,
            log_prefix='configure_libressl'
        )

        sdk_process.run_process(
            ['make', 'install-strip'],
            work_dir=libressl_dir,
            log_prefix='make_libressl'
        )

        os.chdir(zlib_dir)

        sdk_process.run_process(
            ['./configure', '--prefix=%s/.zlib' % zlib_dir],
            work_dir=zlib_dir,
            log_prefix='configure_zlib'
        )

        sdk_process.run_process(
            ['make', 'install'],
            work_dir=zlib_dir,
            log_prefix='make_zlib'
        )

        os.chdir(current_dir)

        brotli_deps_dir = os.path.join(brotli_dir, 'deps/brotli')


        sdk_process.run_process(
            ['mv %s/* %s' % (libbrotli_dir, brotli_deps_dir)],
            shell=True,
            log_prefix="move_libbrotli"
        )

        os.chdir(nginx_dir)

        sdk_process.run_process(
            [
                './configure',
                '--with-openssl=%s' % libressl_dir,
                '--add-module=%s' % brotli_dir,
                '--conf-path=./nginx.conf',
                '--error-log-path=/place/db/www/logs/error.log',
                '--http-client-body-temp-path=./tmp/body',
                '--http-fastcgi-temp-path=./tmp/fastcgi',
                '--http-log-path=/place/db/www/logs/access.log',
                '--http-proxy-temp-path=./tmp/proxy',
                '--http-scgi-temp-path=./tmp/scgi',
                '--http-uwsgi-temp-path=./tmp/uwsgi',
                '--lock-path=./tmp/nginx.lock',
                '--pid-path=./tmp/nginx.pid',
                '--with-cc-opt=-O2',
                '--with-file-aio',
                '--with-http_gunzip_module',
                '--with-http_gzip_static_module',
                '--with-http_realip_module',
                '--with-http_slice_module',
                '--with-http_ssl_module',
                '--with-http_stub_status_module',
                '--with-http_v2_module',
                '--with-ld-opt=-Wl,--no-as-needed,-lrt',
                '--with-pcre-jit',
                '--with-pcre=%s' % pcre_dir,
                '--with-stream',
                '--with-stream_realip_module',
                '--with-stream_ssl_module',
                '--with-stream_ssl_preread_module',
                '--with-threads',
                '--with-zlib=%s' % zlib_dir,
                '--without-http_ssi_module',
                '--without-mail_imap_module',
                '--without-mail_pop3_module',
                '--without-mail_smtp_module',
                '--without-poll_module',
                '--without-select_module'
            ],
            work_dir=nginx_dir,
            log_prefix='configure_nginx'
        )

        sdk_process.run_process(
            ['touch', '%s/.openssl/include/openssl/ssl.h' % libressl_dir],
            work_dir=libressl_dir,
        )

        sdk_process.run_process(
            ['make', '-j1'],
            work_dir=nginx_dir,
            log_prefix='make_nginx'
        )

        self.create_resource(
            arch='linux',
            description=self.descr,
            resource_path='%s/objs/nginx' % nginx_dir,
            resource_type='NGINX_EXECUTABLE',
        )


__Task__ = BuildHAProxy
