# -*- coding: utf-8 -*-

import os
import shutil
import tarfile
import time

import sandbox.common.types.client as ctc

from sandbox.projects import resource_types
from sandbox.projects.common.utils import get_bsconfig
from sandbox.projects.common.BaseBuildPythonBundleTask import BaseBuildPythonBundleTask
from sandbox.sandboxsdk.parameters import SandboxGitUrlParameter
from sandbox.sandboxsdk.paths import remove_path
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.task import SandboxTask


class NishooterRepositoryPath(SandboxGitUrlParameter):
    name = 'nishooter_repository_path'
    description = 'GIT repository with nishooter code'
    default_value = 'https://git.qe-infra.yandex-team.ru/scm/search_infra/nishooter.git'


class BuildNishooter(BaseBuildPythonBundleTask):
    type = 'BUILD_NISHOOTER'

    client_tags = ctc.Tag.Group.LINUX

    owners = ['darkk', 'ivaxer', 'seege']

    REPO_URL_PARAMETER = NishooterRepositoryPath

    RESOURCE_TYPE_VENV_ARCHIVE = resource_types.NISHOOTER
    RESOURCE_TYPE_VENV_SHARD = resource_types.NISHOOTER_FRONT_SHARD

    CODE_NAME = 'nishooter'
    VENV_SHARD_NAME = 'nishooter_shard'
    VENV_ARCHIVE_FILE = 'nishooter.tar.gz'


class BuildNishooterFront(SandboxTask):
    '''
        Сборка http-фронта для nishooter.yandex-team.ru
    '''
    type = 'BUILD_NISHOOTER_FRONT'
    input_parameters = []

    CHECKOUT_PATH = 'nishooter'
    TGZ_NAME = 'nishooter_front.tar.gz'

    TAG = 'testtag'

    def __checkout(self):
        tmp_dir = Arcadia.get_arcadia_src_dir('arcadia:/arc/trunk/arcadia/infra/clusterload')
        shutil.copytree(tmp_dir, self.CHECKOUT_PATH)

    def __create_shard(self):
        shard_name = 'nishooter_front-{0}-{1}'.format(self.TAG, time.strftime("%Y%m%d_%H%M%S"))
        self.ctx['shard_name'] = shard_name

        shard_path = self.path(shard_name)
        remove_path(shard_path)
        os.mkdir(shard_path)

        with tarfile.open(os.path.join(shard_path, self.TGZ_NAME), 'w:gz') as tar:
            for entry in os.listdir(self.CHECKOUT_PATH):
                tar.add(os.path.join(self.CHECKOUT_PATH, entry), entry)

        with open(os.path.join(shard_path, 'shard.conf'), 'w') as fd:
            # FIXME: virtualenv should be rather built at sandbox like yasm-agent does
            fd.write(
                '%install\n'
                'tar -xzf {}\n'
                '/bin/bash get_ve.sh\n'.format(self.TGZ_NAME))
        self.create_resource(
            description='nishooter_front {0} shard'.format(self.TAG),
            resource_path=shard_path,
            resource_type=resource_types.NISHOOTER_FRONT_SHARD,
        )
        run_process(
            ['perl', get_bsconfig(), 'shard_init', '--torrent', shard_name],
            log_prefix="bs_config",
            work_dir=self.path()
        )

    def arcadia_info(self):
        """
        ???: Hacky way to allow this task to be released: provide tag, other fields are not checked.
        """
        return 'testrev', self.TAG, 'trunk'

    def on_execute(self):
        self.__checkout()
        self.__create_shard()


__Task__ = BuildNishooter
