# -*- coding: utf-8 -*-
import logging
import shutil
import tarfile
import json
from os import path, listdir
from contextlib import closing
from sandbox.projects import resource_types
from sandbox.projects.common.arcadia import sdk
import sandbox.projects.common.constants as consts
from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask
from sandbox.sandboxsdk.parameters import SandboxSvnUrlParameter
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.paths import remove_path
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.channel import channel


class OcrArcadiaParameter(SandboxSvnUrlParameter):
    name = 'checkout_ocr_arcadia'
    description = 'SVN URL (to arcadia)'
    default_value = 'arcadia:/arc/trunk/arcadia/'
    group = 'SVN parameters'


class OcrArcadiaDataParameter(SandboxSvnUrlParameter):
    name = 'checkout_ocr_data'
    description = 'SVN URL (to arcadia data)'
    default_value = 'arcadia:/arc/trunk/arcadia_tests_data/images/ocr'
    group = 'SVN parameters'


class OcrArcadiaReleaseParameter(SandboxStringParameter):
    name = 'release_ocr_data'
    description = 'Release target with ocr_runner'
    default_value = ''
    group = 'SVN parameters'


class OcrDataSubpathParameter(SandboxStringParameter):
    name = 'ocr_data_subpath'
    description = 'Path to data folder to copy it'
    default_value = './'
    group = 'SVN parameters'


class OcrLocalYmakeParameter(SandboxStringParameter):
    name = 'ocr_local_ymake'
    description = 'Local ymake'
    default_value = ''
    group = 'Build parameters'
    multiline = True


class BuildOcr(ArcadiaTask):
    type = 'BUILD_OCR'

    OCR_BIN_PATH = 'cv/imageproc/ocr/tools/statistic/auto_tests'

    OCR_RESULT_PATH = 'result'
    OCR_RESULT_BIN_PATH = 'result/bin'
    OCR_RESULT_DATA_PATH = 'result/data'
    OCR_ALL_DATA_PATH = 'allData'
    OCR_SCRIPT_TO_CREATE_DATA = './paths.sh'

    OCR_ALLOW_BIN = ['yweb', 'cv', 'local.ymake']

    OCR_SVN_INFO_RESULT = './svn.info'

    input_parameters = [OcrArcadiaParameter, OcrArcadiaDataParameter, OcrDataSubpathParameter, OcrArcadiaReleaseParameter, OcrLocalYmakeParameter]
    execution_space = 30000

    def on_execute(self):
        logging.info('Start')
        logging.info('Check svn paths')
        self.check_all_paths()
        logging.info('Check svn paths : Done')

        logging.info('Arcadia export')
        codeSvnInfo = Arcadia.info(self.ctx[OcrArcadiaParameter.name])
        dataSvnInfo = Arcadia.info(self.ctx[OcrArcadiaDataParameter.name])

        logging.info(json.dumps(codeSvnInfo))
        logging.info(json.dumps(dataSvnInfo))

        arcadia_src_dir = Arcadia.get_arcadia_src_dir(self.ctx[OcrArcadiaParameter.name])
        if not arcadia_src_dir:
            raise SandboxTaskFailureError(
                'Cannot get repo for url {0}'.format(self.ctx[OcrArcadiaParameter.name])
            )
        logging.info('Arcadia export : Done')

        logging.info('Create folder for result')
        make_folder(self.OCR_RESULT_PATH, True)
        make_folder(self.OCR_RESULT_BIN_PATH, True)
        remove_path(self.OCR_RESULT_DATA_PATH)
        with open(path.join(self.OCR_RESULT_PATH, self.OCR_SVN_INFO_RESULT), "w") as fOut:
            fOut.write(json.dumps({"CodeInfo": codeSvnInfo, "DataInfo": dataSvnInfo}))
        logging.info('Create folder for result : Done')

        logging.info('Create local.ymake')
        with open(path.join(self.OCR_RESULT_BIN_PATH, "local.ymake"), "w") as fYmake:
            fYmake.write(self.ctx[OcrLocalYmakeParameter.name])
        logging.info('Create local.ymake : done')

        targets = []
        logging.info('Build all')
        targets.append(self.OCR_BIN_PATH)
        if (self.ctx[OcrArcadiaReleaseParameter.name] != ''):
            targets.append(path.dirname(self.ctx[OcrArcadiaReleaseParameter.name]))
        self.build_all(arcadia_src_dir, targets, self.OCR_RESULT_BIN_PATH)

        for elem in listdir(self.OCR_RESULT_BIN_PATH):
            if not(elem in self.OCR_ALLOW_BIN):
                remove_path(path.join(self.OCR_RESULT_BIN_PATH, elem))

        logging.info('Build all : Done')

        logging.info('Create data folder')
        Arcadia.export(self.ctx[OcrArcadiaDataParameter.name], self.abs_path(self.OCR_ALL_DATA_PATH), revision=dataSvnInfo['entry_revision'])

        path_to_subdata = path.join(self.OCR_ALL_DATA_PATH, self.ctx[OcrDataSubpathParameter.name])
        if path.isfile(self.abs_path(path.join(path_to_subdata, self.OCR_SCRIPT_TO_CREATE_DATA))):
            run_process(['sh', self.OCR_SCRIPT_TO_CREATE_DATA], shell=True, work_dir=self.abs_path(path_to_subdata))
        shutil.copytree(path_to_subdata, self.OCR_RESULT_DATA_PATH)
        logging.info('Create data folder : Done')

        logging.info('Create resource')
        resource_result = channel.sandbox.get_resource(self.ctx['out_ocr_result_id'])
        with closing(tarfile.open(resource_result.path, 'w:gz', dereference=True)) as tar_file:
            tar_file.add(self.abs_path(self.OCR_RESULT_PATH), arcname="./")
        self.mark_resource_ready(resource_result)
        logging.info('Create resource : Done')

    def build_all(self, arcadiaPath, targetPaths, resultPath):
        build_system = consts.YMAKE_BUILD_SYSTEM
        build_type = consts.RELEASE_BUILD_TYPE
        def_flags = ''
        results_dir = self.abs_path(resultPath)
        sdk.do_build(
            build_system, arcadiaPath, targetPaths, build_type, clear_build=True,
            results_dir=results_dir, def_flags=def_flags
        )

    def check_all_paths(self):
        url = Arcadia.append(self.ctx.get(OcrArcadiaParameter.name), self.OCR_BIN_PATH)
        if not Arcadia.check(url):
            raise SandboxTaskFailureError('Arcadia URL {0} does not exist'.format(url))

        if (self.ctx[OcrArcadiaReleaseParameter.name] != ''):
            url = Arcadia.append(self.ctx.get(OcrArcadiaParameter.name), path.dirname(self.ctx.get(OcrArcadiaReleaseParameter.name)))
            if not Arcadia.check(url):
                raise SandboxTaskFailureError('Arcadia URL {0} does not exist'.format(url))

        svn_data = self.ctx.get(OcrArcadiaDataParameter.name)
        if not Arcadia.check(svn_data):
            raise SandboxTaskFailureError('Arcadia URL {0} does not exist'.format(svn_data))

        url = Arcadia.append(svn_data, self.ctx.get(OcrDataSubpathParameter.name))
        if not Arcadia.check(url):
            raise SandboxTaskFailureError('Arcadia URL {0} does not exist'.format(url))

    def on_enqueue(self):
        ArcadiaTask.on_enqueue(self)
        filename = 'resultOCR.tar.gz'
        resource_path = self.abs_path(filename)
        resource_name = '%s (%s)' % (self.descr, filename)
        self.ctx['out_ocr_result_id'] = self._create_resource(
            resource_name,
            resource_path,
            resource_types.OCR_RUNNER_ARCHIVE).id


__Task__ = BuildOcr
