# coding: utf-8
import json
import shutil
import os
import logging
import tarfile
import time

import sandbox.common.types.client as ctc

from sandbox.projects import resource_types

from sandbox.sandboxsdk import svn
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.paths import make_folder, list_dir

from sandbox.projects.common.utils import svn_revision
from sandbox.projects.common.utils import get_bsconfig
from sandbox.projects.common.build import ArcadiaTask as bbt
import sandbox.projects.common.build.parameters as build_params
import sandbox.projects.common.constants as consts
from sandbox.sandboxsdk.svn import ArcadiaTestData


class BuildOntodbAPI(bbt.ArcadiaTask):
    """
        ontodb api virtualenv building
    """
    type = 'BUILD_ONTODB_API'
    client_tags = ctc.Tag.LINUX_PRECISE

    input_parameters = build_params.get_arcadia_params()

    TGZ_PATH = 'ontodb_api.tar.gz'
    PYPI_URL = 'https://pypi.yandex-team.ru/simple/'

    def get_environment_without_pythonpath(self):
        environment = os.environ.copy()

        del environment['PYTHONPATH']

        return environment

    def _build_tar_gz(self, dir_path, tar_file_name):
        with tarfile.open(tar_file_name, 'w:gz') as tar:
            for entry in os.listdir(dir_path):
                tar.add(os.path.join(dir_path, entry), entry)

    def _get_revision_number(self):
        revision, tag, branch = self.arcadia_info()

        return revision

    def _build_shard_name(self):
        revision = self._get_revision_number()

        return 'ontodb_runtime-{0}-{1}'.format(revision, time.strftime("%Y%m%d_%H%M%S"))

    def _prepare_shard_dir(self, virtualenv_tgz_file, shard_name):
        shard_path = self.path(shard_name)

        make_folder(shard_path, delete_content=True)

        shutil.copy(virtualenv_tgz_file, shard_name)

        # pre initialize shard description file with install procedure
        with open(os.path.join(shard_name, 'shard.conf'), 'w') as f:
            f.write(
                '%install\n'
                'tar -xzf {}\n'.format(virtualenv_tgz_file)
            )

        return shard_path

    def _create_resource_and_shard(self, virtualenv_path):
        logging.info('creating tgz file')
        self._build_tar_gz(virtualenv_path, self.TGZ_PATH)

        shard_name = self._build_shard_name()

        self.ctx['shard_name'] = shard_name

        shard_path = self._prepare_shard_dir(self.TGZ_PATH, shard_name)

        self.create_resource(
            description='{0} shard'.format(shard_name),
            resource_path=shard_path,
            resource_type=resource_types.ONTODB_RUNTIME,
            arch='linux'
        )

        self._bsconfig_shard_init(shard_name)

    def _bsconfig_shard_init(self, shard_name):
        # do initialize shard
        run_process(
            ['perl', get_bsconfig(), 'shard_init', '--torrent', shard_name],
            log_prefix="bs_config",
            work_dir=self.path()
        )

    def _get_pip_path(self, virtualenv_path):
        return os.path.join(virtualenv_path, 'bin', 'pip')

    def _update_pip(self, virtualenv_path):
        """
        Current versions of `yandex-yt` are packaged as "wheels", so we need to update `pip` to latest version.
        """
        pip_path = self._get_pip_path(virtualenv_path)

        run_process([pip_path, 'install',
                     '--upgrade',
                     '-i', BuildOntodbAPI.PYPI_URL,
                     'pip'],
                    log_prefix='ontodb_runtime_pip',
                    environment=self.get_environment_without_pythonpath())

    def _install_requirements(self, virtualenv_path, arcadia_src_dir):
        pip_path = self._get_pip_path(virtualenv_path)

        requirements_file_name = os.path.join(
            arcadia_src_dir,
            'dict', 'ontodb', 'tools', 'api_service', 'requirements.txt'
        )

        run_process([pip_path, 'install',
                     '-i', BuildOntodbAPI.PYPI_URL,
                     '-r', requirements_file_name,
                     '--upgrade',
                     '--force-reinstall'],
                    log_prefix='ontodb_runtime_pip',
                    environment=self.get_environment_without_pythonpath())

    def _install_main_package(self, virtualenv_path, arcadia_src_dir):
        runtime_package_project = os.path.join(arcadia_src_dir, 'dict', 'ontodb', 'tools', 'runtime_package')

        pip_path = self._get_pip_path(virtualenv_path)

        run_process([pip_path, 'install',
                     '-i', 'http://pypi.yandex-team.ru/simple/',
                     runtime_package_project],
                    log_prefix='ontodb_runtime_pip',
                    environment=self.get_environment_without_pythonpath())

    def _make_virtualenv_relocatable(self, virtualenv_path):
        run_process([
            '/skynet/python/bin/virtualenv',
            '--relocatable',
            virtualenv_path
        ])

    def _copy_ontodb_sources(self, virtualenv_path, arcadia_src_dir):
        ontodb_code_src_dir = os.path.join(arcadia_src_dir, 'dict/ontodb')
        ontodb_code_dst_dir = os.path.join(virtualenv_path, 'ontodb_src')

        shutil.copytree(ontodb_code_src_dir, ontodb_code_dst_dir)

    def _prepare_arcadia_src(self):
        arcadia_src_dir = self.get_arcadia_src_dir()
        svn.Arcadia.apply_patch(arcadia_src_dir, self.ctx.get('arcadia_patch'), self.abs_path())

        self.ctx['arcadia_revision'] = svn_revision(arcadia_src_dir)

        return arcadia_src_dir

    def _initialize_virtualenv(self):
        virtualenv_path = self.path('ontodb_runtime_build')

        run_process([
            '/skynet/python/bin/virtualenv',
            '--python', '/usr/bin/python2.7',
            '--system-site-packages',
            virtualenv_path
        ])

        return virtualenv_path

    def _write_local_ymake(self, arcadia_src_dir):
        with open(os.path.join(arcadia_src_dir, 'local.ymake'), 'w') as local_ymake:
            local_ymake.write('USE_ARCADIA_PYTHON=no\n')
            local_ymake.write('PYTHON_INCLUDE_PATH="/usr/include/python2.7"\n')
            local_ymake.write('PYTHON_LIBRARY="python2.7"\n')

            local_ymake.write('WERROR_MODE=none\n')

    def _run_ya_build(self, arcadia_src_dir, build_dir, targets, threads=0):
        ya_cmd = [
            os.path.join(arcadia_src_dir, 'ya'),
            'build',
            '-r',
            '--build-dir=' + build_dir,
            '--target=' + ';'.join(targets)
        ]

        if threads > 0:
            ya_cmd.append('--threads=' + str(threads))

        run_process(
            ya_cmd,
            log_prefix='ya_build',
            work_dir=arcadia_src_dir
        )

    def get_arc_test_data_root_url(self):
        arcadia_src_url = self.ctx[consts.ARCADIA_URL_KEY]
        branch_root = arcadia_src_url.rpartition('/arcadia')[0]

        return branch_root + '/arcadia_tests_data/'

    def get_arc_test_data_dir(self):
        arcadia_tests_data_rootpath, _, _, _ = ArcadiaTestData.test_data_location(self.get_arc_test_data_root_url())

        return arcadia_tests_data_rootpath

    def _build_ontodb_binaries(self, arcadia_src_dir, arc_test_data_dir):
        self._write_local_ymake(arcadia_src_dir)

        build_dir = self.path('ontodb_binaries_build')
        arcadia_fake_root = self.path('arcadia_fake_root')

        make_folder(build_dir)
        make_folder(arcadia_fake_root)

        os.symlink(
            arcadia_src_dir,
            os.path.join(arcadia_fake_root, 'arcadia')
        )

        os.symlink(
            arc_test_data_dir,
            os.path.join(arcadia_fake_root, 'arcadia_tests_data')
        )

        cpu_count = self.client_info['ncpu']

        rebuild_sh_env = os.environ.copy()
        rebuild_sh_env.update({
            'BUILD_PROCESSES_COUNT': str(cpu_count),
            'YA_BUILD_DIR': build_dir,
            'SKIP_YA_TEST': '1',
            'ARCADIA_TOP_ROOT': arcadia_fake_root
        })

        run_process(
            [
                'bash', os.path.join(arcadia_src_dir, 'dict', 'ontodb', '_rebuild.sh')
            ],
            log_prefix='ontodb_rebuild_sh',
            work_dir=os.path.join(arcadia_src_dir, 'dict', 'ontodb'),
            environment=rebuild_sh_env
        )

        return build_dir

    def copy_files(self, from_dir, to_dir, file_path_filter=None):
        files_to_copy = filter(
            file_path_filter,
            list_dir(from_dir, files_only=True, abs_path=True)
        )

        for file_name in files_to_copy:
            shutil.copy(file_name, to_dir)

    def _copy_binaries_to_virtualenv(self, build_dir, virtualenv_path, arcadia_src_dir, arc_test_data_dir):
        self.copy_files(
            os.path.join(build_dir, 'latest', 'bin'),
            os.path.join(virtualenv_path, 'bin'),
            lambda file_name: os.path.basename(file_name) != 'python'
        )

        self.copy_files(
            os.path.join(build_dir, 'latest', 'lib'),
            os.path.join(virtualenv_path, 'lib', 'python2.7', 'site-packages')
        )

        self.copy_files(
            os.path.join(arcadia_src_dir, 'dict', 'ontodb', 'wikidate'),
            os.path.join(virtualenv_path, 'ontodb_src', 'wikidate'),
            lambda file_name: file_name.endswith('.proto') or file_name.endswith('.bin')
        )

    def _prepare_arcadia_test_data(self):
        arcadia_tests_data_root = self.get_arc_test_data_root_url()

        required_test_data_dirs = [
            'wizard_source_data/date',
            'wizard/date',
            'ontodb'
        ]

        for test_data_dir in required_test_data_dirs:
            ArcadiaTestData.get_arcadia_test_data(
                self,
                "{root_url}{dir}@{revision}".format(
                    root_url=arcadia_tests_data_root,
                    dir=test_data_dir,
                    revision=self.ctx['arcadia_revision']
                )
            )

        return self.get_arc_test_data_dir()

    def _write_version(self, virtualenv_path):
        version_file_path = os.path.join(virtualenv_path, 'version.json')

        with open(version_file_path, 'w') as version_file:
            json.dump(
                {
                    'svn_revision': self._get_revision_number()
                },
                version_file
            )

    def _add_lxml_link(self, virtualenv_path):
        os.symlink(
            '/usr/lib/python2.7/dist-packages/lxml',
            os.path.join(virtualenv_path, 'lib', 'python2.7', 'site-packages', 'lxml')
        )

    def do_execute(self):
        arcadia_src_dir = self._prepare_arcadia_src()
        arc_test_data_dir = self._prepare_arcadia_test_data()

        virtualenv_path = self._initialize_virtualenv()

        self._update_pip(virtualenv_path)
        self._install_requirements(virtualenv_path, arcadia_src_dir)
        self._add_lxml_link(virtualenv_path)
        self._install_main_package(virtualenv_path, arcadia_src_dir)

        self._make_virtualenv_relocatable(virtualenv_path)

        self._copy_ontodb_sources(virtualenv_path, arcadia_src_dir)

        build_dir = self._build_ontodb_binaries(arcadia_src_dir, arc_test_data_dir)

        self._copy_binaries_to_virtualenv(build_dir, virtualenv_path, arcadia_src_dir, arc_test_data_dir)

        self._write_version(virtualenv_path)

        self._create_resource_and_shard(virtualenv_path)


__Task__ = BuildOntodbAPI
