# coding: utf-8
import json
import shutil
import os
import logging
import tarfile
import time

import sandbox.common.types.client as ctc

from sandbox.projects import resource_types

from sandbox.sandboxsdk import svn
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.paths import make_folder

from sandbox.projects.common import apihelpers
from sandbox.projects.common.utils import svn_revision
import sandbox.projects.common.constants as consts
from sandbox.projects.common.build import ArcadiaTask as bbt
import sandbox.projects.common.build.parameters as build_params
from sandbox.sandboxsdk.environments import VirtualEnvironment
from sandbox.common.share import skynet_get


class BuildOntodbClarinet(bbt.ArcadiaTask):
    """
        ontodb-clarinet virtualenv building
    """
    type = 'BUILD_ONTODB_CLARINET'
    client_tags = ctc.Tag.LINUX_PRECISE

    privileged = False
    input_parameters = build_params.get_arcadia_params()

    TGZ_PATH = 'ontodb_clarinet.tar.gz'
    PYPI_URL = 'https://pypi.yandex-team.ru/simple/'

    def _build_tar_gz(self, dir_path, tar_file_name):
        with tarfile.open(tar_file_name, 'w:gz') as tar:
            for entry in os.listdir(dir_path):
                tar.add(os.path.join(dir_path, entry), entry)

    def _get_revision_number(self):
        revision, tag, branch = self.arcadia_info()

        return revision

    def _build_shard_name(self):
        revision = self._get_revision_number()

        return 'ontodb_clarinet-{0}-{1}'.format(revision, time.strftime("%Y%m%d_%H%M%S"))

    def _prepare_shard_dir(self, virtualenv_tgz_file, shard_name):
        shard_path = self.path(shard_name)

        make_folder(shard_path, delete_content=True)

        shutil.copy(virtualenv_tgz_file, shard_name)

        # pre initialize shard description file with install procedure
        with open(os.path.join(shard_name, 'shard.conf'), 'w') as f:
            f.write(
                '%install\n'
                'tar -xzf {}'.format(virtualenv_tgz_file)
            )

        return shard_path

    def _create_and_save_resource(self, virtualenv_path):
        logging.info('creating tgz file')
        self._build_tar_gz(virtualenv_path, self.TGZ_PATH)

        self.create_resource(
            description='{0} shard'.format(self.TGZ_PATH),
            resource_path=self.TGZ_PATH,
            resource_type=resource_types.ONTODB_CLARINET,
            arch='linux'
        )

    def get_venv_resource_path(self, virtualenv_path):
        return os.path.join(virtualenv_path, "venv")

    def _get_pip_path(self, virtualenv_path):
        return os.path.join(virtualenv_path, 'bin', 'pip')

    def get_opus_libs_dir(self):
        attrs = {"contents": "opuslib"}
        resource = apihelpers.get_last_resource_with_attrs(resource_types.OTHER_RESOURCE, attrs=attrs)
        if resource:
            return self.sync_resource(resource.id)

    def install_opus_libs(self, target_dir):
        resource_dir = self.get_opus_libs_dir()
        if not resource_dir:
            logging.error("No libopus resource found")
            return

        bin_dir = "bin"
        lib_dir = "lib"

        for path in [lib_dir, bin_dir]:
            source_dir = os.path.join(resource_dir, path)
            new_dir = os.path.join(target_dir, path)
            logging.debug("Scanning %s" % source_dir)
            for filename in os.listdir(source_dir):
                file_path = os.path.join(source_dir, filename)
                if os.path.isfile(file_path):
                    logging.debug("Copying %s -> %s" % (file_path, new_dir))
                    shutil.copy(file_path, new_dir)

    def _update_pip(self, virtualenv_path):
        """
        Current versions of `yandex-yt` are packaged as "wheels", so we need to update `pip` to latest version.
        """
        pip_path = self._get_pip_path(virtualenv_path)

        run_process([pip_path, 'install',
                     '--upgrade',
                     '-i', BuildOntodbClarinet.PYPI_URL,
                     'pip'],
                    log_prefix='ontodb_clarinet_pip')

    def _install_requirements(self, virtualenv_path, project_dir):
        pip_path = self._get_pip_path(virtualenv_path)

        requirements_file_name = os.path.join(
            project_dir, 'requirements.txt'
        )

        # do not install greenlet from broken wheel: workaround with --no-binary
        run_process([pip_path, 'install',
                     '-i', BuildOntodbClarinet.PYPI_URL,
                     '-r', requirements_file_name,
                     '--upgrade',
                     '--force-reinstall',
                     '--no-binary=greenlet'
                     ],
                    log_prefix='ontodb_clarinet_pip')

        bin_path = os.path.join(virtualenv_path, 'bin')
        # https://sandbox.yandex-team.ru/resource/625307480/view
        skynet_get('rbtorrent:311c6f549992fd726e9da2f272dc8def171834af', bin_path)
        # crca_path = os.path.join(virtualenv_path, 'bin', 'crca')

    def _make_virtualenv_relocatable(self, virtualenv_path):
        run_process([
            '/skynet/python/bin/virtualenv',
            '--relocatable',
            virtualenv_path
        ],
            log_prefix="relocating_venv"
        )

    def checkout_project_dir(self):
        arcadia_src_dir = svn.Arcadia.get_arcadia_src_dir(self.get_project_source_url())
        svn.Arcadia.apply_patch(arcadia_src_dir, self.ctx.get('arcadia_patch'), self.abs_path())
        self.ctx['arcadia_revision'] = svn_revision(arcadia_src_dir)

        return arcadia_src_dir

    def get_project_path(self):
        return os.path.join("dict", "clarabot")

    def get_project_source_url(self):
        return svn.Arcadia.append(self.ctx[consts.ARCADIA_URL_KEY], self.get_project_path())

    def _initialize_virtualenv(self):
        virtualenv_path = self.path('ontodb_clarinet_build')

        run_process([
            '/skynet/python/bin/virtualenv',
            '--python', '/usr/bin/python2.7',
            '--system-site-packages',
            virtualenv_path
        ],
            log_prefix="initializing_venv"
        )
        self._virtualenv_path = virtualenv_path

    def _write_version(self, virtualenv_path):
        version_file_path = os.path.join(virtualenv_path, 'version.json')

        with open(version_file_path, 'w') as version_file:
            json.dump(
                {
                    'svn_revision': self._get_revision_number()
                },
                version_file
            )

    def _install_main_package(self, virtualenv_path, project_dir):
        clarinet_package = project_dir

        pip_path = self._get_pip_path(virtualenv_path)

        run_process([pip_path, 'install',
                     '-i', 'http://pypi.yandex-team.ru/simple/',
                     clarinet_package],
                    log_prefix='ontodb_clarinet_pip')

    def _copy_production_config(self, virtualenv_path):
        config_dir = os.path.join(
            virtualenv_path,
            'lib',
            'python2.7',
            'site-packages',
            'clarinet_bot',
            'config'
        )

        production_config_dist_file_name = os.path.join(
            config_dir,
            'production.py'
        )

        current_config_file_name = os.path.join(
            config_dir,
            'current.py'
        )

        shutil.copy(
            production_config_dist_file_name,
            current_config_file_name
        )

    def _copy_production_ya_config(self, virtualenv_path):
        config_dir = os.path.join(
            virtualenv_path,
            'lib',
            'python2.7',
            'site-packages',
            'yandex_bot',
            'config'
        )

        production_config_dist_file_name = os.path.join(
            config_dir,
            'production.py'
        )

        current_config_file_name = os.path.join(
            config_dir,
            'current.py'
        )

        shutil.copy(
            production_config_dist_file_name,
            current_config_file_name
        )

    def do_execute(self):
        self.project_checkout_dir = self.checkout_project_dir()
        self._virtualenv_path = self.path('ontodb_clarinet_build')
        venv_dir = self.get_venv_resource_path(self._virtualenv_path)

        with VirtualEnvironment(self._virtualenv_path, do_not_remove=True) as venv:
            self._update_pip(venv_dir)
            self._install_requirements(venv_dir, self.project_checkout_dir)
            self._install_main_package(venv_dir, self.project_checkout_dir)
            self.install_opus_libs(venv_dir)
            self._copy_production_config(venv_dir)
            self._copy_production_ya_config(venv_dir)
            venv.make_relocatable()
            self._write_version(venv_dir)

        self._create_and_save_resource(venv_dir)


__Task__ = BuildOntodbClarinet
