# coding: utf-8
import json
import shutil
import os
import logging
import tarfile
import time

import sandbox.common.types.client as ctc

from sandbox.projects import resource_types

from sandbox.sandboxsdk import svn
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.paths import make_folder, list_dir

from sandbox.projects.common.utils import svn_revision
from sandbox.projects.common.build import ArcadiaTask as bbt
import sandbox.projects.common.build.parameters as build_params
from sandbox.sandboxsdk.environments import VirtualEnvironment


class BuildOntodbFixesViewer(bbt.ArcadiaTask):
    """
        ontodbfixes virtualenv building
    """
    type = 'BUILD_ONTODB_FIXES_VIEWER'
    client_tags = ctc.Tag.LINUX_PRECISE

    input_parameters = build_params.get_arcadia_params()

    TGZ_PATH = 'ontodbfixes.tar.gz'
    PYPI_URL = 'https://pypi.yandex-team.ru/simple/'

    def get_environment_without_pythonpath(self):
        environment = os.environ.copy()

        del environment['PYTHONPATH']

        return environment

    def _build_tar_gz(self, dir_path, tar_file_name):
        with tarfile.open(tar_file_name, 'w:gz') as tar:
            for entry in os.listdir(dir_path):
                tar.add(os.path.join(dir_path, entry), entry)

    def _get_revision_number(self):
        revision, tag, branch = self.arcadia_info()

        return revision

    def _build_shard_name(self):
        revision = self._get_revision_number()

        return 'ontodbfixes-{0}-{1}'.format(revision, time.strftime("%Y%m%d_%H%M%S"))

    def _prepare_shard_dir(self, virtualenv_tgz_file, shard_name):
        shard_path = self.path(shard_name)

        make_folder(shard_path, delete_content=True)

        shutil.copy(virtualenv_tgz_file, shard_name)

        # pre initialize shard description file with install procedure
        with open(os.path.join(shard_name, 'shard.conf'), 'w') as f:
            f.write(
                '%install\n'
                'tar -xzf {}\n'.format(virtualenv_tgz_file)
            )

        return shard_path

    def _create_and_save_resource(self, virtualenv_path):
        logging.info('creating tgz file')
        self._build_tar_gz(virtualenv_path, self.TGZ_PATH)

        self.create_resource(
            description='{0} pack'.format(self.TGZ_PATH),
            resource_path=self.TGZ_PATH,
            resource_type=resource_types.ONTODB_FIXES_VIEWER,
            arch='linux'
        )

    def _get_pip_path(self, virtualenv_path):
        return os.path.join(virtualenv_path, 'bin', 'pip')

    def _update_pip(self, virtualenv_path):
        """
        Current versions of `yandex-yt` are packaged as "wheels", so we need to update `pip` to latest version.
        """
        pip_path = self._get_pip_path(virtualenv_path)

        run_process([pip_path, 'install',
                     '--upgrade',
                     '-i', BuildOntodbFixesViewer.PYPI_URL,
                     'pip'],
                    log_prefix='ontodbfixes_pip',
                    environment=self.get_environment_without_pythonpath())

    def _install_requirements(self, virtualenv_path, arcadia_src_dir):
        pip_path = self._get_pip_path(virtualenv_path)

        requirements_file_name = os.path.join(
            arcadia_src_dir,
            'dict', 'ontodb', 'requirements.txt'
        )

        run_process([pip_path, 'install',
                     '-i', BuildOntodbFixesViewer.PYPI_URL,
                     '-r', requirements_file_name,
                     '--upgrade',
                     '--force-reinstall'],
                    log_prefix='ontodbfixes_pip',
                    environment=self.get_environment_without_pythonpath())

    def _prepare_arcadia_src(self):
        arcadia_src_dir = self.get_arcadia_src_dir()
        svn.Arcadia.apply_patch(arcadia_src_dir, self.ctx.get('arcadia_patch'), self.abs_path())

        self.ctx['arcadia_revision'] = svn_revision(arcadia_src_dir)

        return arcadia_src_dir

    def get_venv_resource_path(self, virtualenv_path):
        return os.path.join(virtualenv_path, "venv")

    def copy_files(self, from_dir, to_dir, file_path_filter=None):
        files_to_copy = filter(
            file_path_filter,
            list_dir(from_dir, files_only=True, abs_path=True)
        )

        for file_name in files_to_copy:
            shutil.copy(file_name, to_dir)

    def _write_version(self, virtualenv_path):
        version_file_path = os.path.join(virtualenv_path, 'version.json')

        with open(version_file_path, 'w') as version_file:
            json.dump(
                {
                    'svn_revision': self._get_revision_number()
                },
                version_file
            )

    @staticmethod
    def _copy_viewer_files(virtualenv_path, arcadia_src_dir):
        ontodb_dir = os.path.join(arcadia_src_dir, "dict", "ontodb")

        code_src_dir = os.path.join(ontodb_dir, 'tools', 'ontodbfixes')
        code_dst_dir = os.path.join(virtualenv_path, 'ontodbfixes')

        shutil.copytree(code_src_dir, code_dst_dir)

        ontodb_extra_dirs = ("report", "cfg", "utils", "onto_lib", "isa", "config")

        for dir_name in ontodb_extra_dirs:
            source_dir = os.path.join(ontodb_dir, dir_name)
            destination_dir = os.path.join(virtualenv_path, dir_name)
            shutil.copytree(source_dir, destination_dir)

    def do_execute(self):
        arcadia_src_dir = self._prepare_arcadia_src()

        virtualenv_path = self.path('viewer')
        venv_dir = self.get_venv_resource_path(virtualenv_path)

        with VirtualEnvironment(virtualenv_path, do_not_remove=True) as venv:
            self._update_pip(venv_dir)
            self._install_requirements(venv_dir, arcadia_src_dir)
            venv.make_relocatable()
            self._copy_viewer_files(venv_dir, arcadia_src_dir)
            self._write_version(venv_dir)
            self._create_and_save_resource(venv_dir)


__Task__ = BuildOntodbFixesViewer
