# coding: utf-8
import json
import shutil
import os
import logging
import tarfile
import time

import sandbox.common.types.client as ctc

from sandbox.projects import resource_types

from sandbox.sandboxsdk import svn
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.paths import make_folder, list_dir, copy_path
from sandbox.sandboxsdk.channel import channel

from sandbox.projects.common.arcadia import sdk
from sandbox.projects.common.utils import svn_revision
from sandbox.projects.common.build import ArcadiaTask as bbt
import sandbox.projects.common.build.parameters as build_params
import sandbox.projects.common.constants as consts
from sandbox.sandboxsdk.svn import ArcadiaTestData


class BuildOntodbRuntime(bbt.ArcadiaTask):
    """
        ontodb runtime virtualenv building
    """
    type = 'BUILD_ONTODB_RUNTIME'
    client_tags = ctc.Tag.LINUX_PRECISE

    input_parameters = build_params.get_arcadia_params()

    TGZ_PATH = 'ontodb_runtime.tar.gz'
    PYPI_URL = 'https://pypi.yandex-team.ru/simple/'
    ONTO_CODE_SUBPATH = 'arcadia/dict/ontodb_src'
    GEOTHES_CODE_SUBPATH = 'arcadia/FactExtract/geothes'

    def get_environment_without_pythonpath(self):
        environment = os.environ.copy()

        del environment['PYTHONPATH']

        return environment

    def _build_tar_gz(self, dir_path, tar_file_name):
        with tarfile.open(tar_file_name, 'w:gz') as tar:
            for entry in os.listdir(dir_path):
                tar.add(os.path.join(dir_path, entry), entry)

    def _get_revision_number(self):
        revision, tag, branch = self.arcadia_info()

        return revision

    def _build_shard_name(self):
        revision = self._get_revision_number()

        return 'ontodb_runtime-{0}-{1}'.format(revision, time.strftime("%Y%m%d_%H%M%S"))

    def _prepare_shard_dir(self, virtualenv_tgz_file, shard_name):
        shard_path = self.path(shard_name)

        make_folder(shard_path, delete_content=True)

        shutil.copy(virtualenv_tgz_file, shard_name)

        # pre initialize shard description file with install procedure
        with open(os.path.join(shard_name, 'shard.conf'), 'w') as f:
            f.write(
                '%install\n'
                'tar -xzf {}\n'.format(virtualenv_tgz_file)
            )

        return shard_path

    def _create_and_save_resource(self, virtualenv_path):
        logging.info('creating tgz file')
        self._build_tar_gz(virtualenv_path, self.TGZ_PATH)

        self.create_resource(
            description='{0} pack'.format(self.TGZ_PATH),
            resource_path=self.TGZ_PATH,
            resource_type=resource_types.ONTODB_RUNTIME,
            arch='linux'
        )

    def _get_pip_path(self, virtualenv_path):
        return os.path.join(virtualenv_path, 'bin', 'pip')

    def _update_pip(self, virtualenv_path):
        """
        Current versions of `yandex-yt` are packaged as "wheels", so we need to update `pip` to latest version.
        """
        pip_path = self._get_pip_path(virtualenv_path)

        run_process([pip_path, 'install',
                     '--upgrade',
                     '-i', BuildOntodbRuntime.PYPI_URL,
                     'pip'],
                    log_prefix='ontodb_runtime_pip',
                    environment=self.get_environment_without_pythonpath())

    def _install_requirements(self, virtualenv_path, arcadia_src_dir):
        pip_path = self._get_pip_path(virtualenv_path)

        requirements_file_name = os.path.join(
            arcadia_src_dir,
            'dict', 'ontodb', 'requirements_runtime.txt'
        )

        run_process([pip_path, 'install',
                     '-i', BuildOntodbRuntime.PYPI_URL,
                     '-r', requirements_file_name,
                     '--upgrade',
                     '--force-reinstall'],
                    log_prefix='ontodb_runtime_pip',
                    environment=self.get_environment_without_pythonpath())

    def _install_main_package(self, virtualenv_path, arcadia_src_dir):
        runtime_package_project = os.path.join(arcadia_src_dir, 'dict', 'ontodb', 'tools', 'runtime_package')

        pip_path = self._get_pip_path(virtualenv_path)

        run_process([pip_path, 'install',
                     '-i', 'https://pypi.yandex-team.ru/simple/',
                     runtime_package_project],
                    log_prefix='ontodb_runtime_pip',
                    environment=self.get_environment_without_pythonpath())

    def _make_virtualenv_relocatable(self, virtualenv_path):
        run_process([
            '/skynet/python/bin/virtualenv',
            '--relocatable',
            virtualenv_path
        ])

    def _copy_parse_lib(self, ontodb_code_dst_dir, arcadia_src_dir):
        parse_lib_src_file = os.path.join(arcadia_src_dir, 'quality/logs/parse_lib_py/parselib.py')

        shutil.copy2(parse_lib_src_file, ontodb_code_dst_dir)

    def _copy_lang_pack(self, ontodb_code_dst_dir, arc_test_data_src_dir):
        lang_pack_file = os.path.join(arc_test_data_src_dir, 'ontodb/lang_pack.tar.gz')

        shutil.copy(lang_pack_file, ontodb_code_dst_dir)

    def _copy_ontodb_sources(self, virtualenv_path, arcadia_src_dir, arc_test_data_src_dir):
        ontodb_code_src_dir = os.path.join(arcadia_src_dir, 'dict/ontodb')
        geothes_code_src_dir = os.path.join(arcadia_src_dir, 'FactExtract/geothes')
        ontodb_code_dst_dir = os.path.join(virtualenv_path, self.ONTO_CODE_SUBPATH)
        geothes_code_dst_dir = os.path.join(virtualenv_path, self.GEOTHES_CODE_SUBPATH)

        shutil.copytree(ontodb_code_src_dir, ontodb_code_dst_dir)
        shutil.copytree(geothes_code_src_dir, geothes_code_dst_dir)

        self._copy_parse_lib(ontodb_code_dst_dir, arcadia_src_dir)

        self._copy_lang_pack(ontodb_code_dst_dir, arc_test_data_src_dir)

    def _prepare_arcadia_src(self):
        arcadia_src_dir = self.get_arcadia_src_dir()
        svn.Arcadia.apply_patch(arcadia_src_dir, self.ctx.get('arcadia_patch'), self.abs_path())

        self.ctx['arcadia_revision'] = svn_revision(arcadia_src_dir)

        return arcadia_src_dir

    def _initialize_virtualenv(self):
        virtualenv_path = self.path('ontodb_runtime_build')

        run_process([
            '/skynet/python/bin/virtualenv',
            '--python', '/skynet/python/bin/python2',
            '--system-site-packages',
            virtualenv_path
        ])

        return virtualenv_path

    def get_arc_test_data_root_url(self):
        arcadia_src_url = self.ctx[consts.ARCADIA_URL_KEY]
        branch_root = arcadia_src_url.rpartition('/arcadia')[0]

        return branch_root + '/arcadia_tests_data/'

    def get_arc_test_data_dir(self):
        arcadia_tests_data_rootpath, _, _, _ = ArcadiaTestData.test_data_location(self.get_arc_test_data_root_url())

        return arcadia_tests_data_rootpath

    def _build_ontodb_binaries(self, arcadia_src_dir, arc_test_data_dir):
        build_dir = self.path('ontodb_binaries_build')
        arcadia_fake_root = self.path('arcadia_fake_root')

        arcadia_fake_src_dir = os.path.join(arcadia_fake_root, 'arcadia')

        make_folder(build_dir)
        make_folder(arcadia_fake_root)

        os.symlink(
            arcadia_src_dir,
            arcadia_fake_src_dir
        )

        os.symlink(
            arc_test_data_dir,
            os.path.join(arcadia_fake_root, 'arcadia_tests_data')
        )

        def_flags = {
            'PYTHON_INCLUDE_DIRS': '/skynet/python/include/python2.7',
            'PYTHON_LIBRARY': 'python2.7',
            'WERROR_MODE': 'none'
        }

        build_type = consts.RELEASE_BUILD_TYPE

        targets = ["dict/ontodb"]
        build_dir = arcadia_fake_src_dir  # to make _prepare_env.sh work  TBD: create bugfix ticket at OBJECTS
        sdk.do_build(
            consts.YMAKE_BUILD_SYSTEM, arcadia_fake_src_dir, targets, build_type, clear_build=True,
            results_dir=build_dir, def_flags=def_flags,
        )

        prepare_env_sh_env = os.environ.copy()
        prepare_env_sh_env.update({
            'ARCADIA_TOP_ROOT': arcadia_fake_root
        })

        # self.suspend()

        run_process(
            [
                'bash', os.path.join(arcadia_src_dir, 'dict', 'ontodb', '_prepare_env.sh')
            ],
            log_prefix='ontodb_prepare_env',
            work_dir=os.path.join(arcadia_src_dir, 'dict', 'ontodb'),
            environment=prepare_env_sh_env
        )

        return build_dir

    def copy_files(self, from_dir, to_dir, file_path_filter=None):
        files_to_copy = filter(
            file_path_filter,
            list_dir(from_dir, files_only=True, abs_path=True)
        )

        for file_name in files_to_copy:
            shutil.copy(file_name, to_dir)

    def _is_library(self, file_name):
        return '.so' in file_name

    def _copy_binaries_to_virtualenv(self, build_dir, virtualenv_path, arcadia_src_dir, arc_test_data_dir):
        self.copy_files(
            os.path.join(build_dir, 'bin'),
            os.path.join(virtualenv_path, 'bin'),
            lambda file_name: not self._is_library(file_name) and os.path.basename(file_name) != 'python'
        )

        self.copy_files(
            os.path.join(build_dir, 'lib'),
            os.path.join(virtualenv_path, 'lib', 'python2.7', 'site-packages'),
            lambda file_name: self._is_library(file_name)
        )

        if os.path.isfile(
            os.path.join(virtualenv_path, 'lib', 'python2.7', 'site-packages', 'libyandex_remorph_python.so.1')
        ):
            shutil.copyfile(
                os.path.join(virtualenv_path, 'lib', 'python2.7', 'site-packages', 'libyandex_remorph_python.so.1'),
                os.path.join(virtualenv_path, 'lib', 'python2.7', 'site-packages', 'libyandex_remorph_python.so')
            )

        self.copy_files(
            os.path.join(arcadia_src_dir, 'dict', 'ontodb', 'wikidate'),
            os.path.join(virtualenv_path, self.ONTO_CODE_SUBPATH, 'wikidate'),
            lambda file_name: file_name.endswith('.proto') or file_name.endswith('.bin')
        )
        # TODO: remove this block when zoracl from trunk becomes compatible with zora production server version
        resource = channel.sandbox.get_resource(103578656)
        src_path = self.sync_resource(resource.id)
        copy_path(src_path, os.path.join(virtualenv_path, 'bin', 'zoracl'))

    def _prepare_arcadia_test_data(self):
        arcadia_tests_data_root = self.get_arc_test_data_root_url()

        required_test_data_dirs = [
            'wizard_source_data/date',
            'wizard/date',
            'ontodb'
        ]

        for test_data_dir in required_test_data_dirs:
            ArcadiaTestData.get_arcadia_test_data(
                self,
                "{root_url}{dir}@{revision}".format(
                    root_url=arcadia_tests_data_root,
                    dir=test_data_dir,
                    revision=self.ctx['arcadia_revision']
                )
            )

        return self.get_arc_test_data_dir()

    def _write_version(self, virtualenv_path):
        version_file_path = os.path.join(virtualenv_path, 'version.json')

        with open(version_file_path, 'w') as version_file:
            json.dump(
                {
                    'svn_revision': self._get_revision_number()
                },
                version_file
            )

    def do_execute(self):
        arcadia_src_dir = self._prepare_arcadia_src()
        arc_test_data_dir = self._prepare_arcadia_test_data()

        virtualenv_path = self._initialize_virtualenv()

        self._update_pip(virtualenv_path)
        self._install_requirements(virtualenv_path, arcadia_src_dir)
        self._install_main_package(virtualenv_path, arcadia_src_dir)

        self._make_virtualenv_relocatable(virtualenv_path)

        self._copy_ontodb_sources(virtualenv_path, arcadia_src_dir, arc_test_data_dir)

        build_dir = self._build_ontodb_binaries(arcadia_src_dir, arc_test_data_dir)

        self._copy_binaries_to_virtualenv(build_dir, virtualenv_path, arcadia_src_dir, arc_test_data_dir)

        self._write_version(virtualenv_path)

        self._create_and_save_resource(virtualenv_path)


__Task__ = BuildOntodbRuntime
