import logging

from sandbox.projects.common.build.YaPackage import YaPackage, PackageTypeParameter, \
    ForceDuploadParameter, \
    PublishPackageParameter, \
    KeyUserParameter, \
    PublishToParameter, \
    MultiplePublishToParameter, \
    ResourceTypeParameter, \
    ChangelogParameter, \
    MultiplePublishParameter, \
    MultiplePublishMappingParameter, \
    PublishingBlock, \
    SloppyDebianParameter, \
    CompressPackageArchiveParameter

from sandbox.projects.porto.BuildPortoLayer import BuildPortoLayer, ScriptUrl
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.parameters import SandboxParameter, SandboxInfoParameter
from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask
from sandbox.projects import resource_types

from sandbox.projects.common.utils import get_or_default
from sandbox.projects.common.apihelpers import list_task_resources


class PortoLayerBlock(SandboxInfoParameter):
    description = 'Porto layer'


class BuildPackageAndPortoLayer(ArcadiaTask):
    """
        Build package with custom parameters and install it into porto layer
        without publishing to dist
    """

    type = 'BUILD_PACKAGE_AND_PORTO_LAYER'

    _STOP_YA_PACKAGE_PARAMETERS = [
        PackageTypeParameter,
        ResourceTypeParameter,
        PublishToParameter,
        MultiplePublishToParameter,
        CompressPackageArchiveParameter,
        PublishPackageParameter,
        ForceDuploadParameter,
        SloppyDebianParameter,
        KeyUserParameter,
        MultiplePublishParameter,
        PublishingBlock,
        ChangelogParameter,
        MultiplePublishMappingParameter,
    ]

    input_parameters = [
        p
        for p in YaPackage.input_parameters
        if p not in _STOP_YA_PACKAGE_PARAMETERS
    ] + [PortoLayerBlock] + [
        p
        for p in BuildPortoLayer.input_parameters
    ]

    _TASK_PACKAGE_KEY = 'task_ya_package_id'
    _TASK_PORTO_KEY = 'task_build_porto_layer_id'

    def __create_ya_package_task_ctx(self, ctx):
        ctx = ctx.copy()

        ctx[PackageTypeParameter.name] = 'debian'
        ctx[ResourceTypeParameter.name] = 'YA_PACKAGE'
        ctx[PublishPackageParameter.name] = False
        ctx[ForceDuploadParameter.name] = False
        ctx[CompressPackageArchiveParameter.name] = False

        return ctx

    def __create_build_porto_layer_task_ctx(self, ctx):
        ctx = ctx.copy()

        # Set proxy url to generated script
        ctx[ScriptUrl.name] += '\n' + list_task_resources(self.id, resource_types.OTHER_RESOURCE)[0].proxy_url

        return ctx

    def __create_packages_install_script(self):
        resources = list_task_resources(self.ctx[self._TASK_PACKAGE_KEY], resource_types.YA_PACKAGE)
        if not resources:
            raise SandboxTaskFailureError('Subtask %s has no %s resources' % (self.ctx[self._TASK_PACKAGE_KEY], str(resource_types.YA_PACKAGE)))

        install_script = ["""
            #!/bin/sh
            set -e

            apt-get update
            apt-get install -y dpkg
        """]

        for resource in resources:
            install_script.append("""
                mkdir __package__ && cd __package__
                wget --no-check-certificate {resource_link} -O target_package.deb.tar.gz
                tar -xvf target_package.deb.tar.gz
                ls -1 | grep -E ".deb$" | xargs dpkg --install
                cd - && rm -rf __package__
            """.format(resource_link=resource.proxy_url))

            logging.info('resoource_link: %s' % resource.proxy_url)

        with open('install_script.sh', 'w') as out:
            print >>out, ''.join(install_script)

        resource = self.create_resource('', 'install_script.sh', resource_types.OTHER_RESOURCE)
        self.mark_resource_ready(resource.id)

    def __copy_input_parameters(self, subtask):
        return {
            p.name: get_or_default(self.ctx, p)
            for p in self.input_parameters
            if issubclass(p, SandboxParameter) and not issubclass(p, SandboxInfoParameter) and p in subtask.input_parameters
        }

    def __create_build_task(self):
        ctx = self.__copy_input_parameters(YaPackage)
        ctx = self.__create_ya_package_task_ctx(ctx)

        return self.create_subtask(
            task_type=YaPackage.type,
            description='Ya package subtask for #%d (%s)' % (self.id, self.descr),
            input_parameters=ctx
        )

    def __create_porto_task(self):
        self.__create_packages_install_script()

        ctx = self.__copy_input_parameters(BuildPortoLayer)

        return self.create_subtask(
            task_type=BuildPortoLayer.type,
            description='Build porto layer subtask for #%d (%s)' % (self.id, self.descr),
            input_parameters=self.__create_build_porto_layer_task_ctx(ctx)  # ,
            # arch=self.arch
        )

    def __wait_tasks(self):
        self.wait_tasks(
            self.list_subtasks(load=False),
            self.Status.Group.SUCCEED + self.Status.Group.SCHEDULER_FAILURE,
            wait_all=True,
            state='Waiting for build tasks to complete'
        )

    def __check_tasks(self):
        for task in self.list_subtasks(load=True):
            if not task.is_finished():
                raise SandboxTaskFailureError('Subtask %s has failed (status=%s)' % (task.descr, repr(task.status)))

    def on_execute(self):
        if self._TASK_PACKAGE_KEY not in self.ctx:
            build_task = self.__create_build_task()
            logging.info(build_task)
            self.ctx[self._TASK_PACKAGE_KEY] = build_task.id
            self.__wait_tasks()

        else:
            self.__check_tasks()

            if self._TASK_PORTO_KEY not in self.ctx:
                self.ctx[self._TASK_PORTO_KEY] = self.__create_porto_task().id
                self.__wait_tasks()
            else:
                with open('dummy', 'w') as out:
                    print >>out

                # self.create_resource('', 'dummy', resource_types.RELEASABLE_DUMMY)

    # def on_release(self, additional_parameters):
        # logging.debug('.on_release(%s)' % additional_parameters)
        # self.create_release(self.ctx[self._TASK_PORTO_KEY], additional_parameters=additional_parameters)


__Task__ = BuildPackageAndPortoLayer
