import logging

from sandbox.projects.common.build.YaPackage import YaPackage, PackageTypeParameter, \
                                            DEBIAN as PackageTypeDebian, \
                                            TARBALL as PackageTypeTarball, \
                                            ForceDuploadParameter, \
                                            BuildDebianScriptsParameter, \
                                            PublishPackageParameter, \
                                            ChangelogParameter, \
                                            KeyUserParameter, \
                                            MultiplePublishParameter, \
                                            PublishToParameter, \
                                            MultiplePublishToParameter, \
                                            MultiplePublishMappingParameter, \
                                            SloppyDebianParameter, \
                                            RunTestsParameter, \
                                            ResourceTypeParameter, \
                                            StripBinariesParameter, \
                                            CreateDebugPackagesParameter, \
                                            CompressPackageArchiveParameter

from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.parameters import SandboxParameter, SandboxInfoParameter
from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask
from sandbox.projects import resource_types


class BuildPackagesForRem(ArcadiaTask):
    type = 'BUILD_PACKAGES_FOR_REM'

    _PROXIED_PARAMS_TO_HIDE = frozenset([
        PackageTypeParameter,
        StripBinariesParameter,
        CreateDebugPackagesParameter,
        CompressPackageArchiveParameter,
    ])

    input_parameters = [p for p in YaPackage.input_parameters if p not in _PROXIED_PARAMS_TO_HIDE]

    _SUB_TASKS_CREATED_KEY = '_SUB_TASKS_CREATED'

    @staticmethod
    def __create_tarball_task_ctx(ctx):
        ctx = ctx.copy()

        ctx[PackageTypeParameter.name] = PackageTypeTarball

        to_pop = [
            ForceDuploadParameter,
            BuildDebianScriptsParameter,
            PublishPackageParameter,

            ChangelogParameter,
            KeyUserParameter,
            MultiplePublishParameter,
            PublishToParameter,
            MultiplePublishToParameter,
            MultiplePublishMappingParameter,
            SloppyDebianParameter,
        ]

        for p in to_pop:
            ctx.pop(p.name)

        ctx[StripBinariesParameter.name] = True
        ctx[CreateDebugPackagesParameter.name] = True
        ctx[CompressPackageArchiveParameter.name] = False

        return ctx

    @staticmethod
    def __create_debian_task_ctx(ctx):
        ctx = ctx.copy()

        ctx[PackageTypeParameter.name] = PackageTypeDebian

        ctx.pop(RunTestsParameter.name)  # run if need in tarball-task

        ctx[ResourceTypeParameter.name] = 'YA_PACKAGE'  # FIXME

        ctx[StripBinariesParameter.name] = True
        ctx[CreateDebugPackagesParameter.name] = False  # ATW doesn't work for deb
        ctx.pop(CompressPackageArchiveParameter.name, None)

        return ctx

    def __create_build_task(self, type, ctx):
        return self.create_subtask(
            task_type='YA_PACKAGE',
            description='[%s] Build subtask for #%d (%s)' % (type, self.id, self.descr),
            input_parameters=ctx,
            priority=self.priority,
            arch=self.arch
        )

    def __create_sub_tasks(self):
        ctx = {
            p.name: self.ctx.get(p.name, p.default_value)
                for p in self.input_parameters
                    if issubclass(p, SandboxParameter) and not issubclass(p, SandboxInfoParameter)
        }

        return (
            self.__create_build_task('tarball', self.__create_tarball_task_ctx(ctx)),
            self.__create_build_task('debian',  self.__create_debian_task_ctx(ctx))
        )

    def on_execute(self):
        if self._SUB_TASKS_CREATED_KEY not in self.ctx:
            self.ctx['tarball_task_id'] = self.__create_sub_tasks()[0].id
            self.ctx[self._SUB_TASKS_CREATED_KEY] = True

            self.wait_tasks(
                self.list_subtasks(load=False),
                self.Status.Group.SUCCEED + self.Status.Group.SCHEDULER_FAILURE,
                wait_all=True,
                state='Waiting for build tasks to complete'
            )

        else:
            for task in self.list_subtasks(load=True):
                if not task.is_finished():
                    raise SandboxTaskFailureError('Subtask %s has failed (status=%s)' % (task.descr, repr(task.status)))

            with open('dummy', 'w') as out:
                print >>out

            self.create_resource('', 'dummy', resource_types.RELEASABLE_DUMMY)

    def on_release(self, additional_parameters):
        logging.debug('.on_release(%s)' % additional_parameters)
        self.create_release(self.ctx['tarball_task_id'])


__Task__ = BuildPackagesForRem
