# -*- coding: utf-8 -*-

from contextlib import closing
import os
import tarfile

import sandbox.common.types.client as ctc

from sandbox.projects.BuildSearch import BuildSearch

from sandbox.projects import resource_types
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk.errors import SandboxTaskFailureError, SandboxSvnError
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.projects.common.build import ArcadiaTask as bbt
import sandbox.projects.common.constants as consts


class BuildSocialUsersTools(BuildSearch):
    """
        Сборка бинарников из arcadia для Peoplesearch
    """

    type = 'BUILD_PEOPLESEARCH_TOOLS'
    client_tags = ~ctc.Tag.Group.LINUX | ctc.Tag.LINUX_PRECISE

    TARGET_PATH_TO_NAME_MAP = {
        'quality/trailer/suggest/data_builder/suggest-data-builder': 'suggest_data_builder',
    }

    BUILDING_GROUPS = {
        'Social Users tools': [
            'yweb/peoplesearch/social_users/tools/build_groups_graph/build_groups_graph',
            'yweb/peoplesearch/social_users/tools/calc_factors/calc_factors',
            'yweb/peoplesearch/social_users/tools/calc_freq_factors/calc_freq_factors',
            'yweb/peoplesearch/social_users/tools/clone_detector/clone_detector',
            'yweb/peoplesearch/social_users/tools/cpp_post_process/cpp_post_process',
            'yweb/peoplesearch/social_users/tools/hypothetical_matches/hypothetical_matches',
            'yweb/peoplesearch/social_users/tools/make_metaprofile/make_metaprofile',
            'yweb/peoplesearch/social_users/tools/matcher_mr/matcher_mr',
            'yweb/peoplesearch/social_users/tools/print_social_queries/print_social_queries',
            'yweb/peoplesearch/social_users/tools/proto_fields_freq/proto_fields_freq',
            'yweb/peoplesearch/social_users/tools/smart_components/smart_components',
            'yweb/peoplesearch/social_users/tools/profile_factors_calculator/profile_factors_calculator',
            'yweb/peoplesearch/social_users/tools/profiles_diff/profiles_diff',
            'yweb/peoplesearch/social_users/tools/profiles_diff_printer/profiles_diff_printer',
            'yweb/peoplesearch/social_users/tools/visualize_groups_graph/visualize_groups_graph',
            'yweb/peoplesearch/social_users/tools/facets_dictionary_builder/facets_dictionary_builder',

            'yweb/peoplesearch/tools/suggest/names_uploader/people_suggest_names_uploader',
            'yweb/peoplesearch/tools/suggest/names_downloader/people_suggest_names_downloader',
            'yweb/peoplesearch/tools/suggest/dict_builder/people_suggest_builder',
            'yweb/peoplesearch/tools/suggest/nicks/people_suggest_nicks',
            'yweb/peoplesearch/tools/suggest/debug/people_suggest_debug',
            'yweb/peoplesearch/tools/suggest/tester/people_suggest_tester',

            'yweb/peoplesearch/social_users/lib/common/tests/peoplesearch_ut_lib_common',
            'yweb/peoplesearch/social_users/lib/postprocess/tests/peoplesearch_test_lib_postprocess',

            'yweb/peoplesearch/tools/url2urlid/url2urlid',

            'yweb/peoplesearch/people_index_versions/people_index_versions',
            'yweb/peoplesearch/realtime_distributor/people_realtime_distributor',
            'yweb/peoplesearch/social_users/tools/actualize_posts/actualize_posts',
        ],

        'Social Crawler tools': [
            'yweb/peoplesearch/social_users/crawler/tools/generate_schedule/generate_schedule',
            'yweb/peoplesearch/social_users/crawler/tools/upload_discovered_urls/upload_discovered_urls',
            'yweb/peoplesearch/social_users/crawler/tools/url_database_mr/url_database_mr',
            'yweb/peoplesearch/social_users/crawler/tools/url_updates_to_ukrop/url_updates_to_ukrop',
            'yweb/peoplesearch/social_users/crawler/lib/merge_urlinfo/test/peoplesearch_merge_urlinfo_test',
            'yweb/robot/kiwi/kwworm/kwworm',
        ],

        'People SaaS tools': [
            'yweb/peoplesearch/rtysearch/rtyindexer/people_rtyindexer',
            'yweb/peoplesearch/rtysearch/rtysearch_tool/rtysearch_tool',
            'yweb/peoplesearch/rtysearch/indexed_docs_checker/indexed_docs_checker',
            'yweb/peoplesearch/rtysearch/gen_rtydelta/gen_rtydelta',
        ],

        'People Mapreduce Externals': [
            'quality/mapreduce/mapreduce',

            'quality/mr_apps/mr_cat/mr_cat',
            'quality/mr_apps/mr_data_manip/mr_data_manip',
            'quality/mr_apps/mr_du/mr_du',
            'quality/mr_apps/mr_get_keys/mr_get_keys',
            'quality/mr_apps/mr_head/mr_head',
            'quality/mr_apps/mr_mv/mr_mv',
            'quality/mr_apps/mr_sample/mr_sample',
            'quality/mr_apps/mr_sort/mr_sort',
            'quality/mr_apps/mr_touch/mr_touch',
            'quality/mr_apps/mr_wc/mr_wc',
            'quality/mr_apps/mr_cp/mr_cp',
            'quality/mr_apps/mr_diff/mr_diff',
            'quality/mr_apps/mr_find/mr_find',
            'quality/mr_apps/mr_grep/mr_grep',
            'quality/mr_apps/mr_hist/mr_hist',
            'quality/mr_apps/mr_ls/mr_ls',
            'quality/mr_apps/mr_rm/mr_rm',
            'quality/mr_apps/mr_set_ops/mr_set_ops',
            'quality/mr_apps/mr_stat/mr_stat',
            'quality/mr_apps/mr_uniq/mr_uniq',

            # 'quality/mr_apps/mr_bounds',
            # 'quality/mr_apps/mr_bin_view',
        ],

        'YWeb Social Metrics tools': [
            'yweb/peoplesearch/yweb_social_metrics/tools/metrics_value_combiner_io/metrics_value_combiner_io',
            'yweb/peoplesearch/yweb_social_metrics/tools/attrs_harvester/search_social_attrs_harvester',
            'yweb/peoplesearch/yweb_social_metrics/tools/grabber/social_page_info_grabber',
            'yweb/peoplesearch/yweb_social_metrics/tools/metrics_constructor/yweb_social_metrics_constructor',
            'yweb/peoplesearch/yweb_social_metrics/tools/ut_libs/yweb_social_metrics_lib_ut',
        ],

        'People Other Branches Externals': [
            'tools/printwzrd/printwzrd',
            'yweb/robot/zoracl/main/zoracl',
            'contrib/tools/protoc/protoc',
            'quality/trailer/suggest/data_builder/suggest-data-builder',
            'yweb/querydata/querydata_indexer/querydata_indexer',
            'yweb/querydata/tools/url2docid/url2docid',
            'robot/gemini/mr/actualize_kiwi_export/actualize_kiwi_export',
            'yweb/antispam/grimhold/grimhold/grimhold',
        ],

        'YWeb Social Snippet tools': [
            'yweb/peoplesearch/yweb_social_snippet/tools/yweb_social_snippet_generator_mr/yweb_social_snippet_generator_mr',
            'yweb/peoplesearch/yweb_social_snippet/tools/socnetsnip_querydata_formatter/socnetsnip_querydata_formatter',
        ],
    }

    TARGET_RESOURCES = []
    TARGET_RESOURCE_TYPES = []

    build_fields = {}

    for name, elements in BUILDING_GROUPS.iteritems():
        group_content = []
        cgi_name = name.replace(' ', '_')
        for element_path in elements:
            element_name = TARGET_PATH_TO_NAME_MAP.get(element_path, os.path.basename(element_path))
            group_content.append(element_name)
            TARGET_RESOURCES.append((resource_types.PEOPLESEARCH_TOOL, element_name, element_path))
        build_fields[cgi_name] = group_content

    input_parameters = BuildSearch.build_search_params + bbt.gen_input_params(build_fields)

    RESOURCE_TYPE = resource_types.PEOPLESEARCH_TOOLS_ARCHIVE
    ARCHIVE_NAME = 'ppl_tools.tar.gz'

    def _target_resource_id(self, target):
        """
            если id ресурса не получен - сразу упасть
        """
        resource_id = self.ctx.get('%s_resource_id' % target, None)
        if resource_id is None:
            raise SandboxTaskFailureError('no resource for %s_resource_id' % target)
        return resource_id

    def do_on_enqueue(self):
        """
            переопределение BuildSearch.do_on_enqueue с созданием ресурса для архива
        """
        BuildSearch.do_on_enqueue(self)

        try:
            version = Arcadia.info(self.ctx.get(consts.ARCADIA_URL_KEY))['commit_revision']
        except SandboxSvnError:
            version = 0
        attributes = {'version': version}

        resource = self._create_resource(resource_desc=self.descr,
                                         resource_filename=self.ARCHIVE_NAME,
                                         resource_type=self.RESOURCE_TYPE,
                                         arch=self.arch,
                                         attrs=attributes)
        self.set_info('create resource for PEOPLESEARCH_TOOLS_ARCHIVE, %s' % resource.id)
        self.ctx['peoplesearch_tools_id'] = resource.id
        self.set_info('task enqueued for execution')

    def __list_existing_executables(self, path=None):
        if path is None:
            path = self.abs_path(self.LOCAL_BIN_DIR)
        for filename in paths.list_dir(path, abs_path=True, files_only=True):
            if os.access(filename, os.X_OK):
                yield filename
        for dirname in paths.list_dir(path, abs_path=True, folders_only=True):
            for filename in self.__list_existing_executables(dirname):
                yield filename

    def do_execute(self):
        BuildSearch.do_execute(self)

        archive_name = self.abs_path(self.ARCHIVE_NAME)
        with closing(tarfile.TarFile.gzopen(archive_name, mode='w')) as archive:
            for executable in self.__list_existing_executables():
                archive.add(executable, os.path.basename(executable), recursive=False)


__Task__ = BuildSocialUsersTools
