# coding: utf-8

import tarfile
import os
import logging
from sandbox.projects import resource_types
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.environments import GCCEnvironment
from sandbox.sandboxsdk.process import run_process
from sandbox.projects.common.environments import *  # noqa

import sandbox.common.types.client as ctc


class NcursesVer(SandboxStringParameter):
    name = 'ncurses_ver'
    description = 'Ncurses version'
    default_value = '6.0'
    required = True


class BoostVer(SandboxStringParameter):
    name = 'boost_ver'
    description = 'Boost version'
    default_value = '1.59.0'
    required = True


class BisonVer(SandboxStringParameter):
    name = 'bison_ver'
    description = 'Bison version'
    default_value = '3.0.4'
    required = True


class ZlibVer(SandboxStringParameter):
    name = 'zlib_ver'
    description = 'Zlib version'
    default_value = '1.2.8'
    required = True


class LibevVer(SandboxStringParameter):
    name = 'libev_ver'
    description = 'Libev version'
    default_value = '4.22'
    required = True


class GcryptVer(SandboxStringParameter):
    name = 'gcrypt_ver'
    description = 'Libgcrypt version'
    default_value = '1.7.3'
    required = True


class GpgerrorVer(SandboxStringParameter):
    name = 'gpgerror_ver'
    description = 'Libgpg-error version'
    default_value = '1.24'
    required = True


class OpensslVer(SandboxStringParameter):
    name = 'openssl_ver'
    description = 'Openssl version'
    default_value = '1.0.2h'
    required = True


class LibcurlVer(SandboxStringParameter):
    name = 'curl_ver'
    description = 'Libcurl version'
    default_value = '7.50.1'
    required = True


class XtrabackupVer(SandboxStringParameter):
    name = 'xtrabackup_ver'
    description = 'Xtrabackup version'
    default_value = '2.4.4'
    required = True


class BuildPerconaXtraBackup(SandboxTask):
    type = 'BUILD_PERCONA_XTRABACKUP'

    client_tags = ctc.Tag.LINUX_PRECISE & ctc.Tag.IPV4

    DEPS_DIR = ""
    INSTALL_DIR = ""
    AUTOMAKE_DSTDIR = ""
    OPENSSL_DSTDIR = ""
    INCLUDE_DIR = ""
    LIB_DIR = ""
    TGZ_NAME = 'xtrabackup.tar.bz2'
    NCURSES_FLDR = ""
    NCURSES_TAR = ""
    NCURSES_URL = ""
    NCURSES_URL_PREFIX = "ftp://invisible-island.net/ncurses/"
    NCURSES_STATIC = ""
    BOOST_FLDR = ""
    BOOST_TAR = ""
    BOOST_URL_PREFIX = "https://sourceforge.net/projects/boost/files/boost/"
    BOOST_URL = ""
    BOOST_DIR = ""
    BISON_FLDR = ""
    BISON_DIR = ""
    BISON_TAR = ""
    BISON_URL_PREFIX = "ftp://ftp.gnu.org/gnu/bison/"
    BISON_URL = ""
    BISON_BIN = ""
    ZLIB_FLDR = ""
    ZLIB_TAR = ""
    ZLIB_URL = ""
    ZLIB_URL_PREFIX = "http://zlib.net/current/"
    GCRYPT_FLDR = ""
    GCRYPT_TAR = ""
    GCRYPT_URL = ""
    GCRYPT_URL_PREFIX = "ftp://ftp.gnupg.org/gcrypt/libgcrypt/"
    GCRYPT_STATIC = ""
    GPGERROR_FLDR = ""
    GPGERROR_TAR = ""
    GPGERROR_URL = ""
    GPGERROR_URL_PREFIX = "ftp://ftp.gnupg.org/gcrypt/libgpg-error/"
    GPGERROR_STATIC = ""
    LIBEV_FLDR = ""
    LIBEV_TAR = ""
    LIBEV_URL = ""
    LIBEV_URL_PREFIX = "http://dist.schmorp.de/libev/"
    LIBEV_STATIC = ""
    CURL_FLDR = ""
    CURL_TAR = ""
    CURL_URL = ""
    CURL_URL_PREFIX = "http://curl.haxx.se/download/"
    CURL_STATIC = ""
    OPENSSL_FLDR = ""
    OPENSSL_TAR = ""
    OPENSSL_URL_PREFIX = "https://www.openssl.org/source/"
    OPENSSL_URL = ""
    OPENSSL_STATIC = ""
    OPENSSL_DIR = ""
    XTRABACKUP_FLDR = ""
    XTRABACKUP_TAR = ""
    XTRABACKUP_URL = ""
    XTRABACKUP_URL_PREFIX = "https://www.percona.com/downloads/XtraBackup/"

    RESULT_DIR = ""

    environment = [GCCEnvironment(), ]
    input_parameters = [NcursesVer, BoostVer, BisonVer, ZlibVer, LibevVer, GcryptVer, GpgerrorVer, OpensslVer, LibcurlVer, XtrabackupVer]

    def on_execute(self):
        pwd = os.path.abspath(".")
        self.DEPS_DIR = os.path.join(pwd, "dependencies")
        self.INSTALL_DIR = os.path.join(pwd, "install")
        self.AUTOMAKE_DSTDIR = 'DESTDIR={}'.format(self.INSTALL_DIR)
        self.OPENSSL_DSTDIR = 'INSTALL_PREFIX={}'.format(self.INSTALL_DIR)
        self.INCLUDE_DIR = os.path.join(self.INSTALL_DIR, "include")
        self.LIB_DIR = os.path.join(self.INSTALL_DIR, "lib")
        ncurses_version = self.ctx.get('ncurses_ver')
        boost_version = self.ctx.get('boost_ver')
        bison_version = self.ctx.get('bison_ver')
        zlib_version = self.ctx.get('zlib_ver')
        openssl_version = self.ctx.get('openssl_ver')
        xtrabackup_version = self.ctx.get('xtrabackup_ver')
        self.NCURSES_FLDR = "ncurses-{}".format(self.ctx.get('ncurses_ver'))
        self.NCURSES_TAR = "{}.tar.gz".format(self.NCURSES_FLDR)
        self.NCURSES_URL = "{}{}".format(self.NCURSES_URL_PREFIX, self.NCURSES_TAR)
        self.NCURSES_STATIC = os.path.join(self.LIB_DIR, 'libncursesw.a')
        self.BOOST_FLDR = "boost_{}".format(boost_version.replace(".", "_"))
        self.BOOST_TAR = "{}.tar.gz".format(self.BOOST_FLDR)
        self.BOOST_URL = "{}{}/{}".format(self.BOOST_URL_PREFIX, boost_version, self.BOOST_TAR)
        self.BOOST_DIR = os.path.join(self.DEPS_DIR, self.BOOST_FLDR)
        self.BISON_FLDR = "bison-{}".format(bison_version)
        self.BISON_DIR = os.path.join(self.DEPS_DIR, self.BISON_FLDR)
        self.BISON_TAR = "{}.tar.gz".format(self.BISON_FLDR)
        self.BISON_URL = "{}/{}".format(self.BISON_URL_PREFIX, self.BISON_TAR)
        self.BISON_BIN = os.path.join(self.INSTALL_DIR, "bin/bison")
        self.ZLIB_FLDR = "zlib-{}".format(self.ctx.get('zlib_ver'))
        self.ZLIB_TAR = "{}.tar.gz".format(self.ZLIB_FLDR)
        self.ZLIB_URL = "{}{}".format(self.ZLIB_URL_PREFIX, self.ZLIB_TAR)
        self.GCRYPT_FLDR = "libgcrypt-{}".format(self.ctx.get('gcrypt_ver'))
        self.GCRYPT_TAR = "{}.tar.bz2".format(self.GCRYPT_FLDR)
        self.GCRYPT_URL = "{}{}".format(self.GCRYPT_URL_PREFIX, self.GCRYPT_TAR)
        self.GCRYPT_STATIC = os.path.join(self.LIB_DIR, 'libgcrypt.a')
        self.GPGERROR_FLDR = "libgpg-error-{}".format(self.ctx.get('gpgerror_ver'))
        self.GPGERROR_TAR = "{}.tar.bz2".format(self.GPGERROR_FLDR)
        self.GPGERROR_URL = "{}{}".format(self.GPGERROR_URL_PREFIX, self.GPGERROR_TAR)
        self.GPGERROR_STATIC = os.path.join(self.LIB_DIR, 'libgpg-error.a')
        self.LIBEV_FLDR = "libev-{}".format(self.ctx.get('libev_ver'))
        self.LIBEV_TAR = "{}.tar.gz".format(self.LIBEV_FLDR)
        self.LIBEV_URL = "{}{}".format(self.LIBEV_URL_PREFIX, self.LIBEV_TAR)
        self.LIBEV_STATIC = os.path.join(self.LIB_DIR, 'libev.a')
        self.CURL_FLDR = "curl-{}".format(self.ctx.get('curl_ver'))
        self.CURL_TAR = "{}.tar.gz".format(self.CURL_FLDR)
        self.CURL_URL = "{}{}".format(self.CURL_URL_PREFIX, self.CURL_TAR)
        self.CURL_STATIC = os.path.join(self.LIB_DIR, 'libcurl.a')
        self.OPENSSL_FLDR = "openssl-{}".format(openssl_version)
        self.OPENSSL_TAR = "{}.tar.gz".format(self.OPENSSL_FLDR)
        self.OPENSSL_URL = "{}/{}".format(self.OPENSSL_URL_PREFIX, self.OPENSSL_TAR)
        self.OPENSSL_DIR = os.path.join(self.DEPS_DIR, self.OPENSSL_FLDR)
        self.XTRABACKUP_FLDR = "percona-xtrabackup-{}".format(xtrabackup_version)
        self.XTRABACKUP_TAR = "{}.tar.gz".format(self.XTRABACKUP_FLDR)
        self.XTRABACKUP_URL = "{}Percona-XtraBackup-{}/source/tarball/{}".format(self.XTRABACKUP_URL_PREFIX, xtrabackup_version, self.XTRABACKUP_TAR)
        self.RESULT_DIR = os.path.join(pwd, "result")

        self._gcc_init()

        logging.info('Building dependencies for percona')
        os.mkdir(self.DEPS_DIR)
        os.mkdir(self.INSTALL_DIR)
        os.mkdir(self.RESULT_DIR)
        self._build_static_zlib(zlib_version)
        self._build_static_openssl(openssl_version)
        self._build_static_curl()
        self._build_static_libev()
        self._build_static_gpgerror()
        self._build_static_gcrypt()
        self._build_static_ncurses(ncurses_version)
        self._build_bison(bison_version)
        self._build_static_boost(boost_version)
        self._build_xtrabackup()

        logging.info('Creating %s...', self.TGZ_NAME)
        resource_path = self.path(self.TGZ_NAME)
        with tarfile.open(resource_path, 'w:bz2') as tar:
            for entry in os.listdir(self.RESULT_DIR):
                tar.add(os.path.join(self.RESULT_DIR, entry), entry)

        self.create_resource(
            description=self.ctx.get('description', 'Percona Xtrabackup {}'.format(xtrabackup_version)),
            resource_path=resource_path,
            resource_type=resource_types.PERCONA_XTRABACKUP,
            arch='linux'
        )

    def _gcc_init(self):
        logging.info('Preparing environment (gcc, g++)...')
        compilers = self.environment[0].get_compilers()
        os.environ['CC'] = gcc_binary = compilers['CC']
        os.environ['CXX'] = gxx_binary = compilers['CXX']
        logging.info('Fetched compilers are: CC=%s CXX=%s', gcc_binary, gxx_binary)

    def _wget(self, url, dst_dir):
        logging.info('Downloading %s', url)
        run_process(['wget', '--retry-connrefused', url, '-P', dst_dir], log_prefix='wget')

    def _unpack(self, archive_path, dst_dir):
        logging.info("Extracting {} to {}", archive_path, dst_dir)
        tar = tarfile.open(archive_path)
        tar.extractall(path=dst_dir)

    def _build_static_ncurses(self, ncurses_ver):
        logging.info('Building ncurses')
        self._wget(self.NCURSES_URL, self.DEPS_DIR)
        self._unpack(os.path.join(self.DEPS_DIR, self.NCURSES_TAR), self.DEPS_DIR)
        ncurses_dir = os.path.join(self.DEPS_DIR, self.NCURSES_FLDR)
        ncurses_buildflags = ['./configure', '--prefix=/', '--without-shared', '--with-normal',
            '--without-debug', '--without-ada', '--enable-widec', '--enable-pc-files', '--without-cxx-binding',
            '--enable-ext-colors', '--enable-ext-mouse']
        run_process(ncurses_buildflags, work_dir=ncurses_dir, log_prefix='configure_ncurses')
        run_process(['make'], work_dir=ncurses_dir, log_prefix='make_ncurses')
        run_process(['make', self.AUTOMAKE_DSTDIR, 'install'], work_dir=ncurses_dir, log_prefix='make_install_ncurses')

    def _build_static_boost(self, boost_ver):
        logging.info('Building boost')
        self._wget(self.BOOST_URL, self.DEPS_DIR)
        self._unpack(os.path.join(self.DEPS_DIR, self.BOOST_TAR), self.DEPS_DIR)
        run_process(['./bootstrap.sh', '--with-toolset=gcc'], work_dir=self.BOOST_DIR, log_prefix='bootstrap_boost')
        boost_buildflags = ['./b2', '--prefix=/', '--build-type=minimal', '-s', 'NO_BZIP2=1', 'variant=release', 'link=static',
            'threading=multi', 'runtime-link=static']
        run_process(boost_buildflags, work_dir=self.BOOST_DIR, log_prefix='b2_boost')

    def _build_bison(self, bison_ver):
        logging.info('Building bison')
        self._wget(self.BISON_URL, self.DEPS_DIR)
        self._unpack(os.path.join(self.DEPS_DIR, self.BISON_TAR), self.DEPS_DIR)
        bison_buildflags = ['./configure', '--prefix=/']
        run_process(bison_buildflags, work_dir=self.BISON_DIR, log_prefix='configure_bison')
        run_process(['make'], work_dir=self.BISON_DIR, log_prefix='make_bison')
        run_process(['make', self.AUTOMAKE_DSTDIR, 'install'], work_dir=self.BISON_DIR, log_prefix='make_install_bison')

    def _build_static_zlib(self, zlib_ver):
        logging.info('Building zlib')
        self._wget(self.ZLIB_URL, self.DEPS_DIR)
        self._unpack(os.path.join(self.DEPS_DIR, self.ZLIB_TAR), self.DEPS_DIR)
        zlib_dir = os.path.join(self.DEPS_DIR, self.ZLIB_FLDR)
        zlib_buildflags = ['./configure', '--prefix=/', '--static']
        run_process(zlib_buildflags, work_dir=zlib_dir, log_prefix='configure_zlib')
        run_process(['make', 'CFLAGS=-fPIC -O3 -D_LARGEFILE64_SOURCE=1 -DHAVE_HIDDEN'], work_dir=zlib_dir, log_prefix='make_zlib')
        run_process(['make', self.AUTOMAKE_DSTDIR, 'install'], work_dir=zlib_dir, log_prefix='make_install_zlib')

    def _build_static_libev(self):
        logging.info('Building libev')
        self._wget(self.LIBEV_URL, self.DEPS_DIR)
        self._unpack(os.path.join(self.DEPS_DIR, self.LIBEV_TAR), self.DEPS_DIR)
        libev_dir = os.path.join(self.DEPS_DIR, self.LIBEV_FLDR)
        libev_buildflags = ['./configure', '--prefix=/', '--enable-static', '--disable-shared']
        run_process(libev_buildflags, work_dir=libev_dir, log_prefix='configure_libev')
        run_process(['make'], work_dir=libev_dir, log_prefix='make_libev')
        run_process(['make', self.AUTOMAKE_DSTDIR, 'install'], work_dir=libev_dir, log_prefix='make_install_libev')

    def _build_static_gcrypt(self):
        logging.info('Building gcrypt')
        self._wget(self.GCRYPT_URL, self.DEPS_DIR)
        self._unpack(os.path.join(self.DEPS_DIR, self.GCRYPT_TAR), self.DEPS_DIR)
        gcrypt_dir = os.path.join(self.DEPS_DIR, self.GCRYPT_FLDR)
        gcrypt_buildflags = ['./configure', '--prefix=/', '--enable-static', '--disable-shared', '--with-libgpg-error-prefix={}'.format(self.INSTALL_DIR)]
        run_process(gcrypt_buildflags, work_dir=gcrypt_dir, log_prefix='configure_gcrypt')
        run_process(['make'], work_dir=gcrypt_dir, log_prefix='make_gcrypt')
        run_process(['make', self.AUTOMAKE_DSTDIR, 'install'], work_dir=gcrypt_dir, log_prefix='make_install_gcrypt')

    def _build_static_gpgerror(self):
        logging.info('Building libgpgerror')
        self._wget(self.GPGERROR_URL, self.DEPS_DIR)
        self._unpack(os.path.join(self.DEPS_DIR, self.GPGERROR_TAR), self.DEPS_DIR)
        gpgerror_dir = os.path.join(self.DEPS_DIR, self.GPGERROR_FLDR)
        gpgerror_buildflags = ['./configure', '--prefix=/', '--enable-static', '--disable-shared']
        run_process(gpgerror_buildflags, work_dir=gpgerror_dir, log_prefix='configure_gpgerror')
        run_process(['make'], work_dir=gpgerror_dir, log_prefix='make_gpgerror')
        run_process(['make', self.AUTOMAKE_DSTDIR, 'install'], work_dir=gpgerror_dir, log_prefix='make_install_gpgerror')
        to_replace = "^prefix=/"
        new_str = "prefix={}".format(self.INSTALL_DIR)
        insane_sed = ['/bin/sed', '-i', 's%{}%{}%g'.format(to_replace, new_str), os.path.join(self.INSTALL_DIR, "bin/gpg-error-config")]
        run_process(insane_sed, log_prefix='insane_patching_gpg-error-config')

    def _build_static_openssl(self, openssl_ver):
        logging.info('Building openssl')
        self._wget(self.OPENSSL_URL, self.DEPS_DIR)
        self._unpack(os.path.join(self.DEPS_DIR, self.OPENSSL_TAR), self.DEPS_DIR)
        openssl_buildflags = ['/usr/bin/perl', './Configure', '--prefix=/', 'no-shared',
            'enable-ec_nistp_64_gcc_128', 'linux-x86_64', '-Wa,--noexecstack', '-fPIC',
            '-I{}'.format(self.INCLUDE_DIR), '-L{}'.format(self.LIB_DIR)]
        run_process(openssl_buildflags, work_dir=self.OPENSSL_DIR, log_prefix='configure_openssl')
        run_process(['make'], work_dir=self.OPENSSL_DIR, log_prefix='make_openssl')
        run_process(['make', self.OPENSSL_DSTDIR, 'LIBDIR=lib', 'install'], work_dir=self.OPENSSL_DIR, log_prefix='make_install_openssl')

    def _build_static_curl(self):
        logging.info('Building curl')
        self._wget(self.CURL_URL, self.DEPS_DIR)
        self._unpack(os.path.join(self.DEPS_DIR, self.CURL_TAR), self.DEPS_DIR)
        curl_dir = os.path.join(self.DEPS_DIR, self.CURL_FLDR)
        curl_buildflags = ['./configure', '--prefix=/', '--enable-static', '--disable-shared', '--disable-ldap', '--disable-ldaps', '--enable-ipv6',
            '--with-gssapi', '--with-libidn', '--with-random=/dev/urandom', '--with-ca-bundle=/etc/ssl/certs/ca-certificates.crt', '--with-ssl={}'.format(self.INCLUDE_DIR)]
        run_process(curl_buildflags, work_dir=curl_dir, log_prefix='configure_curl')
        run_process(['make'], work_dir=curl_dir, log_prefix='make_curl')
        run_process(['make', self.AUTOMAKE_DSTDIR, 'install'], work_dir=curl_dir, log_prefix='make_install_curl')

    def _build_xtrabackup(self):
        logging.info('Building percona xtrabackup')
        self._wget(self.XTRABACKUP_URL, self.DEPS_DIR)
        self._unpack(os.path.join(self.DEPS_DIR, self.XTRABACKUP_TAR), self.DEPS_DIR)
        tmp_dir = os.path.join(self.DEPS_DIR, 'tmp')
        os.mkdir(tmp_dir)
        tmp_bin_dir = os.path.join(tmp_dir, 'usr/local/xtrabackup/bin/')
        res_bin_dir = os.path.join(self.RESULT_DIR, 'bin')
        xtrabackup_dir = os.path.join(self.DEPS_DIR, self.XTRABACKUP_FLDR)
        os.environ['BISON_PKGDATADIR'] = '{}/share/bison'.format(self.INSTALL_DIR)
        CFLAGS = "-static-libgcc -static-libstdc++ -I{}".format(self.INCLUDE_DIR)
        cmake = ['/usr/bin/cmake', '.', '-DCMAKE_CXX_FLAGS={}'.format(CFLAGS), '-DCMAKE_C_FLAGS={}'.format(CFLAGS),
            '-DBISON_EXECUTABLE={}'.format(self.BISON_BIN), '-DWITH_ZLIB=bundled', '-DWITH_SSL={}'.format(self.INSTALL_DIR),
            '-DCURSES_INCLUDE_PATH={}'.format(self.INCLUDE_DIR), '-DCURSES_LIBRARY={}'.format(self.NCURSES_STATIC),
            '-DGCRYPT_LIB={}'.format(self.GCRYPT_STATIC), '-DGCRYPT_INCLUDE_DIR={}'.format(self.INCLUDE_DIR), '-DLIBEV_INCLUDE_DIRS={}'.format(self.INCLUDE_DIR),
            '-DLIBEV_LIB={}'.format(self.LIBEV_STATIC), '-DCURL_LIBRARY={}'.format(self.CURL_STATIC), '-DCURL_INCLUDE_DIR={}'.format(self.INCLUDE_DIR),
            '-DZLIB_INCLUDE_DIR={}'.format(self.INCLUDE_DIR), '-DWITH_BOOST={}'.format(self.BOOST_DIR), '-DGPG_ERROR_LIB={}'.format(self.GPGERROR_STATIC)]
        run_process(cmake, work_dir=xtrabackup_dir, log_prefix='cmake_xtrabackup')
        run_process(['make'], work_dir=xtrabackup_dir, log_prefix='make_xtrabackup')
        logging.info('make install to {}'.format(self.RESULT_DIR))
        run_process(['make', 'DESTDIR={}'.format(tmp_dir), 'install'], work_dir=xtrabackup_dir, log_prefix='make_install_xtrabackup')
        os.rename(tmp_bin_dir, res_bin_dir)


__Task__ = BuildPerconaXtraBackup
