# coding: utf-8

import tarfile
import os
import shutil
import logging
import glob
from sandbox.projects import resource_types
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.environments import GCCEnvironment
from sandbox.sandboxsdk.process import run_process
from sandbox.projects.common.environments import *  # noqa

import sandbox.common.types.client as ctc


class NcursesVer(SandboxStringParameter):
    name = 'ncurses_ver'
    description = 'Ncurses version'
    default_value = '6.0'
    required = True


class BoostVer(SandboxStringParameter):
    name = 'boost_ver'
    description = 'Boost version'
    default_value = '1.59.0'
    required = True


class BisonVer(SandboxStringParameter):
    name = 'bison_ver'
    description = 'Bison version'
    default_value = '3.0.4'
    required = True


class CheckVer(SandboxStringParameter):
    name = 'check_ver'
    description = 'Check version'
    default_value = '0.10.0'
    required = True


class SconsVer(SandboxStringParameter):
    name = 'scons_ver'
    description = 'Scons version'
    default_value = '2.4.1'
    required = True


class ZlibVer(SandboxStringParameter):
    name = 'zlib_ver'
    description = 'Zlib version'
    default_value = '1.2.8'
    required = True


class OpensslVer(SandboxStringParameter):
    name = 'openssl_ver'
    description = 'Openssl version'
    default_value = '1.0.2h'
    required = True


class PerconaTag(SandboxStringParameter):
    name = 'percona_tag'
    description = 'Percona git tag'
    default_value = 'Percona-XtraDB-Cluster-5.7.14-26.17'
    required = True


class GaleraTag(SandboxStringParameter):
    name = 'galera_tag'
    description = 'Galera git tag'
    default_value = 'pxc_5.7.14-26.17'
    required = True


class BuildPerconaXtradbCluster(SandboxTask):
    type = 'BUILD_PERCONA_XTRADB_CLUSTER'

    client_tags = ctc.Tag.LINUX_PRECISE & ctc.Tag.IPV4

    DEPS_DIR = ""
    INSTALL_DIR = ""
    AUTOMAKE_DSTDIR = ""
    OPENSSL_DSTDIR = ""
    INCLUDE_DIR = ""
    LIB_DIR = ""
    TGZ_NAME = 'percona.tar.bz2'
    NCURSES_FLDR = ""
    NCURSES_TAR = ""
    NCURSES_URL = ""
    NCURSES_URL_PREFIX = "ftp://invisible-island.net/ncurses/"
    NCURSES_STATIC = ""
    BOOST_FLDR = ""
    BOOST_TAR = ""
    BOOST_URL_PREFIX = "https://sourceforge.net/projects/boost/files/boost/"
    BOOST_URL = ""
    BISON_FLDR = ""
    BISON_DIR = ""
    BISON_TAR = ""
    BISON_URL_PREFIX = "ftp://ftp.gnu.org/gnu/bison/"
    BISON_URL = ""
    BISON_BIN = ""
    CHECK_FLDR = ""
    CHECK_TAR = ""
    CHECK_URL_PREFIX = "https://github.com/libcheck/check/archive/"
    CHECK_URL = ""
    CHECK_STATIC = ""
    SCONS_TAR = ""
    SCONS_URL_PREFIX = "http://downloads.sourceforge.net/sourceforge/scons/"
    SCONS_URL = ""
    SCONS_BINARY = ""
    ZLIB_FLDR = ""
    ZLIB_TAR = ""
    ZLIB_URL = ""
    ZLIB_URL_PREFIX = "http://zlib.net/current/"
    OPENSSL_FLDR = ""
    OPENSSL_TAR = ""
    OPENSSL_URL_PREFIX = "https://www.openssl.org/source/"
    OPENSSL_URL = ""
    OPENSSL_STATIC = ""
    OPENSSL_DIR = ""
    PERCONA_URL = "https://github.com/percona/percona-xtradb-cluster.git"
    PERCONA_TAG = ""
    PERCONA_DIR = ""
    GALERA_URL = "https://github.com/percona/galera"
    GALERA_TAG = ""
    GALERA_DIR = ""
    RESULT_DIR = ""

    environment = [GCCEnvironment(), ]
    input_parameters = [NcursesVer, BoostVer, BisonVer, CheckVer, SconsVer, ZlibVer, OpensslVer, PerconaTag, GaleraTag]

    def on_execute(self):
        pwd = os.path.abspath(".")
        self.DEPS_DIR = os.path.join(pwd, "dependencies")
        self.INSTALL_DIR = os.path.join(pwd, "install")
        self.AUTOMAKE_DSTDIR = 'DESTDIR={}'.format(self.INSTALL_DIR)
        self.OPENSSL_DSTDIR = 'INSTALL_PREFIX={}'.format(self.INSTALL_DIR)
        self.INCLUDE_DIR = os.path.join(self.INSTALL_DIR, "include")
        self.LIB_DIR = os.path.join(self.INSTALL_DIR, "lib")
        ncurses_version = self.ctx.get('ncurses_ver')
        boost_version = self.ctx.get('boost_ver')
        bison_version = self.ctx.get('bison_ver')
        check_version = self.ctx.get('check_ver')
        scons_version = self.ctx.get('scons_ver')
        zlib_version = self.ctx.get('zlib_ver')
        openssl_version = self.ctx.get('openssl_ver')
        self.NCURSES_FLDR = "ncurses-{}".format(self.ctx.get('ncurses_ver'))
        self.NCURSES_TAR = "{}.tar.gz".format(self.NCURSES_FLDR)
        self.NCURSES_URL = "{}{}".format(self.NCURSES_URL_PREFIX, self.NCURSES_TAR)
        self.NCURSES_STATIC = os.path.join(self.LIB_DIR, 'libncursesw.a')
        self.BOOST_FLDR = "boost_{}".format(boost_version.replace(".", "_"))
        self.BOOST_TAR = "{}.tar.gz".format(self.BOOST_FLDR)
        self.BOOST_URL = "{}{}/{}".format(self.BOOST_URL_PREFIX, boost_version, self.BOOST_TAR)
        self.BISON_FLDR = "bison-{}".format(bison_version)
        self.BISON_DIR = os.path.join(self.DEPS_DIR, self.BISON_FLDR)
        self.BISON_TAR = "{}.tar.gz".format(self.BISON_FLDR)
        self.BISON_URL = "{}/{}".format(self.BISON_URL_PREFIX, self.BISON_TAR)
        self.BISON_BIN = os.path.join(self.INSTALL_DIR, "bin/bison")
        self.CHECK_FLDR = "check-{}".format(check_version)
        self.CHECK_TAR = "{}.tar.gz".format(check_version)
        self.CHECK_URL = "{}/{}".format(self.CHECK_URL_PREFIX, self.CHECK_TAR)
        self.CHECK_STATIC = os.path.join(self.LIB_DIR, 'libcheck.a')
        self.SCONS_FLDR = "scons-{}".format(scons_version)
        self.SCONS_TAR = "{}.tar.gz".format(self.SCONS_FLDR)
        self.SCONS_URL = "{}/{}".format(self.SCONS_URL_PREFIX, self.SCONS_TAR)
        self.SCONS_BINARY = os.path.join(self.INSTALL_DIR, "bin/scons")
        self.ZLIB_FLDR = "zlib-{}".format(self.ctx.get('zlib_ver'))
        self.ZLIB_TAR = "{}.tar.gz".format(self.ZLIB_FLDR)
        self.ZLIB_URL = "{}{}".format(self.ZLIB_URL_PREFIX, self.ZLIB_TAR)
        self.OPENSSL_FLDR = "openssl-{}".format(openssl_version)
        self.OPENSSL_TAR = "{}.tar.gz".format(self.OPENSSL_FLDR)
        self.OPENSSL_URL = "{}/{}".format(self.OPENSSL_URL_PREFIX, self.OPENSSL_TAR)
        self.OPENSSL_DIR = os.path.join(self.DEPS_DIR, self.OPENSSL_FLDR)
        self.PERCONA_DIR = os.path.join(self.DEPS_DIR, "percona-xtradb-cluster")
        self.PERCONA_TAG = self.ctx.get('percona_tag')
        self.GALERA_DIR = os.path.join(self.PERCONA_DIR, "percona-xtradb-cluster-galera")
        self.GALERA_TAG = self.ctx.get('galera_tag')
        self.RESULT_DIR = os.path.join(pwd, "result")

        self._gcc_init()

        logging.info('Building dependencies for percona')
        os.mkdir(self.DEPS_DIR)
        os.mkdir(self.INSTALL_DIR)
        os.mkdir(self.RESULT_DIR)
        self._build_static_ncurses(ncurses_version)
        self._build_static_zlib(zlib_version)
        self._build_static_openssl(openssl_version)
        self._build_bison(bison_version)
        self._prepare()
        self._build_static_boost(boost_version)
        self._build_precona()
        self._build_static_check(check_version)
        self._build_scons(scons_version)
        self._build_galera()

        logging.info('Creating %s...', self.TGZ_NAME)
        resource_path = self.path(self.TGZ_NAME)
        with tarfile.open(resource_path, 'w:bz2') as tar:
            for entry in os.listdir(self.RESULT_DIR):
                tar.add(os.path.join(self.RESULT_DIR, entry), entry)

        self.create_resource(
            description=self.ctx.get('description', 'Percona XtraDB Cluster tag={} and Galera={}'.format(self.PERCONA_TAG, self.GALERA_TAG)),
            resource_path=resource_path,
            resource_type=resource_types.PERCONA_XTRADB_CLUSTER,
            arch='linux'
        )

    def _gcc_init(self):
        logging.info('Preparing environment (gcc, g++)...')
        compilers = self.environment[0].get_compilers()
        os.environ['CC'] = gcc_binary = compilers['CC']
        os.environ['CXX'] = gxx_binary = compilers['CXX']
        logging.info('Fetched compilers are: CC=%s CXX=%s', gcc_binary, gxx_binary)

    def _wget(self, url, dst_dir):
        logging.info('Downloading %s', url)
        run_process(['wget', '--retry-connrefused', url, '-P', dst_dir], log_prefix='wget')

    def _unpack(self, archive_path, dst_dir):
        logging.info("Extracting {} to {}", archive_path, dst_dir)
        tar = tarfile.open(archive_path)
        tar.extractall(path=dst_dir)

    def _build_static_ncurses(self, ncurses_ver):
        logging.info('Building ncurses')
        self._wget(self.NCURSES_URL, self.DEPS_DIR)
        self._unpack(os.path.join(self.DEPS_DIR, self.NCURSES_TAR), self.DEPS_DIR)
        ncurses_dir = os.path.join(self.DEPS_DIR, self.NCURSES_FLDR)
        ncurses_buildflags = ['./configure', '--prefix=/', '--without-shared', '--with-normal',
            '--without-debug', '--without-ada', '--enable-widec', '--enable-pc-files', '--without-cxx-binding',
            '--enable-ext-colors', '--enable-ext-mouse']
        run_process(ncurses_buildflags, work_dir=ncurses_dir, log_prefix='configure_ncurses')
        run_process(['make'], work_dir=ncurses_dir, log_prefix='make_ncurses')
        run_process(['make', self.AUTOMAKE_DSTDIR, 'install'], work_dir=ncurses_dir, log_prefix='make_install_ncurses')

    def _build_static_boost(self, boost_ver):
        logging.info('Building boost')
        self._wget(self.BOOST_URL, self.DEPS_DIR)
        self._unpack(os.path.join(self.DEPS_DIR, self.BOOST_TAR), self.DEPS_DIR)
        boost_dir = os.path.join(self.DEPS_DIR, self.BOOST_FLDR)
        boost_libs = os.path.join(boost_dir, "stage/lib/*.a")
        run_process(['./bootstrap.sh', '--with-toolset=gcc'], work_dir=boost_dir, log_prefix='bootstrap_boost')
        boost_buildflags = ['./b2', '--prefix=/', '--build-type=minimal', '-s', 'NO_BZIP2=1', 'variant=release', 'link=static',
            'threading=multi', 'runtime-link=static']
        run_process(boost_buildflags, work_dir=boost_dir, log_prefix='b2_boost')
        for lib in glob.glob(boost_libs):
            os.rename(lib, os.path.join(self.LIB_DIR, os.path.join(os.path.basename(lib))))
        shutil.copytree(os.path.join(boost_dir, "boost"), os.path.join(self.INCLUDE_DIR, "boost"))

    def _build_static_check(self, check_ver):
        logging.info('Building check')
        self._wget(self.CHECK_URL, self.DEPS_DIR)
        self._unpack(os.path.join(self.DEPS_DIR, self.CHECK_TAR), self.DEPS_DIR)
        check_dir = os.path.join(self.DEPS_DIR, self.CHECK_FLDR)
        run_process(['/usr/bin/autoreconf', '-fvi'], work_dir=check_dir, log_prefix='autoreconf_check')
        comment_patterns = ('PKG_CHECK_EXISTS(libsubunit, :, enable_subunit=false)', 'PKG_CHECK_MODULES(LIBSUBUNIT, libsubunit)')
        for c in comment_patterns:
            run_process(['/bin/sed', '-i', 's/{}/#{}/g'.format(c, c), 'configure'], work_dir=check_dir, log_prefix='insane_patching')
        run_process(['/bin/sed', '-i', 's/makeinfo/true/g'.format(c, c), 'configure'], work_dir=check_dir, log_prefix='insane_patching_v2')
        run_process(['./configure', '--prefix=/', '--disable-shared', '--enable-static', '--disable-subunit'], work_dir=check_dir, log_prefix='configure_check')
        run_process(['make'], work_dir=check_dir, log_prefix='make_check')
        run_process(['make', self.AUTOMAKE_DSTDIR, 'install'], work_dir=check_dir, log_prefix='make_install_check')

    def _build_scons(self, scons_ver):
        logging.info('Building scons')
        self._wget(self.SCONS_URL, self.DEPS_DIR)
        self._unpack(os.path.join(self.DEPS_DIR, self.SCONS_TAR), self.DEPS_DIR)
        scons_dir = os.path.join(self.DEPS_DIR, self.SCONS_FLDR)
        scons_buildflags = ['/skynet/python/bin/python', 'setup.py', 'install', '--standard-lib', '--prefix={}'.format(self.INSTALL_DIR),
            '--install-data={}'.format(self.INSTALL_DIR), '--optimize=1', '--root=/']
        run_process(scons_buildflags, work_dir=scons_dir, log_prefix='install_scons')
        os.mkdir(os.path.join(self.LIB_DIR, "python2.7/site-packages/scons"))
        os.rename(os.path.join(self.LIB_DIR, "python2.7/site-packages/SCons"),
                os.path.join(self.LIB_DIR, "python2.7/site-packages/scons/SCons"))

    def _build_bison(self, bison_ver):
        logging.info('Building bison')
        self._wget(self.BISON_URL, self.DEPS_DIR)
        self._unpack(os.path.join(self.DEPS_DIR, self.BISON_TAR), self.DEPS_DIR)
        bison_buildflags = ['./configure', '--prefix=/']
        run_process(bison_buildflags, work_dir=self.BISON_DIR, log_prefix='configure_bison')
        run_process(['make'], work_dir=self.BISON_DIR, log_prefix='make_bison')
        run_process(['make', self.AUTOMAKE_DSTDIR, 'install'], work_dir=self.BISON_DIR, log_prefix='make_install_bison')

    def _build_static_zlib(self, zlib_ver):
        logging.info('Building zlib')
        self._wget(self.ZLIB_URL, self.DEPS_DIR)
        self._unpack(os.path.join(self.DEPS_DIR, self.ZLIB_TAR), self.DEPS_DIR)
        zlib_dir = os.path.join(self.DEPS_DIR, self.ZLIB_FLDR)
        zlib_buildflags = ['./configure', '--prefix=/', '--static']
        run_process(zlib_buildflags, work_dir=zlib_dir, log_prefix='configure_zlib')
        run_process(['make', 'CFLAGS=-fPIC -O3 -D_LARGEFILE64_SOURCE=1 -DHAVE_HIDDEN'], work_dir=zlib_dir, log_prefix='make_zlib')
        run_process(['make', self.AUTOMAKE_DSTDIR, 'install'], work_dir=zlib_dir, log_prefix='make_install_zlib')

    def _build_static_openssl(self, openssl_ver):
        logging.info('Building openssl')
        self._wget(self.OPENSSL_URL, self.DEPS_DIR)
        self._unpack(os.path.join(self.DEPS_DIR, self.OPENSSL_TAR), self.DEPS_DIR)
        openssl_buildflags = ['/usr/bin/perl', './Configure', '--prefix=/', 'no-shared', 'no-ssl3-method',
            'enable-ec_nistp_64_gcc_128', 'linux-x86_64', '-Wa,--noexecstack', '-fPIC',
            '-I{}'.format(self.INCLUDE_DIR), '-L{}'.format(self.LIB_DIR)]
        run_process(openssl_buildflags, work_dir=self.OPENSSL_DIR, log_prefix='configure_openssl')
        run_process(['make'], work_dir=self.OPENSSL_DIR, log_prefix='make_openssl')
        run_process(['make', self.OPENSSL_DSTDIR, 'LIBDIR=lib', 'install'], work_dir=self.OPENSSL_DIR, log_prefix='make_install_openssl')

    def _checkout_sources(self, url, tag, dir_name):
        logging.info('Checking out %s, tag={} to {}', url, tag, dir_name)
        run_process(['git', 'clone', url, dir_name], log_prefix='git_clone')
        run_process(['git', 'checkout', tag], work_dir=dir_name, log_prefix='git_checkout')

    def _prepare(self):
        logging.info('Preparing sources')
        os.mkdir(self.PERCONA_DIR)
        self._checkout_sources(self.PERCONA_URL, self.PERCONA_TAG, self.PERCONA_DIR)
        if not os.path.isdir(self.GALERA_DIR):
            os.mkdir(self.GALERA_DIR)
        self._checkout_sources(self.GALERA_URL, self.GALERA_TAG, self.GALERA_DIR)

    def _build_precona(self):
        logging.info('Building percona')
        os.environ['BISON_PKGDATADIR'] = '{}/share/bison'.format(self.INSTALL_DIR)
        CFLAGS = "-static-libgcc -static-libstdc++"
        cmake = ['/usr/bin/cmake', '.', '-DCMAKE_CXX_FLAGS={}'.format(CFLAGS), '-DCMAKE_C_FLAGS={}'.format(CFLAGS),
            '-DENABLED_PROFILING=OFF', '-DENABLE_DEBUG_SYNC=OFF', '-DENABLE_GCOV=OFF', '-DINSTALL_LAYOUT=STANDALONE',
            '-DWITH_PIC=ON', '-DBISON_EXECUTABLE={}'.format(self.BISON_BIN), '-DWITH_ZLIB=bundled', '-DWITH_SSL={}'.format(self.INSTALL_DIR),
            '-DCMAKE_INSTALL_PREFIX={}'.format(self.RESULT_DIR), '-DCURSES_INCLUDE_PATH={}'.format(self.INCLUDE_DIR), '-DCURSES_LIBRARY={}'.format(self.NCURSES_STATIC),
            '-DWITH_BOOST={}'.format(self.INCLUDE_DIR), '-DWITH_EDITLINE=bundled']
        run_process(cmake, work_dir=self.PERCONA_DIR, log_prefix='cmake_precona')
        run_process(['make'], work_dir=self.PERCONA_DIR, log_prefix='make_precona')
        logging.info('make install to {}'.format(self.RESULT_DIR))
        run_process(['make', 'install'], work_dir=self.PERCONA_DIR, log_prefix='make_install_precona')
        shutil.rmtree(os.path.join(self.RESULT_DIR, "mysql-test"))

    def _build_galera(self):
        logging.info('Building galera')
        CFLAGS = "-static-libgcc -static-libstdc++ -I{}".format(self.INCLUDE_DIR)
        LDFLAGS = " -static-libgcc -static-libstdc++"
        os.environ['CFLAGS'] = CFLAGS
        os.environ['CXXFLAGS'] = CFLAGS
        os.environ['LIBPATH'] = self.LIB_DIR
        os.environ['LDFLAGS'] = LDFLAGS
        os.environ['PYTHONUSERBASE'] = os.path.join(self.LIB_DIR, "python2.7")
        galera_rev = ""
        garbd = 'garb/garbd'
        libgalera = 'libgalera_smm.so'
        with open(os.path.join(self.GALERA_DIR, 'GALERA-REVISION')) as revfile:
            galera_rev = revfile.read().rstrip('\n')
        scons = [self.SCONS_BINARY, '--config=force', 'static_ssl=1', 'with_ssl={}'.format(self.LIB_DIR), 'revno={}'.format(galera_rev),
            'strict_build_flags=0', garbd, libgalera]
        run_process(scons, work_dir=self.GALERA_DIR, log_prefix="make_galera")
        shutil.copy(os.path.join(self.GALERA_DIR, garbd), os.path.join(self.RESULT_DIR, "bin"))
        shutil.copy(os.path.join(self.GALERA_DIR, libgalera), os.path.join(self.RESULT_DIR, "lib"))


__Task__ = BuildPerconaXtradbCluster
