import glob
import os.path

import sandbox.projects.common.gnupg
import sandbox.projects.common.debpkg
from sandbox import sandboxsdk

import sandbox.common.types.client as ctc
from sandbox.projects import resource_types
from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask
from sandbox.projects.common.arcadia import sdk
import sandbox.projects.common.constants as consts
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.paths import copy_path, make_folder
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.svn import Arcadia, ArcadiaTestData


def copy2package(build_dir, package_dir, file_path, file_name_tail):
    copy_path(
        os.path.join(build_dir, file_path),
        os.path.join(package_dir, "%s%s" % (file_path.split("/")[-1], file_name_tail))
    )


class BuildPersTools(ArcadiaTask):

    """Builds PERS_TOOLS binaries
       Creates and uploads Debian package"""

    type = "BUILD_PERS_TOOLS"

    class RepositoryId(parameters.SandboxStringParameter):
        name = "repository_id"
        description = "Repository ID"
        default_value = "search"

    input_parameters = ArcadiaTask.input_parameters + [RepositoryId]

    DUPLOAD_CONFIG = {
        "search": {
            "fqdn": "search.dupload.dist.yandex.ru",
            "method": "scpb",
            "login": "robot-pers-releaser",
            "incoming": "/repo/search/mini-dinstall/incoming/",
            "dinstall_runs": 1,
        }
    }

    client_tags = ctc.Tag.Group.LINUX

    def do_execute(self):
        # prepare build environment
        build_dir = self.abs_path("build")
        make_folder(build_dir)
        os.chdir(build_dir)

        arcadia_dir = self.get_arcadia_src_dir()
        pers_path = "quality/pers"
        s2j_path = "rtmapreduce/mrtasks/sessions2json"
        rearrange_data_path = "search/web/rearrs_upper/rearrange/personalization"

        # build binaries
        sdk.do_build(
            consts.YMAKE_BUILD_SYSTEM, arcadia_dir, [pers_path, s2j_path, rearrange_data_path],
            consts.RELEASE_BUILD_TYPE, clear_build=True, results_dir=build_dir
        )

        url = self.ctx[consts.ARCADIA_URL_KEY]
        parsed_url = Arcadia.parse_url(url)
        test_data_path = parsed_url.path.rstrip("/")
        assert test_data_path.endswith("arcadia")
        assert parsed_url.revision is not None
        arcadia_tests_data_path = test_data_path[:-8] + "/arcadia_tests_data"

        pkg_version = str(parsed_url.revision)
        file_name_tail = ".linux.%s" % pkg_version

        # create Debian package
        if self.ctx["repository_id"]:
            do_upload = True
        else:
            do_upload = False

        package_dir = os.path.join(build_dir, "package")
        make_folder(package_dir)
        os.chdir(package_dir)

        # copy package contents to package_dir
        copy_path(
            os.path.join(arcadia_dir, "packages/debian/quality/pers/tools/debian"),
            os.path.join(package_dir, "debian")
        )

        copy2package(build_dir, package_dir, "quality/pers/fcalc/pers-fcalc", file_name_tail)
        copy2package(build_dir, package_dir, "quality/pers/user_profiles/get_user_profiles", file_name_tail)
        copy2package(build_dir, package_dir, "quality/pers/misctools/filtertable/filtertable", file_name_tail)
        copy2package(build_dir, package_dir, "quality/pers/misctools/mrsupplement/mrsupplement", file_name_tail)
        copy2package(build_dir, package_dir, "quality/pers/misctools/sessionspath/sessionspath", file_name_tail)
        copy2package(build_dir, package_dir, "rtmapreduce/mrtasks/sessions2json/sessions2json", file_name_tail)
        copy2package(arcadia_dir, package_dir, "quality/pers/fcalc/calc.py", file_name_tail)
        copy2package(arcadia_dir, package_dir, "quality/pers/fcalc/up_confs", file_name_tail)
        copy2package(arcadia_dir, package_dir, "yweb/urlrules/areas.lst", file_name_tail)

        arcadia_tests_data_dir = ArcadiaTestData.get_arcadia_test_data(self, Arcadia.replace(url, path=arcadia_tests_data_path + "/pers/get_user_profiles/"))
        copy2package(arcadia_tests_data_dir, package_dir, "airports_geodata.txt", file_name_tail)
        copy2package(arcadia_tests_data_dir, package_dir, "blockstat.dict", file_name_tail)
        copy2package(arcadia_tests_data_dir, package_dir, "city_geodata.txt", file_name_tail)

        copy2package(build_dir, package_dir, rearrange_data_path + "/data/wizard_url_mapping.txt", file_name_tail)

        pkg_name = "pers%s" % file_name_tail
        # run_process("dch -v %s -b -D unstable --force-distribution %s" % (pkg_version, pkg_name), shell = True, log_prefix = "dch")

        # debuild_cmd = "debuild --no-tgz-check"
        # if not do_upload:
        #     debuild_cmd += " -i -us -uc -b"
        # run_process(debuild_cmd, shell = True, log_prefix = "debuild")
        environment = os.environ.copy()
        environment["DEBEMAIL"] = "Robot Pers Releaser <robot-pers-releaser@yandex-team.ru>"
        with sandbox.projects.common.gnupg.GpgKey(self, "qkrorlqr", "robot-pers-releaser-gpg-private", "robot-pers-releaser-gpg-public"):
            # run_process("dch -a %s" % unicode(self.ctx["release_comments"]), wait=True, check=False, log_prefix="dch", environment=environment)
            run_process("dch -v %s -b -D unstable --force-distribution %s" % (pkg_version, pkg_name), wait=True, check=False, log_prefix="dch", environment=environment)
            run_process("debuild -krobot-pers-releaser@yandex-team.ru", wait=True, check=True, log_prefix="debuild")

        debian_pkg = glob.glob(os.path.join(build_dir, "*.deb"))[-1]

        # upload to repository
        if do_upload:
            with sandbox.projects.common.debpkg.DebRelease(BuildPersTools.DUPLOAD_CONFIG) as deb:
                with sandboxsdk.ssh.Key(self, "qkrorlqr", "robot-pers-releaser-ssh"):
                    deb.debrelease(["--to", self.ctx["repository_id"]])

        # save created packages as resources
        resource_dir = self.abs_path()
        copy_path(debian_pkg, resource_dir)

        debian_name = os.path.basename(debian_pkg)

        self.create_resource(
            description=debian_name,
            resource_path=os.path.join(resource_dir, debian_name),
            resource_type=resource_types.PERS_TOOLS_DEB,
            arch=self.arch
        )


__Task__ = BuildPersTools
