# -*- coding: utf-8 -*-

import logging
import os

import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc

from sandbox.projects.resource_types import PROJECT_STUB_NODEJS_PACKAGE
from sandbox.projects.resource_types import PROJECT_STUB_NODEJS_ARTIFACT

from sandbox.projects.common.BaseBuildPythonBundleTask import GitCommit
from sandbox.projects.common.BaseBuildPythonBundleTask import GitLocalRepository
from sandbox.projects.common.BaseBuildPythonBundleTask import Utils

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk import parameters

import sandbox.sandboxsdk.process as sdk_process


class CommandToRun(parameters.SandboxStringParameter):
    """
    Command to run after cloning repository
    """
    name = 'cmd'
    description = 'Command to run after cloning repository'
    default_value = 'build.sh'
    multiline = True
    required = True


class EnvironmentContainer(parameters.Container):
    description = 'Environment container resource:'
    default_value = 149550728  # https://sandbox.yandex-team.ru/resource/146073718/view
    required = True


class CustomData(parameters.DictRepeater, parameters.SandboxStringParameter):
    name = 'custom_data'
    description = 'Dict of strings with custom data'
    required = False


class Environment(parameters.DictRepeater, parameters.SandboxStringParameter):
    name = 'env'
    description = 'Dict with environment vars'
    required = False


class ArtifactMap(parameters.DictRepeater, parameters.SandboxStringParameter):
    name = 'artifact_map'
    description = 'Dict with mapping for artifacts\n(all files will be packed into tarballs)'
    required = False


class BuildProjectStubNodejs(SandboxTask):
    """
    Based on BaseBuildPythonBundleTask

    Basic task for Nodejs code deployment preparation. It can get the code from git repository, pack it to archive and
    and export as resource. Besides it can build a virtual environment for code execution and export it in separate
    resource.

    To use this code you should create your own task class inherited from this one and define some required parameters
    (class variables). Class variables are used to change final task behaviour.
    List of required and optional parameters (variables) and their meaning is listed below.

    Task also has some hook methods. You can redefine them to add some custom steps and logic to final task.
    Hooks are named using next template:
        <build step>_<before|after>_<step part>
    for example:
        code_before_archivation will be called before code archivation process start

    Special hooks before_execution and after_execution are called at the beginning of task execution and after all steps
    done.

    Required parameters (class variables):
        REPO_URL_PARAMETER      git repository URL parameter. Will be used to clone python code.

        REPO_BRANCH_PARAMETER   checkout branch parameter. This branch will be cloned during task execution. I think,
                                you may need to replace this parameter with your own class only when you want to change
                                default value.

        COMMAND_PARAMETER       command to run after cloning repository to running test or build project

    Optional parameters (class variables):
        RESOURCE_TYPE_CODE      code archive resource type class. When it is not None, task will pack source code to
                                archive and export it in a resource of given type.

        CODE_NAME               code archive name part. Also used as part of resource descriptions.

        TEMP_DIR                temporary storage of files, used at task execution time. You will hardly want to change
                                this, I think.
        TEMP_CODE_PATH          temporary directory used for storing your code before archiving (checkout to this path)
    """

    type = 'BUILD_PROJECT_STUB_NODEJS_PACKAGE'
    client_tags = ctc.Tag.LINUX_TRUSTY
    owners = ['dench']
    dns = ctm.DnsType.DNS64
    execution_space = 1024
    privileged = True

    REPO_URL_PARAMETER = parameters.SandboxGitUrlParameter
    REPO_BRANCH_PARAMETER = GitCommit
    COMMAND_PARAMETER = CommandToRun
    ENVIRONMENT_PARAMETER = Environment
    ARTIFACT_MAP_PARAMETER = ArtifactMap

    RESOURCE_TYPE_CODE = PROJECT_STUB_NODEJS_PACKAGE
    ARTIFACT_RESOURCE_TYPE_CODE = PROJECT_STUB_NODEJS_ARTIFACT

    CODE_NAME = 'nodejs_code'

    TEMP_DIR = 'tmp/'
    TEMP_CODE_PATH = os.path.join(TEMP_DIR, CODE_NAME)

    input_parameters = (
        parameters.SandboxGitUrlParameter,
        GitCommit,
        CommandToRun,
        EnvironmentContainer,
        CustomData,
        Environment,
        ArtifactMap
    )

    def on_execute(self):
        """
        What should it do:
            * checkout repository with specific tag/branch/commit
            * pack and export archive with code as a resource (optional)
        """

        # Get code from repository. Creates 'self.git' attribute.
        self._get_code()

        # Generate common attributes list
        common_attributes = self.gen_common_attributes()

        # Hook: before common execution steps
        self.before_execution(common_attributes=common_attributes)

        # Run command from context
        logging.debug('Running command:\n{cmd}\nAt work dir: {path}'.format(
                path=self.TEMP_CODE_PATH,
                cmd=self.ctx.get(self.COMMAND_PARAMETER.name)
            ))

        # Add custom.sh and execute it
        self.execute_sh_file_with_command(self.ctx.get(self.COMMAND_PARAMETER.name))

        # Create sandbox resource from cloned folder after build
        self.create_nodejs_code_resource(self.git.local_path, common_attributes)

        # Create sandbox resource with artifacts
        if self.ctx.get(self.ARTIFACT_MAP_PARAMETER.name):
            self.create_artifacts_resources(self.git.local_path)

        # Hook: after common execution steps
        self.after_execution(common_attributes=common_attributes)

    def execute_sh_file_with_command(self, content):
        path = '/custom.sh'

        f = open(path, "w")
        f.writelines(content)
        f.close()

        self.execute('/bin/bash {path}'.format(path=path))
        self.execute('/bin/rm {path}'.format(path=path))

    def execute(self, command_line):
        command_line = "sh -c '" + command_line + "'"
        env = self.ctx.get(self.ENVIRONMENT_PARAMETER.name)

        execute_custom_env = os.environ.copy()
        execute_custom_env.update(env)

        sdk_process.run_process(
            cmd=command_line,
            log_prefix='cmd-run',
            work_dir=self.TEMP_CODE_PATH,
            environment=execute_custom_env,
            wait=True
        )

    def _get_code(self):
        """
        Get project code from remote source using Git.
        Set 'self.git' attribute for further usage.
        """
        git_url = self.ctx[self.REPO_URL_PARAMETER.name]
        git_branch = self.ctx[self.REPO_BRANCH_PARAMETER.name]
        local_code_path = self.TEMP_CODE_PATH

        self.git = GitLocalRepository(remote_source=git_url, local_path=local_code_path)

        # Clone repository
        self.git.clone(git_branch)

    def gen_common_attributes(self):
        """
        Generate common resources attributes list using task's runtime context and current repository state.
        When local repository copy exists, 'commit_hash' and 'branch' or 'tag' attributes are generated.
        Custom attributes are used to update common attributes list, so you can recover any common attribute from
        'CustomResourceAttributes' task parameter.

        :param repo_path: path to local git repository copy. This copy's state is used to generate attributes.
                          Generated attributes:
                            'commit_hash' (always)
                            'tag' (optional, depends on HEAD)
                            'branch' (optional, depends on HEAD)

                          Current repository HEAD contents results in:
                            * 'tag' attribute generation when HEAD is a symbolic reference to tag.
                            * 'branch' attribute generation when HEAD is a symbolic reference to branch.
        """

        common_attributes = {}

        common_attributes["repository"] = self.git.remote_source

        if os.path.exists(self.git.local_path):
            common_attributes["commit_hash"] = self.git.current_commit_hash

            if self.git.is_tag():
                common_attributes["tag"] = self.git.commit
            elif self.git.is_branch():
                common_attributes["branch"] = self.git.commit

        return common_attributes

    def create_nodejs_code_resource(self, code_path, common_attributes):
        """
        Create NodeJS code archive and export it as a resource
        """

        # Resource creation
        code_archive_name = '{}.tar.gz'.format(self.CODE_NAME)

        resource_name = 'Архив с кодом NodeJS ({})'.format(self.CODE_NAME)

        self.code_resource = self.create_resource(resource_name, code_archive_name,
                                                  self.RESOURCE_TYPE_CODE,
                                                  attributes=common_attributes)

        # Hook
        self.code_before_archivation(common_attributes=common_attributes)

        # Archivation
        Utils.pack_archive(code_path, self.code_resource.path)

        # Hook
        self.code_after_archivation(common_attributes=common_attributes)

    def create_artifacts_resources(self, code_path):
        """
        Create archives with artifacts and export it as a resource
        """

        for source_path, target in self.ctx.get(self.ARTIFACT_MAP_PARAMETER.name).iteritems():
            code_archive_name = '{}.tar.gz'.format(target)

            artifact_resource = self.create_resource(target, code_archive_name,
                                 self.ARTIFACT_RESOURCE_TYPE_CODE,
                                 attributes={'original_name': source_path})

            artifact_source_path = os.path.join(code_path, source_path)

            # Archivation
            Utils.pack_archive(artifact_source_path, artifact_resource.path)

    def arcadia_info(self):
        """
        Hacky way to allow this task to be released: provide tag, other fields are not checked.
        """
        return None, self.ctx[self.REPO_BRANCH_PARAMETER.name], None

    def before_execution(self, common_attributes):
        """
        This method is called after common task configuration (repository cloning, common attributes list generation,
        etc.) and before any real work (virtual environment generation, code archiving, etc).

        If you want to make some actions here, just redefine method in child class.

        :param common_attributes: list of common attributes that should be set for all task resources.
        :type common_attributes: dict
        """
        pass

    def after_execution(self, common_attributes):
        """
        This method is called when all real work is already done (virtual environment hed been generated, code archive
        had been created, etc).

        If you want to make some actions here, just redefine method in child class.

        :param common_attributes: list of common attributes that should be set for all task resources.
        :type common_attributes: dict
        """
        pass

    def code_before_archivation(self, common_attributes):
        """
        This method is called before code archivation process start.

        If you want to make some actions here, just redefine method in child class.

        :param common_attributes: list of common attributes that should be set for all task resources.
        :type common_attributes: dict
        """
        pass

    def code_after_archivation(self, common_attributes):
        """
        This method is called after code archivation process end.

        If you want to make some actions here, just redefine method in child class.

        :param common_attributes: list of common attributes that should be set for all task resources.
        :type common_attributes: dict
        """
        pass


__Task__ = BuildProjectStubNodejs
