# -*- coding: utf-8 -*-
import os
import logging
import tarfile
import shutil

from sandbox.projects.resource_types import PYTHON_SOURCE, PYTHON_BUNDLE, PYTHON_EXTERNAL_LIBRARIES
from sandbox.projects.common.nanny import nanny

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxGitUrlParameter, LastReleasedResource
from sandbox.sandboxsdk.sandboxapi import Sandbox
import sandbox.common.types.client as ctc


class GitUrl(SandboxGitUrlParameter):
    name = 'git_url'
    default_value = 'what'
    required = True


class GitBranch(SandboxStringParameter):
    name = 'git_branch'
    description = 'Git branch:'
    default_value = 'master'
    required = True


class TargetVirtualenvName(SandboxStringParameter):
    name = 'target_virtualenv_name'
    description = 'Target virtualenv name (build from source only):'
    default_value = 'env'
    required = True


class TargetRequirements(SandboxStringParameter):
    name = 'target_requirements'
    description = 'Target requirements filename:'
    default_value = 'requirements.txt'
    required = True


class PythonSource(LastReleasedResource):
    name = 'python_src'
    description = 'Compressed Python source:'
    resource_type = PYTHON_SOURCE
    required = True


class ExternalLibraries(LastReleasedResource):
    name = 'python_ext_lib'
    description = 'External libraries:'
    resource_type = PYTHON_EXTERNAL_LIBRARIES
    required = False


class BuildId(SandboxStringParameter):
    name = 'build'
    description = 'Build id:'
    default_value = ''
    required = False


class BuildPythonBundle(SandboxTask, nanny.ReleaseToNannyTask):
    """
        Build python bundle - relocatable virtualenv with installed target.
        Virtualenv can be built either from python source or from existing virtualenv with 'virtualenv' module.
    """
    type = 'BUILD_PYTHON_BUNDLE'
    client_tags = ctc.Tag.GENERIC & ~ctc.Tag.LXC

    input_parameters = (
        GitUrl,
        GitBranch,
        TargetRequirements,
        TargetVirtualenvName,
        PythonSource,
        ExternalLibraries,
        BuildId
    )
    exec_context = {}

    def on_execute(self):
        logging.info('==> checking out target source from {}'.format(self.ctx['git_url']))
        target_src = self.path('target_src')
        self.exec_context['target_src'] = target_src
        run_process(
            ['git', 'clone', self.ctx['git_url'], target_src],
            log_prefix='git', check=True, wait=True, environment={}
        )

        if self.ctx['git_branch'].startswith('refs/heads'):
            # Need to fetch ref first.
            actual_branch_name = self.ctx['git_branch'][11:]
            run_process(
                ['git', 'checkout', actual_branch_name],
                log_prefix='git', check=True, wait=True, work_dir=target_src, environment={}
            )
            run_process(
                ['git', 'fetch'],
                log_prefix='git', check=True, wait=True, work_dir=target_src, environment={}
            )

        run_process(
            ['git', 'checkout', self.ctx['git_branch']],
            log_prefix='git', check=True, wait=True, work_dir=target_src, environment={}
        )

        logging.info('==> looking for last commit')
        self.exec_context['release_commits'] = []
        try:
            out, _ = run_process(
                ['git', 'log', '-1'],
                check=True, wait=True, work_dir=target_src, environment={}, outs_to_pipe=True
            ).communicate()
            self.ctx['last_commit'] = out.decode('utf-8').split('\n')[0].split()[1]
            self.ctx['revision'] = '{}@{}'.format(self.ctx['git_branch'], self.ctx['last_commit'])
            logging.info('==> last commit is {}'.format(self.ctx['last_commit']))

            logging.info('==> looking for previous release')
            releases = Sandbox().list_releases(PYTHON_BUNDLE, limit=1)
            if releases:
                last_release = releases[0]
                last_released_bundle = [r for r in last_release.resources if r.type == PYTHON_BUNDLE][0]
                logging.info('==> last released bundle: {}'.format(last_released_bundle))

                last_released_commit = last_released_bundle.attributes.get('commit')
                if last_released_commit:
                    out, err = run_process(
                        ['git', 'rev-list', '--pretty', '{}..HEAD'.format(last_released_commit)],
                        check=True, wait=True, work_dir=target_src, environment={}, outs_to_pipe=True
                    ).communicate()
                    out, err = out.decode('utf-8'), err.decode('utf-8')
                    logging.debug('==> ERR:\n{}'.format(err))
                    logging.debug('==> OUT:\n{}'.format(out))
                    self.ctx['changelog'] = out
                    logging.info('==> changelog:\n{}'.format(self.ctx['changelog']))

        except Exception as e:
            logging.exception('failed to build changelog', exc_info=e)

        logging.info('==> BUILDING PYTHON FROM SOURCE')

        logging.info('==> downloading python source')
        compressed_python_src = self.sync_resource(self.ctx['python_src'])

        python_src = self.path('python-source')
        logging.info('==> decompressing python source to {}'.format(python_src))
        with tarfile.open(compressed_python_src) as t:
            t.extractall(path=python_src)
            source_prefix = t.getnames()[0]
            python_version = source_prefix.split('-')[1][:3]
            python_src = os.path.join(python_src, source_prefix)

        ext_lib_path = None
        if self.ctx['python_ext_lib']:
            # Download and unpack external libraries.
            logging.info('==> downloading external libraries')
            compressed_ext_lib = self.sync_resource(self.ctx['python_ext_lib'])

            ext_lib_path = self.path('ext_lib')
            logging.info('==> unpacking external libraries')
            with tarfile.open(compressed_ext_lib) as t:
                t.extractall(path=ext_lib_path)

            # Dirty hack for external libsqlite3.
            with open(os.path.join(python_src, 'setup.py'), 'r') as f:
                setup_py = f.read().split('\n')

            last_include_lineno = setup_py.index([l for l in setup_py if '/usr/local/include/sqlite3' in l][0])
            new_setup_py = setup_py[:last_include_lineno]
            spaces_before_line = setup_py[last_include_lineno].count(' ')
            new_setup_py.append("{}'{}',".format(
                ' ' * spaces_before_line,
                os.path.join(ext_lib_path, 'include')
            ))
            new_setup_py += setup_py[last_include_lineno + 1:]

            with open(os.path.join(python_src, 'setup.py'), 'w') as f:
                f.write('\n'.join(new_setup_py))

        python_dist = self.path('python-dist')
        logging.info('==> configuring python')
        run_process(
            [
                './configure',
                '--enable-shared',
                '--prefix={}'.format(python_dist)
            ],
            log_prefix='python_configure', check=True, wait=True, work_dir=python_src, environment={}
        )

        logging.info('==> building python')
        run_process(
            ['make'],
            log_prefix='python_make', check=True, wait=True, work_dir=python_src, environment={}
        )

        logging.info('==> installing python to {}'.format(python_dist))
        run_process(
            ['make', 'install'],
            log_prefix='python_install', check=True, wait=True, work_dir=python_src, environment={}
        )
        python_bin = os.path.join(python_dist, 'bin', 'python{}'.format(python_version))
        assert os.path.isfile(python_bin)

        source_lib_content = os.listdir(os.path.join(python_dist, 'lib'))
        logging.debug('==> lib from source:\n{}'.format(
            '\n'.join(source_lib_content)
        ))

        logging.info('==> installing virtualenv')
        run_process(
            [
                python_bin, '-m', 'pip', 'install',
                '--only-binary', 'all', '-f', os.path.join(target_src, 'wheel'), 'virtualenv'
            ],
            log_prefix='pip_install_virtualenv', check=True, wait=True, environment={
                'PATH': '{}:{}'.format(os.path.join(python_dist), os.environ['PATH']),
                'LD_LIBRARY_PATH': os.path.join(python_dist, 'lib')
            }
        )

        target_env = self.path(self.ctx['target_virtualenv_name'])
        self.exec_context['target_env'] = target_env
        python_bin = os.path.join(python_dist, 'bin', 'python{}'.format(python_version))

        logging.info('==> using python at {}'.format(python_bin))
        assert os.path.isfile(python_bin)

        logging.info('==> creating new virtualenv at {}'.format(target_env))
        run_process(
            [
                python_bin, '-m', 'virtualenv', '--always-copy', '-p', python_bin, target_env
            ],
            log_prefix='virtualenv_create_new', check=True, wait=True,
            environment={
                'PATH': '{}:{}'.format(os.path.join(python_dist), os.environ['PATH']),
                'LD_LIBRARY_PATH': os.path.join(python_dist, 'lib')
            }
        )

        # Even with --always-copy virtualenv may leave some symlinks.
        existing_lib = os.path.join(target_env, 'lib')
        logging.info('==> deleting existing lib: {}'.format(existing_lib))
        shutil.rmtree(existing_lib)

        logging.info('==> copying lib from {}'.format(python_dist, 'lib'))
        shutil.copytree(
            os.path.join(python_dist, 'lib'),
            os.path.join(target_env, 'lib')
        )

        if self.ctx['python_ext_lib']:
            logging.info('==> copying external libraries from {}'.format(ext_lib_path))
            for i in os.listdir(os.path.join(ext_lib_path, 'lib')):
                current_lib_path = os.path.join(ext_lib_path, 'lib', i)
                if os.path.isfile(current_lib_path):
                    logging.info('\t-> {}'.format(i))
                    shutil.copy2(current_lib_path, os.path.join(target_env, 'lib'))
                else:
                    logging.info('\t-> {} (skipped)'.format(i))

        logging.info('==> BUILDING TARGET')

        target_python_bin = os.path.join(target_env, 'bin', 'python')
        self.exec_context['target_python_bin'] = target_python_bin
        assert os.path.isfile(target_python_bin)

        target_lib_path = os.path.join(target_env, 'lib')
        self.exec_context['target_lib_path'] = target_lib_path
        assert os.path.isdir(target_lib_path)

        logging.warning('!!TARGET_SRC\n\n{}\n\n'.format(list(os.walk(target_src))))
        logging.warning('!!TARGET_SRC_WHEEL\n\n{}\n\n'.format(list(os.walk(os.path.join(target_src, 'wheel')))))

        # TODO: custom wheels path
        logging.info('==> upgrading pip')
        run_process(
            [
                target_python_bin, '-m', 'pip', 'install',
                '--no-index',
                '--find-links', os.path.join(target_src, 'wheel'),
                '--verbose',
                '--upgrade',
                'pip'
            ],
            log_prefix='pip_upgrade', check=True, wait=True,
            environment={
                'PYTHONPATH': target_lib_path,
                'LD_LIBRARY_PATH': target_lib_path,
            }
        )

        logging.info('==> installing target requirements')
        run_process(
            [
                target_python_bin, '-m', 'pip', 'install',
                '--no-index',
                '--find-links', os.path.join(target_src, 'wheel'),
                '--verbose',
                '-r', os.path.join(target_src, self.ctx['target_requirements'])
            ],
            log_prefix='pip_install_target', check=True, wait=True,
            environment={
                'PYTHONPATH': target_lib_path,
                'LD_LIBRARY_PATH': target_lib_path,
            }
        )

        self.pre_install()

        logging.info('==> installing target')
        run_process(
            [
                target_python_bin, '-m', 'pip', 'install',
                '--no-index',
                '--find-links', os.path.join(target_src, 'wheel'),
                '--verbose',
                target_src
            ],
            log_prefix='pip_install_target', check=True, wait=True,
            environment={
                'PYTHONPATH': target_lib_path,
                'LD_LIBRARY_PATH': target_lib_path,
            }
        )

        logging.info('==> making virtualenv relocatable')
        run_process(
            [target_python_bin, '-m', 'virtualenv', '--relocatable', target_env],
            log_prefix='virtualenv_make_relocatable', check=True, wait=True,
            environment={
                'PYTHONPATH': target_lib_path,
                'LD_LIBRARY_PATH': target_lib_path,
            }
        )

        self.exec_context['target_path'] = target_env

        self.do_create_resource()
        self.post_execute()

        logging.info('==> SUCCESS!')

    def do_create_resource(self):
        target_env = self.exec_context['target_path']
        target_path = self.path('bundle.tar')

        logging.info('==> compressing virtualenv')
        with tarfile.TarFile(target_path, 'w') as t:
            t.add(target_env, self.ctx['target_virtualenv_name'])

        logging.info('==> creating resource: {}'.format('target.tar'))
        self.create_resource(
            description=self.ctx.get(
                'description', 'Python bundle built from {}, rev.{}'.format(self.ctx['git_url'], self.ctx['revision'])
            ),
            resource_path=target_path,
            resource_type=PYTHON_BUNDLE
        )

    def pre_install(self):
        pass

    def post_execute(self):
        pass

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)


__Task__ = BuildPythonBundle
