# -*- coding: utf-8 -*-
import os
import logging
import tarfile

from sandbox.projects.resource_types import PYTHON_BUNDLE, PYTHON_VIRTUALENV
from sandbox.projects.common.nanny import nanny

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.parameters import SandboxBoolParameter, SandboxStringParameter, SandboxGitUrlParameter, LastReleasedResource
from sandbox.sandboxsdk.sandboxapi import Sandbox
import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc


class GitUrl(SandboxGitUrlParameter):
    name = 'git_url'
    default_value = 'what'
    required = True


class GitBranch(SandboxStringParameter):
    name = 'git_branch'
    description = 'Git branch or tag:'
    default_value = 'master'
    required = True


class TargetRequirements(SandboxStringParameter):
    name = 'target_requirements'
    description = 'Target requirements filename:'
    default_value = 'requirements.txt'
    required = True


class PythonVirtualenv(LastReleasedResource):
    name = 'python_venv'
    description = 'Relocatable Python virtualenv:'
    resource_type = PYTHON_VIRTUALENV
    required = True


class UseWheel(SandboxBoolParameter):
    name = 'wheel'
    description = 'Use wheel only'
    default_value = True
    required = True


class BuildId(SandboxStringParameter):
    name = 'build'
    description = 'TeamCity build ID:'
    default_value = ''
    required = False


class BuildPythonBundle2(SandboxTask, nanny.ReleaseToNannyTask):
    """
    Build python bundle - relocatable virtualenv with installed target.
    Virtualenv can be built either from python source or from existing virtualenv with 'virtualenv' module.
    """
    type = 'BUILD_PYTHON_BUNDLE2'
    dns = ctm.DnsType.DNS64

    client_tags = ctc.Tag.GENERIC & ~ctc.Tag.LXC

    input_parameters = (
        GitUrl,
        GitBranch,
        TargetRequirements,
        PythonVirtualenv,
        UseWheel,
        BuildId,
    )

    exec_context = {}

    def checkout_target(self):
        """
        Clones target GIT repository.

        :returns: path to checked out repository.
        """
        logging.info('-> checking out target source from {}'.format(self.ctx['git_url']))
        target_src = self.path('target_src')
        self.exec_context['target_src'] = target_src

        # Workaround for missing read-only access in BitBucket.
        tokenized_git_url = self.ctx['git_url']
        token = self.get_vault_data(self.owner, 'robot_maestro_stash_oauth')
        if tokenized_git_url.startswith('https://'):
            tokenized_git_url = tokenized_git_url.replace('https://', 'https://x-oauth-token:{}@'.format(token))
            logging.warning('git url was tokenized: {}'.format(tokenized_git_url.replace(token, '***')))

        run_process(
            ['git', 'clone', tokenized_git_url, target_src],
            log_prefix='git', check=True, wait=True, environment={}
        )

        if self.ctx['git_branch'].startswith('refs/heads'):
            # Need to fetch ref first.
            actual_branch_name = self.ctx['git_branch'][11:]
            run_process(
                ['git', 'checkout', actual_branch_name],
                log_prefix='git', check=True, wait=True, work_dir=target_src, environment={}
            )
            run_process(
                ['git', 'fetch'],
                log_prefix='git', check=True, wait=True, work_dir=target_src, environment={}
            )

        run_process(
            ['git', 'checkout', self.ctx['git_branch']],
            log_prefix='git', check=True, wait=True, work_dir=target_src, environment={}
        )

        return target_src

    def build_changelog(self, path_to_repo):
        """
        FIXME

        :param path_to_repo:
        :return:
        """
        logging.info('-> looking for last commit')
        self.exec_context['release_commits'] = []

        out, _ = run_process(
            ['git', 'log', '-1'],
            check=True, wait=True, work_dir=path_to_repo, environment={}, outs_to_pipe=True
        ).communicate()
        self.ctx['last_commit'] = out.decode('utf-8').split('\n')[0].split()[1]
        self.ctx['revision'] = '{}@{}'.format(self.ctx['git_branch'], self.ctx['last_commit'])
        logging.info('-> last commit is {}'.format(self.ctx['last_commit']))

        logging.info('-> looking for previous release')
        releases = Sandbox().list_releases(PYTHON_BUNDLE, limit=1)
        if releases:
            last_release = releases[0]
            last_released_bundle = [r for r in last_release.resources if r.type == PYTHON_BUNDLE][0]
            logging.info('-> last released bundle: {}'.format(last_released_bundle))

            last_released_commit = last_released_bundle.attributes.get('commit')
            if last_released_commit:
                out, err = run_process(
                    ['git', 'rev-list', '--pretty', '{}..HEAD'.format(last_released_commit)],
                    check=True, wait=True, work_dir=path_to_repo, environment={}, outs_to_pipe=True
                ).communicate()
                out, err = out.decode('utf-8'), err.decode('utf-8')
                logging.debug('-> ERR:\n{}'.format(err))
                logging.debug('-> OUT:\n{}'.format(out))
                self.ctx['changelog'] = out
                logging.info('-> changelog:\n{}'.format(self.ctx['changelog']))

    def fetch_venv(self):
        """
        FIXME

        :returns: tuple of path to virtualenv and python version
        """
        logging.info('-> downloading python virtualenv')
        compressed_python_venv = self.sync_resource(self.ctx['python_venv'])

        python_venv = self.path('python-venv')
        logging.info('-> decompressing python source to {}'.format(python_venv))
        with tarfile.open(compressed_python_venv) as t:
            t.extractall(path=python_venv)
            names = t.getnames()
            venv_dir_name = names[0]
            version_file_name = [x for x in names if x.startswith('VERSION-')][0]

            python_version = version_file_name.split('-')[1][:3]
            python_venv = os.path.join(python_venv, venv_dir_name)

        return python_venv, python_version

    def on_execute(self):
        target_src = self.checkout_target()

        try:
            self.build_changelog(target_src)
        except Exception as e:
            logging.exception('failed to build changelog', exc_info=e)

        target_env, python_version = self.fetch_venv()

        logging.info('-> BUILDING TARGET')

        target_python_bin = os.path.join(target_env, 'bin', 'python')
        self.exec_context['target_python_bin'] = target_python_bin

        if not os.path.isfile(target_python_bin):
            raise RuntimeError('{} is not a file!'.format(target_python_bin))

        target_lib_path = os.path.join(target_env, 'lib')
        self.exec_context['target_lib_path'] = target_lib_path
        if not os.path.isdir(target_lib_path):
            raise RuntimeError('{} is not a directory!'.format(target_lib_path))

        logging.warning('!!TARGET_SRC\n\n{}\n\n'.format(list(os.walk(target_src))))
        logging.warning('!!TARGET_SRC_WHEEL\n\n{}\n\n'.format(list(os.walk(os.path.join(target_src, 'wheel')))))

        # TODO: custom wheels path
        logging.info('-> installing target requirements')
        if self.ctx['wheel']:
            run_process(
                [
                    target_python_bin, '-m', 'pip', 'install',
                    '--no-index',
                    '--find-links', os.path.join(target_src, 'wheel'),
                    '--verbose',
                    '-r', os.path.join(target_src, self.ctx['target_requirements'])
                ],
                log_prefix='pip_install_target', check=True, wait=True,
                environment={
                    'PYTHONPATH': target_lib_path,
                    'LD_LIBRARY_PATH': target_lib_path,
                }
            )
        else:
            run_process(
                [
                    target_python_bin, '-m', 'pip', 'install',
                    '--find-links', os.path.join(target_src, 'wheel'),
                    '-i', 'https://pypi.yandex-team.ru/simple',
                    '--verbose',
                    '-r', os.path.join(target_src, self.ctx['target_requirements'])
                ],
                log_prefix='pip_install_target', check=True, wait=True,
                environment={
                    'PYTHONPATH': target_lib_path,
                    'LD_LIBRARY_PATH': target_lib_path,
                }
            )

        self.pre_install()

        logging.info('-> installing target')
        if self.ctx['wheel']:
            run_process(
                [
                    target_python_bin, '-m', 'pip', 'install',
                    '--no-index',
                    '--find-links', os.path.join(target_src, 'wheel'),
                    '--verbose',
                    target_src
                ],
                log_prefix='pip_install_target', check=True, wait=True,
                environment={
                    'PYTHONPATH': target_lib_path,
                    'LD_LIBRARY_PATH': target_lib_path,
                }
            )
        else:
            run_process(
                [
                    target_python_bin, '-m', 'pip', 'install',
                    '--find-links', os.path.join(target_src, 'wheel'),
                    '-i', 'https://pypi.yandex-team.ru/simple',
                    '--verbose',
                    target_src
                ],
                log_prefix='pip_install_target', check=True, wait=True,
                environment={
                    'PYTHONPATH': target_lib_path,
                    'LD_LIBRARY_PATH': target_lib_path,
                }
            )

        self.exec_context['target_path'] = target_env

        self.post_install()

        self.do_create_resource()
        self.post_execute()

        logging.info('-> SUCCESS!')

    def do_create_resource(self):
        target_env = self.exec_context['target_path']
        target_path = self.path('bundle.tar')

        logging.info('-> compressing virtualenv')
        with tarfile.TarFile(target_path, 'w') as t:
            # FIXME no more custom names
            t.add(target_env, 'ENV')

        logging.info('-> creating resource: {}'.format('target.tar'))
        self.create_resource(
            description=self.ctx.get(
                'description', 'Python bundle built from {}, rev.{}'.format(self.ctx['git_url'], self.ctx['revision'])
            ),
            resource_path=target_path,
            resource_type=PYTHON_BUNDLE
        )

    def pre_install(self):
        pass

    def post_install(self):
        pass

    def post_execute(self):
        pass

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)


__Task__ = BuildPythonBundle2
