import urllib2
import hashlib
import logging
import os
import traceback
import tarfile
import platform

import sandbox.common.types.client as ctc

from sandbox.projects import resource_types
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.paths import remove_path
from sandbox.sandboxsdk.paths import copy_path
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.errors import SandboxTaskFailureError


PYTHON_DEV_VERSIONS = {
    '2.7.3': ('http://mirror.yandex.ru/ubuntu/pool/main/p/python2.7/python2.7_2.7.3.orig.tar.gz', '20e7d671051f7a8c81d58965efcdf638'),
    '2.7.5': ('http://mirror.yandex.ru/ubuntu/pool/main/p/python2.7/python2.7_2.7.5.orig.tar.gz', '7221db8b6654a9b3d21db0bf02da1e2c'),
    '2.7.6': ('http://mirror.yandex.ru/ubuntu/pool/main/p/python2.7/python2.7_2.7.6.orig.tar.gz', '786610dcd37198ce69f464206efadc63'),
    '2.7.8': ('http://mirror.yandex.ru/ubuntu/pool/main/p/python2.7/python2.7_2.7.8.orig.tar.gz', 'fff1249cfb7c0866cb83afa49c0d71b4'),
    '2.7.9': ('http://mirror.yandex.ru/ubuntu/pool/main/p/python2.7/python2.7_2.7.9.orig.tar.gz', '6c8157ba0f2e705d7ef74593f78e0325'),
    '3.4.3': ('http://mirror.yandex.ru/ubuntu/pool/main/p/python3.4/python3.4_3.4.3.orig.tar.xz', '03536b875a267c68bb7ea686ade1fe89'),
}


class PythonDevVersion(SandboxStringParameter):
    description = 'python-dev version'
    default_value = '2.7.3'
    choices = [(x, x) for x in PYTHON_DEV_VERSIONS.keys()]
    name = 'version'
    required = True


class BuildPythonDev(SandboxTask):
    type = 'BUILD_PYTHON_DEV'

    client_tags = ctc.Tag.Group.LINUX

    input_parameters = [PythonDevVersion]

    execution_space = 150

    BUILD_DIR = 'python-dev'
    TGZ_PATH = '{}.tar.gz'.format(BUILD_DIR)
    COPY_DIRS = {'Include': 'include'}

    def _assemble_bundle(self, copy_dirs, remove_dirs=[]):
        for copy_dir in copy_dirs.keys():
            copy_path(os.path.join(self.FOLDER_NAME, copy_dir), os.path.join(self.BUILD_DIR, copy_dirs[copy_dir]))
        for remove_dir in remove_dirs:
            remove_path(remove_dir)

    def _download(self, name, url, md5_expected):
        logging.info('Downloading %s from %s' % (name, url))
        data = self._wget(url, md5_expected)
        with open(name, 'w') as f:
            f.write(data)
        logging.info('%s downloaded' % name)

    def _get_version(self, version):
        return ".".join(version.split('.')[0:2])

    def _wget(self, url, md5_expected):
        n = 3

        for i in range(0, n):
            logging.info('Downloading %s, try %i' % (url, i + 1))

            try:
                data = urllib2.urlopen(url).read()
                md5_actual = self._calc_md5(data)
                self._compare_md5(md5_actual, md5_expected)
                return data
            except Exception:
                if i == n - 1:
                    raise
                else:
                    logging.error('Exception occured:\n%s' % traceback.format_exc())

    def _compare_md5(self, actual, expected):
        if actual != expected:
            raise Exception('Wrong md5: %s, expected %s' % (actual, expected))

    def _calc_md5(self, data):
        h = hashlib.new('md5')
        h.update(data)
        md5sum = h.hexdigest()
        return md5sum

    def _untar(self, path, result):
        run_process(
            ['tar', 'xf', path],
            log_prefix='tar',
            shell=True)
        if not os.path.exists(result):
            raise Exception('%s not untarred or untarred to different place' % result)

    def _make_resource(self, path):
        logging.info('creating tar.gz file')
        with tarfile.open(self.TGZ_PATH, 'w:gz') as tar:
            tar.add(self.path(path), path)
        self.create_resource(
            description='{} tarball version {}'.format(self.BUILD_DIR, self.ctx['version']),
            resource_path=self.TGZ_PATH,
            resource_type=resource_types.PYTHON_DEV,
            attributes={'ttl': 'inf', 'version': self.ctx['version'], 'platform': platform.platform()},
            arch='linux'
        )

    def arcadia_info(self):
        """
        Hacky way to allow this task to be released: provide tag, other fields are not checked.
        """
        return None, self.ctx.get('version'), None

    def on_execute(self):
        version = self.ctx.get('version')

        if version not in PYTHON_DEV_VERSIONS.keys():
            raise SandboxTaskFailureError("Don't know about python-dev version {version}".format(version=version))

        mm_version = self._get_version(version)

        self.URL, self.MD5 = PYTHON_DEV_VERSIONS[version]
        self.FOLDER_NAME = 'python%s-%s' % (mm_version, version)
        self.ARCHIVE_NAME = 'python%s_%s.orig.tar.gz' % (mm_version, version)

        make_folder(self.path(self.BUILD_DIR))

        self._download(self.ARCHIVE_NAME, self.URL, self.MD5)

        self._untar(self.ARCHIVE_NAME, self.FOLDER_NAME)

        self._assemble_bundle(copy_dirs=self.COPY_DIRS)

        self._make_resource(self.BUILD_DIR)


__Task__ = BuildPythonDev
