# -*- coding: utf-8 -*-

import datetime
import paramiko
import requests
import StringIO

from sandbox import sdk2
import sandbox.sdk2.helpers

import sandbox.common.types.resource as ctr
from sandbox.projects.juggler.resource_types import JUGGLER_CLIENT_BINARY


class _BaseQloudInit(sdk2.Resource):
    releasable = True
    releasers = ['QLOUD-DEV']
    executable = True

    build_version = sdk2.parameters.String("Version")
    build_commit = sdk2.parameters.String("Commit")
    build_date = sdk2.parameters.String("Date")
    build_branch = sdk2.parameters.String("Branch")


class QloudInit(_BaseQloudInit):
    pass


class QloudInitSSH(_BaseQloudInit):
    pass


class QloudInitLogger(_BaseQloudInit):
    pass


class QloudInitTestResults(_BaseQloudInit):
    executable = False


class BuildQloudInit(sdk2.Task):
    """ Task for building qloud-init on QYP agent """

    class Parameters(sdk2.Task.Parameters):
        source_branch = sdk2.parameters.String("git branch", default="master", required=True)
        agent_hostname = sdk2.parameters.String("agent hostname", required=True)
        juggler_client_binary = sdk2.parameters.LastReleasedResource(
            'juggler client to include',
            resource_type=JUGGLER_CLIENT_BINARY,
            state=ctr.State.READY,
            attrs=dict(released='stable'),
        )

    def on_execute(self):
        username = 'root'
        bitbucket_token = sdk2.Vault.data(self.owner, 'qloud_ci_bb_token')

        private_key_object = StringIO.StringIO(sdk2.Vault.data(self.owner, 'qloud_ci_ssh_private_key'))
        ssh_password = sdk2.Vault.data(self.owner, 'qloud_ci_ssh_password')
        private_key = paramiko.RSAKey(file_obj=private_key_object, password=ssh_password)

        with sandbox.sdk2.helpers.ProcessLog(self, logger="remote_build") as pl:
            pl.logger.propagate = 1
            build_remote(hostname=self.Parameters.agent_hostname, username=username, private_key=private_key,
                         ssh_password=ssh_password, bitbucket_token=bitbucket_token,
                         branch=self.Parameters.source_branch, logger=pl.logger)

        with open('version.txt') as version:
            build_commit = version.readline().strip()
            build_version = version.readline().strip()
        build_date = datetime.datetime.now().strftime('%Y-%m-%d_%H:%M:%S_UTC')
        branch_suffix = self.Parameters.source_branch.split('/')[-1]

        QloudInit(
            self, 'qloud-init', 'qloud-init', build_version=build_version, build_commit=build_commit,
            build_date=build_date, build_branch=branch_suffix
        )
        QloudInitSSH(
            self, 'qloud-init-ssh', 'qloud-init-ssh', build_version=build_version, build_commit=build_commit,
            build_date=build_date, build_branch=branch_suffix
        )
        QloudInitLogger(
            self, 'qloud-init-logger', 'qloud-init-logger', build_version=build_version,
            build_commit=build_commit, build_date=build_date, build_branch=branch_suffix
        )
        QloudInitTestResults(
            self, 'test results', 'test_results.log', build_version=build_version,
            build_commit=build_commit, build_date=build_date, build_branch=branch_suffix
        )

        self.set_info('Built version #{} commit {}'.format(build_version, build_commit))

    def on_release(self, params):
        for installation in ['qloud', 'qloud-ext']:
            self.__process_release_info(self.__upload_to(installation))
        super(BuildQloudInit, self).on_release(params)

    def __process_release_info(self, res):
        if res['status_code'] == '200':
            self.set_info('Uploaded to {installation}, message: {text}'.format(**res))
        else:
            raise Exception(
                "Can't upload to {installation}, url: {url}, code: {status_code}, message: {text}".format(**res))

    def __upload_to(self, installation):
        url = 'https://{}.yandex-team.ru/api/management/qloud-init/release'.format(installation)
        res = requests.post(url, data={'task-id': self.id})
        return dict(
            installation=installation,
            url=url,
            status_code=str(res.status_code),
            text=res.text,
        )


def build_remote(hostname, username, private_key, ssh_password, bitbucket_token, branch, logger):
    with paramiko.SSHClient() as client:
        client.set_missing_host_key_policy(paramiko.AutoAddPolicy())
        logger.info("connecting to remote host {}@{} to build branch {}".format(username, hostname, branch))
        client.connect(hostname, username=username, pkey=private_key, password=ssh_password)

        tran = client.get_transport()
        chan = tran.open_session()
        chan.get_pty()
        with chan.makefile() as f:
            try:
                chan.exec_command('''
                    rm -rf qloud-init &&
                    git clone https://x-oauth-token:{}@bb.yandex-team.ru/scm/qloud/qloud-init.git --branch {} &&
                    cd qloud-init &&
                    git rev-parse --short HEAD > version.txt &&
                    git log --pretty=oneline | wc -l >> version.txt &&
                    make all
                '''.format(bitbucket_token, branch))
            except Exception as e:
                logger.info('Remote build failed {}'.format(str(e)))
                raise
            finally:
                logger.info(f.read())

        with client.open_sftp() as sftp:
            sftp.get('qloud-init/qloud-init', 'qloud-init')
            sftp.get('qloud-init/qloud-init-ssh', 'qloud-init-ssh')
            sftp.get('qloud-init/empty', 'qloud-init-logger')
            sftp.get('qloud-init/version.txt', 'version.txt')
            sftp.get('qloud-init/test_results.log', 'test_results.log')
