# -*- coding: utf-8 -*-

import os
import shutil
import logging
import traceback
import time
import rtline_resource_types

from sandbox.common.types.client import Tag

from sandbox.sandboxsdk.parameters import SandboxBoolParameter, SandboxStringParameter
from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask
import sandbox.projects.common.build.parameters as build_params
import sandbox.projects.common.constants as consts
from sandbox.projects.common import context_managers
import sandbox.projects.yadrive.resources as drive_resources
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.common.nanny import const as nanny_const
from sandbox.projects.common.nanny import nanny
from sandbox.projects.BuildSearch import BuildSearch
from sandbox.projects import resource_types
from sandbox.projects.common import utils
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.errors import SandboxTaskFailureError, SandboxTaskUnknownError, SandboxSvnIncorrectUrl
from sandbox.sandboxsdk.channel import channel


class PYmakeParameters(SandboxStringParameter):
    name, description, default_value = 'ymake_parameters', 'ymake parameters (all -D)', ''


class PYmakeFlags(SandboxStringParameter):
    name, description, default_value = 'ymake_flags', 'ymake -W, -f flags', ''


class PSaveMakePaths(SandboxBoolParameter):
    name, description, default_value = 'save_make_paths', 'Save make paths', False


class PWriteDepsTree(SandboxBoolParameter):
    name, description, default_value = 'write_deps_tree', 'Write dependencies tree', False


class PCheckoutFlag(SandboxBoolParameter):
    name, description, default_value = 'ymake_checkout_flag', 'clone+checkout(without copy all arcadia)', False


class PDebugFlag(SandboxBoolParameter):
    name, description, default_value = 'debug_flag', 'flag for debugging', False


def GPBuildBin(bin_name):
    class PBuildBin(SandboxBoolParameter):
        name = 'build_rtline_' + bin_name
        description = name

    return PBuildBin


def get_build_bins_parameters():
    bin_names = [
        'drive_backend_configs',
        'drive_backend_config_admin',
        'drive_backend_config_chat',
        'drive_backend_config_prestable',
        'drive_backend_config_production',
        'drive_backend_config_qa',
        'drive_backend_config_robot',
        'drive_backend_config_service',
        'drive_backend_config_testing',
        'drive_backend_config_maas',
        'drive_backend_environment',
        'drive_telematics_configs',
        'drive_telematics_config_stable',
        'drive_telematics_config_prestable',
        'drive_telematics_config_testing',
        'rtyserver',
        'ext_rtyserver',
        'searchproxy',
        'indexerproxy',
        'ext_indexerproxy',
        'deploy_manager',
        'roads_graph',
        'masstransit_snippets',
        'frontend',
        'producer',
        'telematics_server',
    ]
    return [GPBuildBin(bin_name) for bin_name in bin_names]


def get_build_rty_server_params():
    return build_params.get_arcadia_params() + [
        build_params.ClearBuild,
        build_params.BuildSystemPlain,
        build_params.BuildType,
        build_params.Sanitize,
        build_params.TargetPlatform,
        build_params.UseArcadiaApiFuse,
        build_params.UseArcInsteadOfArcadiaApi
    ]


class BuildRTLine(nanny.ReleaseToNannyTask, BuildSearch):
    type = 'BUILD_RTLINE'
    client_tags = Tag.GENERIC & Tag.LINUX_PRECISE
    input_parameters = get_build_rty_server_params() \
                       + get_build_bins_parameters() \
                       + [PYmakeParameters, PYmakeFlags,
                          PSaveMakePaths,
                          PWriteDepsTree, PCheckoutFlag, PDebugFlag]
    """
        TARGET_RESOURCE_EX consists of four fields: (resource_type, target, target_path, filename)
        resource_type - type of the target resource
        target - title of the target
        target_path - final path of the built target resource. dirname of the target path is the path to checkout from arcadia
        filename is the name of target resource file.
    """
    TARGET_PATH_TO_NAME_MAP = {
        'drive/backend/configs': 'rtline_drive_backend_configs',
        'drive/backend/server': 'rtline_frontend',
        'drive/telematics/configs': 'rtline_drive_telematics_configs',
        'drive/telematics/server': 'rtline_telematics_server',
        'rtline/rtyserver': 'rtline_rtyserver',
        'rtline/searchproxy': 'rtline_searchproxy',
        'rtline/indexerproxy': 'rtline_indexerproxy',
        'rtline/deploy_manager': 'rtline_deploy_manager',
        'rtline/tools/roads_graph': 'rtline_roads_graph',
    }

    TARGET_RESOURCES = []

    TARGET_RESOURCES_EX = (
        (drive_resources.YaDriveBackendConfigs, 'rtline_drive_backend_configs', 'drive/backend/configs', ''),
        (drive_resources.YaDriveBackendConfigAdmin, 'rtline_drive_backend_config_admin', 'drive/backend/configs', 'admin.conf'),
        (drive_resources.YaDriveBackendConfigChat, 'rtline_drive_backend_config_chat', 'drive/backend/configs', 'chat.conf'),
        (drive_resources.YaDriveBackendConfigPrestable, 'rtline_drive_backend_config_prestable', 'drive/backend/configs', 'prestable.conf'),
        (drive_resources.YaDriveBackendConfigProduction, 'rtline_drive_backend_config_production', 'drive/backend/configs', 'production.conf'),
        (drive_resources.YaDriveBackendConfigQa, 'rtline_drive_backend_config_qa', 'drive/backend/configs', 'qa.conf'),
        (drive_resources.YaDriveBackendConfigRobot, 'rtline_drive_backend_config_robot', 'drive/backend/configs', 'robot.conf'),
        (drive_resources.YaDriveBackendConfigService, 'rtline_drive_backend_config_service', 'drive/backend/configs', 'service.conf'),
        (drive_resources.YaDriveBackendConfigTesting, 'rtline_drive_backend_config_testing', 'drive/backend/configs', 'testing.conf'),
        (drive_resources.YaDriveBackendConfigMaas, 'rtline_drive_backend_config_maas', 'drive/backend/configs', 'maas.conf'),
        (drive_resources.YaDriveBackendEnvironment, 'rtline_drive_backend_environment', 'drive/backend/configs', 'environment.lua'),
        (drive_resources.YaDriveTelematicsConfigs, 'rtline_drive_telematics_configs', 'drive/telematics/configs', ''),
        (drive_resources.YaDriveTelematicsConfigStable, 'rtline_drive_telematics_config_stable', 'drive/telematics/configs', 'stable.conf'),
        (drive_resources.YaDriveTelematicsConfigPrestable, 'rtline_drive_telematics_config_prestable', 'drive/telematics/configs', 'prestable.conf'),
        (drive_resources.YaDriveTelematicsConfigTesting, 'rtline_drive_telematics_config_testing', 'drive/telematics/configs', 'testing.conf'),
        (drive_resources.YaDriveTelematicsConfigWialon, 'rtline_drive_telematics_config_wialon', 'drive/telematics/configs', 'wialon.conf'),
        (resource_types.RTLINE_RTYSERVER, 'rtline_rtyserver', 'rtline/rtyserver', 'rtyserver'),
        (resource_types.RTLINE_SEARCHPROXY, 'rtline_searchproxy', 'rtline/searchproxy', 'searchproxy'),
        (resource_types.RTLINE_INDEXERPROXY, 'rtline_indexerproxy', 'rtline/indexerproxy', 'indexerproxy'),
        (resource_types.RTLINE_DEPLOY_MANAGER, 'rtline_deploy_manager', 'rtline/deploy_manager', 'deploy_manager'),
        (resource_types.RTYSERVER_UTILS_ROADS_GRAPH, 'rtline_roads_graph', 'rtline/tools/roads_graph', 'roads_graph'),
        (rtline_resource_types.RTLINE_TELEMATICS_SERVER, 'rtline_telematics_server', 'drive/telematics/server', 'telematics_server'),
        (rtline_resource_types.RTLINE_FRONTEND, 'rtline_frontend', 'drive/backend/server', 'server'),
    )

    LOCAL_BIN_DIR = 'binaries'
    rtline_configs_res_dir = 'rtline_configs'
    release_dir = ''
    required_ram = 80000
    execution_space = 80000

    def on_enqueue(self):
        ArcadiaTask.on_enqueue(self)
        if self.ctx['build_bundle']:
            return
        self.do_on_enqueue()
        dprio = 0
        try:
            dprio = int(self.ctx.get('dprio', 0))
        except:
            pass
        self.score = dprio

    def mark_stage_start(self, stage):
        self.ctx['__STAGES'][stage + '_started'] = time.time()

    def mark_stage_finish(self, stage):
        self.ctx['__STAGES'][stage + '_finished'] = time.time()

    def do_on_enqueue(self):
        self.TARGET_RESOURCES = []
        for resource_type, target, target_path, filename in self.TARGET_RESOURCES_EX:
            target_resource = (resource_type, target, target_path)
            self.TARGET_RESOURCES.append(target_resource)

        if self.ctx.get('save_make_paths') and self.ctx.get('build_system') in (consts.YMAKE_BUILD_SYSTEM, consts.DISTBUILD_BUILD_SYSTEM):
            return

        for resource_type, target, target_path, filename in self.TARGET_RESOURCES_EX:
            if self._target_enabled(target):
                resource_filename = filename if (filename and len(filename) > 0) else target
                resource_path = os.path.join(self.LOCAL_BIN_DIR, resource_filename)
                resource_name = '%s (%s)' % (self.descr, resource_filename)
                resource = self._create_resource(resource_name,
                                                 resource_path,
                                                 resource_type,
                                                 arch=self.arch)
                self.ctx['%s_resource_id' % target] = resource.id
                logging.info('created resource {} {} {}'.format(resource_name, resource_path, resource.id))
            else:
                logging.info('target {} from path {} skipped'.format(target, target_path))

        self.ctx["don_t_release"] = []

        self.set_info('task enqueued for execution')

    def _run_ymake_build(self, enabled_targets, build_type, clear_build, sanitize):
        self.release_dir = self.abs_path(self.LOCAL_RELEASE_DIR)

        if self.ctx.get('save_make_paths'):
            make_folder(self.release_dir)
            os.chdir(self.release_dir)
            ya_path = os.path.join(self.arcadia_src_dir, 'ya')
            ymake_cmd = [ya_path, 'make',
                         '-j0', '--dump-graph',
                         '--source-root=' + self.arcadia_src_dir,
                         '--results-root=' + self.release_dir,
                         ]
            ymake_cmd.append('--build-dir=' + self.abs_path('tmp'))
            if self.ctx.get('ymake_parameters', '').strip():
                ymake_cmd.extend(self.ctx.get('ymake_parameters').strip().split())
            if not self.ctx.get('write_deps_tree'):
                deps = sdk.dump_targets_deps(self.arcadia_src_dir, enabled_targets)
                self.save_dir_list_resource(deps, 'all')
                self.ctx['deps_count_all'] = len(deps)
            else:
                targets = [(os.path.basename(os.path.join(self.arcadia_src_dir, t)),
                            t)
                           for t in enabled_targets]
                for targ in targets:
                    deps = sdk.dump_targets_deps(self.arcadia_src_dir, [targ[1], ])
                    self.save_dir_list_resource(deps, targ[0])
                    self.ctx['deps_count_' + targ[0]] = len(deps)
            self.ctx['task_done'] = True
        else:
            user_flags = sdk.parse_flags(self.ctx.get('ymake_parameters', ''))
            no_d_flags = self.ctx.get('ymake_flags', '')
            if no_d_flags:
                try:
                    no_d_flags = 'CFLAGS=' + no_d_flags.strip().strip(';')
                    no_d_flags = no_d_flags.split(';')
                    no_d_flags = dict(
                        (f.split('=', 1)[0], '' + f.split('=', 1)[1] + '') for f in no_d_flags[:] if '=' in f
                    )
                except Exception as e:
                    logging.error('while parsing flags, error: %s' % e)

            flags = {}
            if user_flags:
                flags.update(user_flags)
            if no_d_flags:  # and 'flags' in self.descr:
                flags.update(no_d_flags)
                logging.info('all_flags: %s' % flags)

            platform = self.ctx.get(consts.TARGET_PLATFORM_KEY) or None

            build_system = self.ctx.get('build_system', consts.YMAKE_BUILD_SYSTEM)
            sdk.do_build(
                build_system, self.arcadia_src_dir, enabled_targets, clear_build=clear_build,
                build_type=build_type, def_flags=flags, results_dir=self.release_dir, target_platform=platform,
                sanitize=sanitize,
                timeout=7200, checkout=True
            )

    def _build(self, enabled_targets):
        clear_build = self.ctx.get(consts.CLEAR_BUILD_KEY, True)
        build_type = self.ctx.get(consts.BUILD_TYPE_KEY)
        sanitize = self.ctx.get(consts.SANITIZE, False)
        self._run_ymake_build(enabled_targets, build_type, clear_build, sanitize)

    def do_execute(self):
        self.ctx['queue_time_min'] = round((self.timestamp_start - self.timestamp)/60, 1)
        self.ctx['__STAGES'] = self.ctx.get('__STAGES', {})
        os.chdir(self.abs_path())
        binaries_dir = os.path.join(self.abs_path(), self.LOCAL_BIN_DIR)
        make_folder(binaries_dir)

        self.mark_stage_start('get_src_dir')
        arcadia_ctx = self._get_src_dir()
        self.mark_stage_finish('get_src_dir')

        with arcadia_ctx as arcadia_src_dir:
            self.arcadia_src_dir = arcadia_src_dir
            self.ctx['arcadia_revision'] = utils.svn_revision(self.arcadia_src_dir)
            arcadia_patch = self.ctx.get('arcadia_patch')
            if arcadia_patch.strip() and arcadia_patch.strip() not in ('None', ):
                Arcadia.apply_patch(self.arcadia_src_dir, self.ctx.get('arcadia_patch'), self.abs_path())

            # build dependencies tree
            if self.ctx.get('write_deps_tree'):
                # todo something here
                pass

            #
            self.fill_system_info()

            enabled_targets = []
            for _, target, target_path, filename in self.TARGET_RESOURCES_EX:
                if self._target_enabled(target):
                    enabled_targets.append(target_path)

            #
            self.mark_stage_start('build')
            self._build(enabled_targets)
            self.mark_stage_finish('build')
            if self.ctx.get('task_done', False):
                return

            for _, target, target_path, filename in self.TARGET_RESOURCES_EX:
                logging.info('Check target %s, path %s, filename %s' % (target, target_path, filename))
                if self._target_enabled(target):
                    # build resource
                    build_dir = self.abs_path(os.path.join(self.release_dir, target_path))

                    os.chdir(build_dir)  # check abs path
                    logging.info('read_resource: %s, %s', target, self._target_resource_id(target))
                    resource_path = channel.sandbox.get_resource(self._target_resource_id(target)).path

                    target_file = os.path.join(target_path, filename)

                    ready_file = self.abs_path(os.path.join(self.release_dir, target_file))
                    try:
                        if filename and len(filename) > 0:
                            logging.info('Target file: %s copying to: %s' % (ready_file, resource_path))
                            shutil.copy(ready_file, resource_path)
                            logging.info('Target file: %s copied to: %s' % (ready_file, resource_path))
                        else:
                            logging.info('Target directory: %s copying to: %s' % (ready_file, resource_path))
                            shutil.copytree(ready_file, resource_path)
                            logging.info('Target directory: %s copied to: %s' % (ready_file, resource_path))
                    except IOError as e:
                        raise SandboxTaskFailureError('Error with copy : %s, file %s' % (repr(e), ready_file))

            # cleanup
            self.clean_release_dir(self.release_dir)

            self.set_info('build completed')
            self.ctx['completed'] = True
            self.ctx['exec_time_min'] = round((int(time.time()) - self.timestamp_start)/60, 1)

    def save_dir_list_resource(self, lines, targ):
        res_path = self.abs_path("dir_list_" + targ)
        with open(res_path, 'w') as f:
            f.write('\n'.join(lines))
        res = self.create_resource('dir list for ' + targ, res_path, resource_types.DIR_LIST)
        return res

    def _get_src_dir(self):
        """Returns MountPoint context manager, either mounted by Arc or Svn"""
        use_arc_instead_of_aapi = utils.get_or_default(self.ctx, build_params.UseArcInsteadOfArcadiaApi)
        if utils.get_or_default(self.ctx, build_params.UseArcadiaApiFuse) and self.ctx.get('debug_flag', 0):
            arcadia_url = self.ctx.get(consts.ARCADIA_URL_KEY, "")
            return sdk.mount_arc_path(arcadia_url, fallback=True, use_arc_instead_of_aapi=use_arc_instead_of_aapi)

        if not self.ctx.get('ymake_checkout_flag', 0):
            try:
                arcadia_src_dir = self.get_arcadia_src_dir()
            except SandboxSvnIncorrectUrl as e:
                logging.error(traceback.format_exc())
                url = self.ctx[consts.ARCADIA_URL_KEY]
                rev = Arcadia.parse_url(url).revision
                if rev:
                    url = url.replace('@'+str(rev), '')
                    rev1 = Arcadia.get_revision(url)
                    if (int(rev) - int(rev1)) < 2000:
                        raise SandboxTaskUnknownError('cannot find revision %s, found %s' % (rev, rev1))
                raise SandboxSvnIncorrectUrl("cannot get svn: "+str(e))
        else:
            arcadia_src_dir = sdk.do_clone(self.ctx[consts.ARCADIA_URL_KEY], self)
        return context_managers.nullcontext(arcadia_src_dir)

    def mark_all_ready(self):
        for resource in self.list_resources():
            if resource.type.name == 'TASK_LOGS':
                continue
            logging.info('found resource %s' % str(resource))
            if not resource.is_ready():
                try:
                    resource.mark_ready()
                except Exception as e:
                    raise SandboxTaskUnknownError('Problems with resource %s, error %s' % (str(resource.id), e))

    def on_finish(self):
        if not self.ctx.get('completed', False):
            return
        self.mark_stage_start('share')
        self.mark_all_ready()
        self.mark_stage_finish('share')

    def get_nanny_release_info(self, additional_parameters):
        result = nanny.ReleaseToNannyTask.get_nanny_release_info(self, additional_parameters)
        release_tag = additional_parameters.get('release_tag')
        if release_tag:
            result['meta']['labels']['release_tag'] = release_tag
        if release_tag and release_tag != 'unknown' and release_tag != 'none':
            release_type = result['spec']['sandboxRelease']['releaseType']
            result['spec']['sandboxRelease']['releaseType'] = '{}-{}'.format(release_type, release_tag)
        return result

    def on_release(self, additional_parameters):
        channel.task = self
        for resource in self.list_resources():
            if resource.type.releasable:
                if resource.id in self.ctx.get('don_t_release', []):
                    continue
                if not resource.is_ready():
                    resource.mark_ready()
        if additional_parameters:
            release_subject = additional_parameters.get(nanny_const.RELEASE_SUBJECT_KEY, '')
            if len(release_subject) == 0:
                additional_parameters[nanny_const.RELEASE_SUBJECT_KEY] = 'trunk/r{}\t{}\t{}'.format(
                    self.ctx.get('arcadia_revision', '<unknown_revision>'),
                    self.ctx.get('build_type', '<unknown_build_type>'),
                    self.ctx.get(consts.SANITIZE, 'non-sanitized'),
                )

        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)


__Task__ = BuildRTLine
