# coding: utf-8

import os
import logging
import re
import shutil
import time
import tarfile

from sandbox.projects import resource_types
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common import utils

from sandbox.sandboxsdk.parameters import SandboxArcadiaUrlParameter
from sandbox.sandboxsdk.parameters import LastReleasedResource
from sandbox.sandboxsdk.process import run_process
from sandbox.sandboxsdk.paths import remove_path
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.task import SandboxTask


class ArcadiaUrlParameter(SandboxArcadiaUrlParameter):
    default_value = "arcadia:/arc/trunk/arcadia"


class VirtualenvPackageParameter(LastReleasedResource):
    name = 'virtualenv_package'
    description = 'Virtualenv like package ready for razladki'
    resource_type = resource_types.RAZLADKI_VIRTUALENV_PACKAGE


class IssShardsPackageParameter(LastReleasedResource):
    name = 'iss_shards_package'
    description = 'iss_shards utility'
    resource_type = resource_types.ISS_SHARDS


class BuildRazladki(nanny.ReleaseToNannyTask, SandboxTask):
    """
       Build Razladki.
       Checkouts razladki from SVN and packs it with ready virtualenv package.
    """
    type = 'BUILD_RAZLADKI'

    input_parameters = [ArcadiaUrlParameter, VirtualenvPackageParameter, IssShardsPackageParameter]

    URL_RAZLADKI_SVN = 'razladki'

    PATH_PACKET = 'razladki'
    PATH_TGZ = 'razladki.tar.gz'

    def arcadia_info(self):
        return self.ctx.get("revision"), "Razladki {} r{}".format(self.ctx.get("branch"), self.ctx.get("revision")), self.ctx.get("branch")

    def on_execute(self):
        url_arcadia = self.ctx[ArcadiaUrlParameter.name]

        revision = Arcadia.get_revision(url_arcadia)
        branch = utils.get_short_branch_name(url_arcadia)
        if not branch:
            try:
                branch_match = re.search(r'(branch.+)$', url_arcadia).group(0)
                branch = branch_match
            except IndexError:
                pass

        logging.info("ARCADIA {} {}".format(branch, revision))
        assert revision, 'Trying to fetch project from SVN, but no revision specified'

        self.ctx['revision'] = revision
        self.ctx['branch'] = branch

        path_packet = self.path(self.PATH_PACKET)
        path_checkout = path_packet  # os.path.join(path_packet, "razladki")

        os.mkdir(path_packet)
        self._export_arcadia(self.URL_RAZLADKI_SVN, path_checkout)

        iss_shards_tool = self.sync_resource(self.ctx[IssShardsPackageParameter.name])
        logging.info('Extracting iss_shards package %s to %s' % (iss_shards_tool, path_packet))
        run_process(['chmod', '+x', iss_shards_tool])

        virtualenv_package = self.sync_resource(self.ctx[VirtualenvPackageParameter.name])
        logging.info('Extracting virtualenv package %s to %s' % (virtualenv_package, path_packet))
        tarfile.open(virtualenv_package).extractall(path_packet)

        db_shared_json = os.path.join(path_packet, 'db-shared.json')
        if os.path.exists(db_shared_json):
            logging.info('Removing default db conf file %s' % db_shared_json)
            os.remove(db_shared_json)

        python_path = os.path.join(path_packet, 'env', 'bin', 'python')
        if not os.path.exists(python_path):
            raise ValueError('Seems no "python" in %s after installing dependencies package' % python_path)

        self._create_shard(iss_shards_tool, path_packet)

    def _create_shard(self, iss_shards_tool, path_packet):
        shard_name = 'razladki-r{0}-{1}'.format(self.ctx['revision'], time.strftime("%Y%m%d_%H%M%S"))

        logging.info('create shard {0}'.format(shard_name))
        self.ctx['shard_name'] = shard_name

        shard_path = self.path(shard_name)
        remove_path(shard_path)
        os.mkdir(shard_name)

        with tarfile.open(self.path(self.PATH_TGZ), 'w:gz') as tar:
            for entry in os.listdir(path_packet):
                tar.add(os.path.join(path_packet, entry), entry)

        shutil.copy(self.path(self.PATH_TGZ), shard_path)

        logging.info('iss_shards configure')
        run_process([iss_shards_tool, 'configure', shard_path,
                   '--id', shard_name],
            log_prefix="iss_shards_configure",
            work_dir=self.path()
        )

        with open(os.path.join(shard_path, 'shard.conf'), 'a') as f:
            f.write(
                '%install\n'
                'tar -xzf {}\n'.format(self.PATH_TGZ)
            )

        logging.info('iss_shards register %s' % '../')
        run_process([iss_shards_tool, 'register', './'],
            log_prefix="iss_shards_register",
            work_dir=shard_path
        )

        logging.info('create shard resource')
        self.create_resource(
            description='razladki shard r{0}'.format(self.ctx['revision']),
            resource_path=shard_path,
            resource_type=resource_types.RAZLADKI_SHARD,
            arch='linux'
        )

    def _get_arcadia_url(self, arcadia_path):
        url_arcadia = self.ctx[ArcadiaUrlParameter.name]

        url_parsed = Arcadia.parse_url(url_arcadia)
        path_new = re.sub(r'/arcadia.*', '/arcadia/' + arcadia_path, url_parsed.path, count=1)

        return Arcadia.replace(url_arcadia, path=path_new)

    def _export_arcadia(self, arcadia_path, path):
        url = self._get_arcadia_url(arcadia_path)
        logging.info("EXPORT '{}' TO '{}'".format(url, path))

        Arcadia.export(url, path)


__Task__ = BuildRazladki
