# -*- coding: utf-8 -*-
import logging
import os
import tempfile

from sandbox.projects.common import file_utils as fu

import sandbox.common.types.misc as ctm
import sandbox.common.types.client as ctc

from Utils import Utils

from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import svn

from sandbox import sdk2
import sandbox.sdk2.helpers
from sandbox.sdk2.helpers import subprocess


class ReactUiResource(sdk2.Resource):
    auto_backup = True
    any_arch = True

    app_name = sdk2.parameters.String()
    app_version = sdk2.parameters.String()
    svn_branch = sdk2.parameters.String()
    svn_path = sdk2.parameters.String()
    svn_revision = sdk2.parameters.String()
    svn_subpath = sdk2.parameters.String()
    svn_tag = sdk2.parameters.String()
    svn_trunk = sdk2.parameters.String()


class BuildReactUi(sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        dns = ctm.DnsType.DNS64
        disk_space = 5 * 1024
        client_tags = ctc.Tag.LINUX_TRUSTY

    class Parameters(sdk2.Task.Parameters):
        description = "-"
        checkout_arcadia_from_url = sdk2.parameters.ArcadiaUrl()
        cmd = sdk2.parameters.String("Command to run after cloning repository",
                                     default_value='build.sh', multiline=True, required=True)

        env = sdk2.parameters.Dict("Dict with environment vars")

        _container = sdk2.parameters.Container("Environment container resource",
                                               default_value=190569595, required=True)

    class Context(sdk2.Task.Context):
        pass

    CODE_NAME = 'nodejs_code'

    def on_prepare(self):
        """
        What should it do:
            * checkout repository with specific tag/branch/commit
        """

        logging.debug('on_prepare')

        self.Context.checkout_path = os.path.join(self.path(), tempfile.mkdtemp())

        # Get code from repository.
        self._get_code()

        logging.debug('ARCADIA_SRC_DIR: {}'.format(self.Context.checkout_path))

    def on_execute(self):
        """
        What should it do:
            * pack and export archive with code as a resource (optional)
        """

        logging.debug('on_execute')

        # Generate common attributes list
        common_attributes = self.gen_common_attributes()

        logging.debug('COMMON_ATTRIBUTES: {}'.format(common_attributes))

        # Hook: before common execution steps
        self.before_execution(common_attributes=common_attributes)

        # Run command from context
        logging.debug('Running command:\n{cmd}\nAt work dir: {path}'.format(
                cmd=self.Parameters.cmd,
                path=self.Context.checkout_path
            ))

        # Add custom.sh and execute it
        self.execute_sh_file_with_command(self.Parameters.cmd)

        # Create sandbox resource from cloned folder after build
        self.create_nodejs_code_resource(self.Context.checkout_path, common_attributes)

        # Hook: after common execution steps
        self.after_execution(common_attributes=common_attributes)

    def execute_sh_file_with_command(self, content):
        tf = tempfile.NamedTemporaryFile().name

        fu.write_file(tf, content)

        self.execute('/bin/bash {}'.format(tf))
        self.execute('/bin/rm {}'.format(tf))

    def execute(self, command_line):
        env = self.Parameters.env

        execute_custom_env = os.environ.copy()
        execute_custom_env.update(env)

        with sandbox.sdk2.helpers.ProcessLog(self, logger=logging.getLogger("cmd-run")) as pl:
            subprocess.Popen(
                command_line, shell=True, stdout=pl.stdout, stderr=subprocess.STDOUT,
                cwd=self.Context.checkout_path, env=execute_custom_env
            ).wait()

    def _get_code(self):
        """
        Get project code from remote source using Arcadia.checkout.
        Set 'self.git' attribute for further usage.
        """

        svn_url = svn.Arcadia.svn_url(self.Parameters.checkout_arcadia_from_url)

        logging.debug('SVN URL:\n{svn_url}\nCHECKOUT DIR: {path}'.format(
            svn_url=svn_url, path=self.Context.checkout_path
        ))

        if not svn.Arcadia.check(svn_url):
            raise errors.SandboxTaskFailureError('Sandbox branch {0} does not exist.'.format(svn_url))

        svn.Arcadia.checkout(svn_url, self.Context.checkout_path)

    def gen_common_attributes(self):
        """
        Generate common resources attributes list using task's runtime context and current repository state.
        When local repository copy exists, 'commit_hash' and 'branch' or 'tag' attributes are generated.
        Custom attributes are used to update common attributes list, so you can recover any common attribute from
        'CustomResourceAttributes' task parameter.

        :param repo_path: path to local git repository copy. This copy's state is used to generate attributes.
                          Generated attributes:
                            'commit_hash' (always)
                            'tag' (optional, depends on HEAD)
                            'branch' (optional, depends on HEAD)

                          Current repository HEAD contents results in:
                            * 'tag' attribute generation when HEAD is a symbolic reference to tag.
                            * 'branch' attribute generation when HEAD is a symbolic reference to branch.
        """

        arcadia_parse_url = svn.Arcadia.parse_url(self.Parameters.checkout_arcadia_from_url)

        return {
            'svn_path': arcadia_parse_url.path,
            'svn_revision': arcadia_parse_url.revision,
            'svn_branch': arcadia_parse_url.branch,
            'svn_tag': arcadia_parse_url.tag,
            'svn_trunk': arcadia_parse_url.trunk,
            'svn_subpath': arcadia_parse_url.subpath
        }

    def get_package_info(self, code_path):
        content = fu.json_load(os.path.join(code_path, 'package.json'))
        return content['name'], content['version']

    def create_nodejs_code_resource(self, code_path, common_attributes):
        """
        Create NodeJS code archive and export it as a resource
        """

        # Get current version from package.json
        (project_name, project_version) = self.get_package_info(code_path)

        # Resource creation
        resource_name = 'Архив с кодом React UI проекта {} ({}) версии {}'\
            .format(project_name, self.CODE_NAME, project_version)

        code_archive_name = '{}.{}.{}.tar.gz'.format(project_name, self.CODE_NAME, project_version)

        common_attributes.update({'app_name': project_name, 'app_version': project_version})

        resource = ReactUiResource(self, resource_name, code_archive_name, **common_attributes)

        # Hook
        self.code_before_archivation(common_attributes=common_attributes)

        out_path = os.path.join(code_path, 'build', project_version)

        logging.debug('Create pack from dir: {} with version: {} code resource path: {} resource attrs: {}'
                      .format(out_path, project_version, resource.path, resource))

        # Archivation
        Utils.pack_archive(out_path, unicode(resource.path))

        # Hook
        self.code_after_archivation(common_attributes=common_attributes)

    def before_execution(self, common_attributes):
        """
        This method is called after common task configuration (repository cloning, common attributes list generation,
        etc.) and before any real work (virtual environment generation, code archiving, etc).

        If you want to make some actions here, just redefine method in child class.

        :param common_attributes: list of common attributes that should be set for all task resources.
        :type common_attributes: dict
        """
        pass

    def after_execution(self, common_attributes):
        """
        This method is called when all real work is already done (virtual environment hed been generated, code archive
        had been created, etc).

        If you want to make some actions here, just redefine method in child class.

        :param common_attributes: list of common attributes that should be set for all task resources.
        :type common_attributes: dict
        """
        pass

    def code_before_archivation(self, common_attributes):
        """
        This method is called before code archivation process start.

        If you want to make some actions here, just redefine method in child class.

        :param common_attributes: list of common attributes that should be set for all task resources.
        :type common_attributes: dict
        """
        pass

    def code_after_archivation(self, common_attributes):
        """
        This method is called after code archivation process end.

        If you want to make some actions here, just redefine method in child class.

        :param common_attributes: list of common attributes that should be set for all task resources.
        :type common_attributes: dict
        """
        pass
