# -*- coding: utf-8 -*-

import os

from sandbox.projects import resource_types
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.paths import copy_path
from sandbox.sandboxsdk.process import run_process, check_process_return_code

MAIN_REPO_URI = 'https://git.qe-infra.yandex-team.ru/scm/nanny/base-search-readahead.git'
GIT_COMMIT_KEY = 'git_commit'


class GitTagParameter(SandboxStringParameter):
    name = 'tag'
    description = 'Git tag'
    default_value = 'master'


class BuildReadahead(SandboxTask):
    type = 'BUILD_READAHEAD'

    input_parameters = [GitTagParameter]

    def get_readahead_path(self):
        return self.abs_path('readahead.py')

    def arcadia_info(self):
        return self.ctx.get(GIT_COMMIT_KEY, None), 'readahead/%s' % (self.ctx[GitTagParameter.name]), None

    def on_enqueue(self):
        self.ctx['readahead_executable_id'] = self._create_resource(
                self.descr,
                self.get_readahead_path(),
                resource_types.READAHEAD_EXECUTABLE, arch='any'
            ).id

    def on_execute(self):
        _clone_repo(self.ctx[GitTagParameter.name], 'readahead')

        self.ctx[GIT_COMMIT_KEY] = _get_git_commit(self.abs_path('readahead'))

        copy_path(os.path.join(self.abs_path('readahead'), 'readahead.py'), self.get_readahead_path())


def _clone_repo(rev, path):
    run_process(['git', 'clone', MAIN_REPO_URI, path], log_prefix='clone_master_repo')
    run_process(['git', 'checkout', rev], work_dir=path, log_prefix='checkout_master_repo')


def _get_git_commit(path):
    process = run_process(["git", "rev-parse", "HEAD"], work_dir=path, outs_to_pipe=True, wait=False)
    stdout, stderr = process.communicate()
    check_process_return_code(process)
    return stdout.strip()


__Task__ = BuildReadahead
