# coding: utf-8

import logging
import os
import re

import sandbox.sdk2.parameters as parameters2
import sandbox.common.types.client as ctc

from sandbox.projects import resource_types
from sandbox.projects.common import constants
from sandbox.projects.common import utils
from sandbox.projects.common.build import parameters as bp
from sandbox.projects.common.build.YaMake import YaMakeTask
from sandbox.projects.common.nanny import nanny
import sandbox.projects.common.constants as consts

from sandbox.projects.common.noapacheupper import build as nb

from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import process as sp
from sandbox.sandboxsdk.svn import Arcadia


class Params:
    class RearrangeSvnUrl(parameters.SandboxArcadiaUrlParameter):
        name = 'rearrange_svn_url'
        description = 'Rearrange svn url'
        default_value = 'arcadia:/arc/trunk/arcadia/search/web/rearrs_upper/rearrange'
        group = bp.BASE_BUILD_GROUP_NAME

    class ClearBuild(parameters.SandboxBoolParameter):
        name = consts.CLEAR_BUILD_KEY
        description = 'Clear build'
        default_value = False
        group = 'Base build params'

    class BuildSystem(parameters2.String):
        name = consts.BUILD_SYSTEM_KEY
        description = 'Build system'
        required = True
        default_value = consts.SEMI_DISTBUILD_BUILD_SYSTEM
        group = 'Build system params'
        choices = [
            ('Ya', consts.YMAKE_BUILD_SYSTEM),
            # TODO: remove after experiments
            ('Ya force', consts.YA_MAKE_FORCE_BUILD_SYSTEM),
            ('semi-distbuild', consts.SEMI_DISTBUILD_BUILD_SYSTEM),
            ('distbuild', consts.DISTBUILD_BUILD_SYSTEM),
        ]
        sub_fields = {
            consts.YMAKE_BUILD_SYSTEM: [consts.TARGET_PLATFORM_KEY],
            consts.YA_MAKE_FORCE_BUILD_SYSTEM: [consts.TARGET_PLATFORM_KEY],
        }

    class TrivialNaming(parameters.SandboxBoolParameter):
        """
            Включение опции отключает сложную генерацию имён для ресурса и позволяет создать их
            на ранних стадиях пуска задачи, что необходимо testenv-у для организации каскада
            задач (сцепка через ресурсы)
        """
        name = 'trivial_naming'
        description = 'Use trivial resources names/Early resource creation'
        default_value = False

    params = (
        ClearBuild,
        BuildSystem,
        RearrangeSvnUrl,
        bp.ArcadiaUrl,
        bp.ArcadiaPatch,
        TrivialNaming,
        bp.UseArcadiaApiFuse,
        bp.AllowArcadiaApiFallback,
    )


# alias for TestingRelease task
RearrangeSvnUrlParameter = Params.RearrangeSvnUrl

_REARRANGE_DIR = 'rearrange_dir'


class BuildRearrange(nanny.ReleaseToNannyTask, YaMakeTask):
    """
        Сборка переранжирований для верхнего метапоиска
    """

    type = 'BUILD_REARRANGE'

    client_tags = ctc.Tag.Group.LINUX

    input_parameters = Params.params
    execution_space = 80 * 1024  # Mb
    cores = 24

    def on_enqueue(self):
        self.ctx['build_system'] = 'ya'
        YaMakeTask.on_enqueue(self)
        channel.task = self
        if self._trivial_naming():
            self.ctx[_REARRANGE_DIR] = 'rearrange_data'
            self._create_resources()

    def on_execute(self):
        if not self._trivial_naming():
            self.ctx[_REARRANGE_DIR] = self.generate_rearrange_name(self.ctx[Params.RearrangeSvnUrl.name])
        rearrange = self.ctx[_REARRANGE_DIR]
        rearrange_dir = self.abs_path(rearrange)

        self.ctx['targets'] = 'search/web/rearrs_upper/rearrange'
        YaMakeTask.on_execute(self)
        svn_url = self.ctx[bp.ArcadiaUrl.name]
        # move result to rearrange_dir in post_build() hook

        with open(os.path.join(rearrange_dir, '.svninfo'), 'w') as svn_info_file:
            if svn_url.startswith('arcadia:'):  # native svn not support 'arcadia:', so fix it
                svn_url = Arcadia.svn_url(svn_url)
            sp.run_process(['svn', 'info', svn_url], stdout=svn_info_file)
            logging.info('write .svninfo')
            svn_info_file.write('Sandbox task: {}\n\n'.format(self.id))

        if not self._trivial_naming():
            logging.info('create resources')
            self._create_resources()

    def post_build(self, source_dir, output_dir, pack_dir):
        YaMakeTask.post_build(self, source_dir, output_dir, pack_dir)

        nb.export_ya_make_result(
            os.path.join(output_dir, 'search/web/rearrs_upper/rearrange/'),
            self.ctx[_REARRANGE_DIR],
        )

    # rearrange_data is released together with 'upper' component,
    # so we cannot use generic `release_template` property here
    @property
    def release_template(self):
        tag_url = self.ctx.get(constants.ARCADIA_URL_KEY)
        return self.ReleaseTemplate(
            subject='{0}: {1}'.format('rearrange_data', utils.branch_tag_from_svn_url(tag_url)[1])
        )

    def arcadia_info(self):
        parsed_url = Arcadia.parse_url(self.rearrange_svn_url())
        return parsed_url.revision, parsed_url.tag, parsed_url.branch

    def rearrange_svn_url(self):
        return self.ctx[Params.RearrangeSvnUrl.name]

    def _trivial_naming(self):
        return self.ctx.get(Params.TrivialNaming.name, False)

    def generate_rearrange_name(self, svn_url):
        svn_info = Arcadia.info(svn_url)
        url = svn_info['url']
        url = re.findall('arc/(.*)/arcadia', url)
        short_info = ''
        prev = ' '
        for elem in url[0]:
            if elem.isdigit():
                if prev == '-':
                    short_info += '-'
                short_info += elem
            elif elem.isalpha() and not prev.isalpha():
                short_info += elem
            prev = elem
        return 'rearrange_bundle' + '.' + short_info + '.' + str(svn_info['commit_revision'])

    def _create_resources(self):
        self.create_resource(
            self.descr,
            self.ctx[_REARRANGE_DIR],
            resource_types.REARRANGE_DATA,
            attributes={
                "ttl": 30,
            },
            arch="any",
        )


__Task__ = BuildRearrange
