# coding: utf-8

import logging
import os
import re

import sandbox.projects.release_machine.core.task_env as task_env
import sandbox.projects.release_machine.mixins.build as rm_build_mixin
from sandbox.common.types.client import Tag
from sandbox.projects import resource_types

from sandbox.projects.common import constants as consts
from sandbox.projects.common import decorators
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import link_builder as lb
from sandbox.projects.common import utils

from sandbox.projects.common.build import parameters as bp
from sandbox.projects.common.build.YaMake import YaMakeTask

from sandbox.projects.common.nanny import nanny
from sandbox.projects.release_machine.core import const as rm_const
from sandbox.projects.release_machine import input_params as rm_params
from sandbox.projects.release_machine.helpers.startrek_helper import STHelper
from sandbox.projects.release_machine.components import all as rmc
from sandbox.projects.release_machine.components.configs.rearrange_dynamic import RearrangeDynamicCfg
from sandbox.projects.common.constants.constants import ARCADIA_URL_KEY

from sandbox.projects.common.noapacheupper import build as nb

from sandbox.sandboxsdk import parameters as sp
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.svn import Arcadia

# 3.8 Gb (as stated in BLNDR-1425)
# 6 Gb
_MAX_REARRANGE_DIR_SIZE = 8300  # Mb (SPINCIDENTS-1351, NOAPACHE-129)


class CheckArchiveSize(sp.SandboxBoolParameter):
    name = 'check_archive_size'
    description = 'Check archive size'
    default_value = True
    required = False


class CheckoutParameter(sp.SandboxBoolParameter):
    name = consts.CHECKOUT
    description = 'Run ya make with --checkout'
    required = False
    default_value = True


class Params:

    params = (
        nb.Params.UseArchiver,
        nb.Params.ExtensionsToArchive,
        bp.ArcadiaUrl,
        bp.ArcadiaPatch,
        bp.UseArcadiaApiFuse,
        bp.AllowArcadiaApiFallback,
    )


class BuildRearrangeDynamic(rm_build_mixin.ComponentReleaseTemplate, YaMakeTask, nanny.ReleaseToNannyTask):
    """
        Сборка динамических переранжирований для верхнего метапоиска
    """

    type = 'BUILD_REARRANGE_DYNAMIC'
    execution_space = 80 * 1024  # 80 Gb
    cores = 24
    client_tags = task_env.TaskTags.startrek_client & Tag.Group.LINUX
    input_parameters = (
        CheckArchiveSize,
        CheckoutParameter,
    ) + Params.params
    environment = [task_env.TaskRequirements.startrek_client]

    def on_break(self):
        self._comment_problem_to_st()

    def on_failure(self):
        self._comment_problem_to_st()

    def _comment_problem_to_st(self):
        """UPREL-914"""
        c_name = self.ctx.get(rm_params.ComponentName.name)
        if c_name and not self.ctx.get("comment_on_break"):
            c_info = rmc.COMPONENTS[c_name]()
            st_helper = STHelper(self.get_vault_data(rm_const.COMMON_TOKEN_OWNER, rm_const.COMMON_TOKEN_NAME))
            st_helper.comment(
                self.ctx.get(rm_params.ReleaseNum.name),
                "Build {} is broken!".format(lb.task_wiki_link(self.id)),
                c_info
            )
            self.ctx["comment_on_break"] = True

    @staticmethod
    def generate_bundle_name(svn_url):
        svn_info = Arcadia.info(svn_url)
        url = svn_info['url']
        url = re.findall('arc/(.*)/arcadia', url)
        short_info = ''
        prev = ' '
        for elem in url[0]:
            if elem.isdigit():
                if prev == '-':
                    short_info += '-'
                short_info += elem
            elif elem.isalpha() and not prev.isalpha():
                short_info += elem
            prev = elem
        return 'rearrange.dynamic_bundle' + '.' + short_info + '.' + str(svn_info['commit_revision'])

    def on_enqueue(self):
        self.ctx['bundle_name'] = self.generate_bundle_name(utils.get_or_default(self.ctx, bp.ArcadiaUrl))
        self.ctx['build_system'] = 'ya'
        self.ctx[rm_const.COMPONENT_CTX_KEY] = RearrangeDynamicCfg.name
        YaMakeTask.on_enqueue(self)
        channel.task = self
        self.create_resource(
            self.descr,
            self.ctx['bundle_name'],
            resource_types.REARRANGE_DYNAMIC_DATA,
            attributes={
                "ttl": 90,
            },
            arch="any",
        )

    def on_execute(self):
        rearrange = self.ctx['bundle_name']
        rearrange_dir = self.abs_path(rearrange)
        nb.calculate_targets(self, static=False, dynamic=True, fast=False)
        YaMakeTask.on_execute(self)

        with open(os.path.join(rearrange_dir, '.svninfo'), 'w') as svn_info_file:
            svn_url = self.rearrange_svn_url()
            if svn_url.startswith('arcadia:'):
                svn_url = Arcadia.svn_url(svn_url)

            @decorators.retries(max_tries=20, delay=300)
            def write_svn_info():
                process.run_process(['svn', 'info', svn_url], stdout=svn_info_file)

            write_svn_info()

            svn_info_file.write('Sandbox task: {}\n\n'.format(self.id))

        dir_size = nb.get_directory_size(rearrange_dir)
        self.ctx['directory_size'] = dir_size
        self.set_info("Directory size: {} mb".format(dir_size))

        if utils.get_or_default(self.ctx, CheckArchiveSize):
            eh.ensure(
                dir_size < _MAX_REARRANGE_DIR_SIZE,
                "Rearrange.dynamic exceeds maximum allowed size {} mb".format(_MAX_REARRANGE_DIR_SIZE)
            )

        try:
            st_helper = STHelper(self.get_vault_data(rm_const.COMMON_TOKEN_OWNER, rm_const.COMMON_TOKEN_NAME))
            c_info = rmc.COMPONENTS[RearrangeDynamicCfg.name]()
            _, release_num = c_info.get_tag_info_from_build_task(self.id, ARCADIA_URL_KEY)
            logging.info("Got release number: %s", release_num)
            st_helper.write_grouped_comment(
                group_name=rm_const.TicketGroups.BuildTest,
                title="",
                content="Task was successfully built: {}".format(lb.task_wiki_link(self.id)),
                release_num=release_num,
                c_info=c_info
            )
        except Exception as exc:
            eh.log_exception("Unable to write comment about build", exc)

    def post_build(self, source_dir, output_dir, pack_dir):
        YaMakeTask.post_build(self, source_dir, output_dir, pack_dir)
        rearrange_dynamic_dir = self.ctx['bundle_name']

        nb.export_ya_make_result(
            os.path.join(output_dir, 'search/web/rearrs_upper/rearrange.dynamic/'),
            rearrange_dynamic_dir,
        )
        nb.archive(self, rearrange_dynamic_dir, output_dir)

    def on_release(self, additional_parameters):
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)

    def arcadia_info(self):
        parsed_url = Arcadia.parse_url(self.rearrange_svn_url())
        return parsed_url.revision, parsed_url.tag, parsed_url.branch

    def rearrange_svn_url(self):
        return self.ctx[bp.ArcadiaUrl.name]


__Task__ = BuildRearrangeDynamic
