# coding: utf-8
import os
import shutil
import multiprocessing as mp

import sandbox.common.types.client as ctc

import sandbox.sandboxsdk.process as sdk_process
import sandbox.sandboxsdk.parameters as sdk_parameters
from sandbox.projects import resource_types
from sandbox.projects.common.juggler import base


class RedisSource(sdk_parameters.ResourceSelector):
    name = 'redis_source'
    description = 'Redis source'
    resource_type = resource_types.OTHER_RESOURCE
    required = True


class BuildRedisPackage(base.BaseBuildJugglerTask):
    """
    Build Redis from sources.
    """

    type = "BUILD_REDIS"

    client_tags = ctc.Tag.LINUX_PRECISE

    input_parameters = [RedisSource]

    def on_execute(self):
        with self.current_action("Downloading sources package"):
            srcfile = self.sync_resource(self.ctx[RedisSource.name])

        srcpath = self.abs_path("redis")
        with self.current_action("Extracting sources to '{0}'".format(srcpath)):
            if os.path.exists(srcpath):
                shutil.rmtree(srcpath)
            os.mkdir(srcpath)
            sdk_process.run_process(
                ["tar", "xf", srcfile, "--strip-components=1", "--directory", srcpath],
                log_prefix="extract"
            )
            with open(os.path.join(srcpath, 'src', 'version.h')) as stream:
                version = stream.read().split('"')[-2]

        with self.current_action("Building sources at '{0}'".format(srcpath)):
            with self._compiler_environ():
                sdk_process.run_process(
                    ["make", "-j", str(mp.cpu_count()), "MALLOC=libc"],
                    log_prefix="build", work_dir=srcpath
                )

        installpath = self.abs_path("root")
        with self.current_action("Installing into '{0}'".format(installpath)):
            if os.path.exists(installpath):
                shutil.rmtree(installpath)
            env = os.environ.copy()
            env["PREFIX"] = installpath
            sdk_process.run_process(
                ["make", "install"],
                log_prefix="install", environment=env, work_dir=srcpath
            )

        distpkg = self.abs_path("redis-{}.tar.gz".format(version))
        binpath = os.path.join(installpath, "bin")
        with self.current_action("Packaging distributive to '{0}'".format(distpkg)):
            sdk_process.run_process(
                ["tar", "-zcvf", distpkg, "-C", binpath] + os.listdir(binpath),
                log_prefix="pack"
            )

        self._make_resource(
            path=distpkg,
            resource_type=resource_types.REDIS_PACKAGE,
            description='redis {0}'.format(version),
            version=version)


__Task__ = BuildRedisPackage
