# -*- coding: utf-8 -*-
import os
import shutil
import tarfile
import contextlib

import sandbox.common.types.client as ctc

from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk.process import run_process

import sandbox.projects.common.build.parameters as build_params
from sandbox.projects import resource_types
from sandbox.projects.common.utils import svn_revision
from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask


class BuildRemClient(ArcadiaTask):
    """
        Build Rem Client python module
    """
    type = "BUILD_REM_CLIENT"

    client_tags = ctc.Tag.Group.LINUX

    input_parameters = build_params.get_arcadia_params()

    secret_owner = "MR-USERS"
    secret_attr = "pypi_credentials"
    python_binary = "python2.7"
    package_source_arc_project = ["quality", "remclient_py_distro"]
    client_source_arc_location = ["rem", "client", "remclient", "remclient.py"]
    constants_source_arc_location = ["rem", "client", "remclient", "constants.py"]
    tool_source_arc_location = ["rem", "client", "rem-tool"]

    def _prepare_arcadia_src(self):
        arcadia_src_dir = self.get_arcadia_src_dir()
        self.ctx['arcadia_revision'] = svn_revision(arcadia_src_dir)
        return arcadia_src_dir

    def _prepare_pypirc(self, directory):
        if not os.path.isdir(directory):
            make_folder(directory)
        pypirc_data = self.get_vault_data(self.secret_owner, self.secret_attr)
        with open(os.path.join(directory, ".pypirc"), "w") as rc_prn:
            rc_prn.write(pypirc_data)

    def do_execute(self):
        arcadia_src_dir = self._prepare_arcadia_src()
        build_dir = self.path("remclient_package")
        src_dir = os.path.join(build_dir, "remclient")
        shutil.copytree(os.path.join(arcadia_src_dir, *self.package_source_arc_project), build_dir)
        shutil.copy(os.path.join(arcadia_src_dir, *self.client_source_arc_location), src_dir)
        shutil.copy(os.path.join(arcadia_src_dir, *self.constants_source_arc_location), src_dir)
        shutil.copy(os.path.join(arcadia_src_dir, *self.tool_source_arc_location), os.path.join(src_dir, "rem_tool.py"))

        make_folder(os.path.join(build_dir, "root"))

        run_process(
            [self.python_binary, "setup.py", "install", "--root", os.path.join(build_dir, "root")],
            log_prefix="install",
            work_dir=build_dir
        )
        root_path = os.path.join(build_dir, "root")
        archive_path = "{}.tar.gz".format(root_path)
        with contextlib.closing(tarfile.open(archive_path, "w:gz", dereference=True)) as archive:
            for fname in os.listdir(root_path):
                archive.add(os.path.join(root_path, fname), arcname=fname)
        self.create_resource(
            description="distribution directory",
            resource_path=archive_path,
            resource_type=resource_types.REMCLIENT_PACKAGE
        )


__Task__ = BuildRemClient
