import json
import logging
import time
import random

from httplib import HTTPConnection

import sandbox.common.types.client as ctc

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.projects import resource_types

GENCFG_URL = 'api.gencfg.yandex-team.ru'
MAGIC_X_NUMBER = 5


def get_group_instances(group, tag='trunk'):
    conn = HTTPConnection(GENCFG_URL)
    conn.connect()
    conn.request('GET', '/' + tag + '/searcherlookup/groups/' + group + '/instances')
    return json.load(conn.getresponse())['instances']


def gen_maps(timestamp, b, x, z, nidx):
    tiers = set()

    x_instances = []
    for i in x:
        x_instances.append(i['hostname'] + ':' + str(i['port']))

    z_instances = []
    for i in z:
        z_instances.append(i['hostname'] + ':' + str(i['port']))

    replicas_b = {}
    for i in b:
        replicas_b.setdefault(i['shard_name'], set())
        replicas_b[i['shard_name']].add(i['hostname'] + ':' + str(i['port']))

    replicas_nidx = {}
    for i in nidx:
        replicas_nidx.setdefault(i['shard_name'], set())
        replicas_nidx[i['shard_name']].add(i['hostname'] + ':' + str(i['port']))

    random.seed(228)

    shards = {}
    replica_map = {'config': {'generation': timestamp, 'slot_size_gb': 20}, 'shards': shards}
    for s, r in replicas_b.iteritems():
        replica_list = list(r)
        random.shuffle(replica_list)

        replica_list_nidx = list(replicas_nidx[s])
        random.shuffle(replica_list_nidx)

        parts = s.split('-')
        tiers.add(parts[1])
        shard = parts[2] + '-' + parts[3]

        shards[s.replace('0000000000', timestamp)] = {'shard': shard, 'instances':
            [{'instance': i, 'policy': 'b'} for i in replica_list]\
            + [{'instance': i, 'policy': 'x'} for i in random.sample(x_instances, MAGIC_X_NUMBER)]\
            + [{'instance': i, 'policy': 'z'} for i in random.sample(z_instances, len(z_instances))]\
            + [{'instance': i, 'policy': 'd'} for i in replica_list_nidx]
                                                      }

    assert len(tiers) == 1
    replica_map['config']['tier'] = tiers.pop()

    return json.dumps(replica_map, indent=4)


class GroupB(SandboxStringParameter):
    name = 'group_b'
    description = 'Group B'
    default_value = 'MAN_WEB_TIER0_JUPITER_BASE_BUILD'


class GroupX(SandboxStringParameter):
    name = 'group_x'
    description = 'Group X'
    default_value = 'MAN_WEB_TIER0_JUPITER_BASE_BUILD_X'


class GroupZ(SandboxStringParameter):
    name = 'group_z'
    description = 'Group Z'
    default_value = 'MAN_WEB_TIER0_JUPITER_BASE_BUILD_Z'


class GroupNidx(SandboxStringParameter):
    name = 'group_nidx'
    description = 'Group NIDX'
    default_value = 'MAN_WEB_TIER0_JUPITER_BASE_BUILD_NIDX'


class Timestamp(SandboxStringParameter):
    name = 'timestamp'
    description = 'Timestamp'
    default_value = str(int(time.time()))


class GencfgTag(SandboxStringParameter):
    name = 'gencfg_tag'
    description = 'GencfgTag'
    default_value = 'trunk'


class BuildReplicamap(SandboxTask):
    type = 'BUILD_REPLICAMAP'
    client_tags = ctc.Tag.Group.LINUX
    input_parameters = [GroupB, GroupX, GroupZ, GroupNidx, Timestamp, GencfgTag]

    def _make_replicamap(self, data, descr):
        logging.info('creating replicamap file')

        filename = 'replicamap.json'

        with open(filename, 'w') as f:
            f.write(data)

        self.create_resource(
            description=descr,
            resource_path=filename,
            resource_type=resource_types.REPLICAMAP,
        )

    def on_execute(self):
        timestamp = self.ctx.get(Timestamp.name)
        replicamap = gen_maps(timestamp,
                              get_group_instances(self.ctx.get(GroupB.name), self.ctx.get(GencfgTag.name)),
                              get_group_instances(self.ctx.get(GroupX.name), self.ctx.get(GencfgTag.name)),
                              get_group_instances(self.ctx.get(GroupZ.name), self.ctx.get(GencfgTag.name)),
                              get_group_instances(self.ctx.get(GroupNidx.name), self.ctx.get(GencfgTag.name)),
                              )
        self._make_replicamap(replicamap, 'replicamap1')


__Task__ = BuildReplicamap
