import json
import random
import requests
import logging
import datetime

import sandbox.common.types.client as ctc

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.projects.resource_types import REPLICAMAP

GENCFG_API = 'http://api.gencfg.yandex-team.ru'
REPLICAMAP_FILENAME = 'replicamap.json'
MAGIC_X_NUMBER = 5
TIER_BUILD_SIZES = {'PlatinumTier0': 20, 'WebTier1': 520}


def get_shards(group, tag):
    shards = {}

    r = requests.get(GENCFG_API + '/' + tag + '/searcherlookup/groups/' + group + '/instances')
    if r.status_code == 200:
        for i in r.json()['instances']:
            shards.setdefault(i['shard_name'], [])
            shards[i['shard_name']].append(i['hostname'] + ':' + str(i['port']))

    logging.info('Get {0} instances from group {1}'.format(len(shards), group))
    return shards


def get_builders(policy, instances, sample_size=None):
    return [{'instance': i, 'policy': policy}
            for i in random.sample(instances, sample_size if sample_size else len(instances))]


def gen_map(group_prefix, tag, build_state, backup1_state, backup2_state, task_id):
    """
        MAGIC_ROLES:
        b - builder, _BUILD
        x - extended builder, _BUILD_X
        d - downloader, _NIDX
    """

    shards = {}
    replica_map = {
        'meta': {'sandbox_task_id': task_id},
        'shards': shards,
        'config': {
            'generation': build_state,
            'build_state': build_state,
            'backup1_state': backup1_state,
            'backup2_state': backup2_state,
        }
    }

    def get_int_state(state):
        return str(int(datetime.datetime.strptime(state, '%Y%m%d-%H%M%S').strftime("%s")))

    build_state_int = get_int_state(build_state)
    backup1_state_int = get_int_state(backup1_state)
    backup2_state_int = get_int_state(backup2_state)

    if group_prefix.endswith('_PIP'):
        b = get_shards(group_prefix.replace('_PIP', '') + '_BUILD', tag)
        x = get_shards(group_prefix.replace('_PIP', '') + '_BUILD_X', tag)
    else:
        b = get_shards(group_prefix + '_BUILD', tag)
        x = get_shards(group_prefix + '_BUILD_X', tag)

    nidx = get_shards(group_prefix + '_NIDX', tag)

    d = {}
    for s in set(nidx.keys()):
        d[s] = nidx.get(s, [])

    tiers = set()
    random.seed(228)
    for s, r in b.iteritems():
        parts = s.split('-')
        tiers.add(parts[1])
        shard = parts[2] + '-' + parts[3]

        shards[s.replace('0000000000', build_state_int)] = {
            'shard': shard,
            'instances': get_builders('b', r) +
                         get_builders('x', x['none'], min(MAGIC_X_NUMBER, len(x['none']))) +
                         get_builders('d', d.get(s, []))
        }

        if backup1_state != build_state:
            shards[s.replace('0000000000', backup1_state_int)] = {
                'shard': shard,
                'instances': get_builders('d', d.get(s, []))
            }
        if backup2_state != build_state and backup2_state != backup1_state:
            shards[s.replace('0000000000', backup2_state_int)] = {
                'shard': shard,
                'instances': get_builders('d', d.get(s, []))
            }

    assert len(tiers) == 1
    tier = tiers.pop()
    replica_map['config']['tier'] = tier
    slot_size = TIER_BUILD_SIZES.get(tier, 100)
    replica_map['config']['slot_size_gb'] = slot_size

    logging.info('Set slot_size_gb {0} for tier {1}'.format(slot_size, tier))
    return json.dumps(replica_map, indent=4)


class GroupPrefix(SandboxStringParameter):
    name = 'group_prefix'
    description = 'Gencfg topology group prefix'
    default_value = 'MAN_WEB_TIER0_JUPITER_BASE'


class GencfgTag(SandboxStringParameter):
    name = 'gencfg_tag'
    description = 'Gencfg topology version'
    default_value = 'trunk'


class YTState(SandboxStringParameter):
    name = 'yt_state'
    description = 'YT state (timestamp)'
    default_value = '19700101-000001'


class BuildState(YTState):
    name = 'build_state'
    description = 'Build state (timestamp)'
    required = True


class Backup1State(YTState):
    name = 'backup1_state'
    description = 'Backup1 state (timestamp)'
    required = True


class Backup2State(YTState):
    name = 'backup2_state'
    description = 'Backup2 state (timestamp)'
    required = True


class BuildReplicamap2(SandboxTask):
    type = 'BUILD_REPLICAMAP2'
    client_tags = ctc.Tag.Group.LINUX
    input_parameters = [GroupPrefix, GencfgTag, BuildState, Backup1State, Backup2State]

    def on_execute(self):
        replicamap = gen_map(self.ctx.get(GroupPrefix.name),
                             self.ctx.get(GencfgTag.name),
                             self.ctx.get(BuildState.name),
                             self.ctx.get(Backup1State.name),
                             self.ctx.get(Backup2State.name),
                             self.id
                             )

        with open(REPLICAMAP_FILENAME, 'w') as f:
            f.write(replicamap)

        self.create_resource(
            resource_path=REPLICAMAP_FILENAME,
            resource_type=REPLICAMAP,
            description='',
        )


__Task__ = BuildReplicamap2
