import json
import random
import requests
import logging
import datetime

import sandbox.common.types.client as ctc

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxIntegerParameter
from sandbox.projects.resource_types import REPLICAMAP

GENCFG_API = 'http://api.gencfg.yandex-team.ru'
REPLICAMAP_FILENAME = 'replicamap.json'
MAGIC_X_NUMBER = 5


def get_shards(group, tag):
    shards = {}

    r = requests.get(GENCFG_API + '/' + tag + '/searcherlookup/groups/' + group + '/instances')
    if r.status_code == 200:
        for i in r.json()['instances']:
            tier = 'none'
            for t in i['tags']:
                if t.startswith('a_tier_'):
                    tier = t.replace('a_tier_', '')
                    break

            shards.setdefault(
                i['shard_name'],
                {
                    'tier': tier,
                    'instances': []
                }
            )
            shards[i['shard_name']]['instances'].append(i['hostname'] + ':' + str(i['port']))

    logging.info('Get {0} instances from group {1}'.format(len(shards), group))
    return shards


def get_builders(policy, instances, sample_size=None):
    return [{'instance': i, 'policy': policy}
            for i in random.sample(instances, sample_size if sample_size else len(instances))]


def gen_map(build_group, buildx_group, nidx_group, tag, build_state, slot_size_gb, task_id):
    """
        MAGIC_ROLES:
        b - builder, _BUILD
        x - extended builder, _BUILD_X
        d - downloader, _NIDX
    """

    shards = {}
    replica_map = {
        'meta': {'sandbox_task_id': task_id},
        'shards': shards,
        'config': {
            'generation': build_state,
            'build_state': build_state,
        }
    }

    def get_int_state(state):
        return str(int(datetime.datetime.strptime(state, '%Y%m%d-%H%M%S').strftime("%s")))

    build_state_int = get_int_state(build_state)

    b = get_shards(build_group, tag)
    x = get_shards(buildx_group, tag)
    if nidx_group:
        d = get_shards(nidx_group, tag)
    else:
        d = {}

    tiers = set()
    random.seed(228)
    for s, r in b.iteritems():
        tiers.add(r['tier'])
        parts = s.split('-')
        shard = parts[2] + '-' + parts[3]

        shards[s.replace('0000000000', build_state_int)] = {
            'shard': shard,
            'instances': get_builders('b', r['instances']) +
                         get_builders('x', x['none']['instances'], min(MAGIC_X_NUMBER, len(x['none']['instances']))) +
                         get_builders('d', d.get(s, {}).get('instances', []))
        }

    assert len(tiers) == 1
    tier = tiers.pop()
    replica_map['config']['tier'] = tier
    replica_map['config']['slot_size_gb'] = slot_size_gb

    logging.info('Set slot_size_gb {0} for tier {1}'.format(slot_size_gb, tier))
    return json.dumps(replica_map, indent=4)


class GroupBuild(SandboxStringParameter):
    name = 'build_group'
    description = 'Gencfg builders group'
    required = True


class GroupBuildX(SandboxStringParameter):
    name = 'buildx_group'
    description = 'Gencfg builders extenders group'
    required = True


class GroupNidx(SandboxStringParameter):
    name = 'nidx_group'
    description = 'Gencfg nidxes group'


class GencfgTag(SandboxStringParameter):
    name = 'gencfg_tag'
    description = 'Gencfg topology version'
    default_value = 'trunk'


class BuildState(SandboxStringParameter):
    name = 'build_state'
    description = 'Build state'
    default_value = '19700101-000001'
    required = True


class SlotSizeGb(SandboxIntegerParameter):
    name = 'slot_size_gb'
    description = 'Slot size in GB'
    default_value = 10
    required = True


class BuildReplicamap3(SandboxTask):
    type = 'BUILD_REPLICAMAP3'
    client_tags = ctc.Tag.Group.LINUX
    input_parameters = [
        GroupBuild,
        GroupBuildX,
        GroupNidx,
        GencfgTag,
        BuildState,
        SlotSizeGb,
    ]

    def on_execute(self):
        replicamap = gen_map(
            self.ctx.get(GroupBuild.name),
            self.ctx.get(GroupBuildX.name),
            self.ctx.get(GroupNidx.name),
            self.ctx.get(GencfgTag.name),
            self.ctx.get(BuildState.name),
            self.ctx.get(SlotSizeGb.name),
            self.id
        )

        with open(REPLICAMAP_FILENAME, 'w') as f:
            f.write(replicamap)

        self.create_resource(
            resource_path=REPLICAMAP_FILENAME,
            resource_type=REPLICAMAP,
            description='',
        )


__Task__ = BuildReplicamap3
