import json
import random
import requests
import logging
import datetime

import sandbox.common.types.client as ctc

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.projects.resource_types import REPLICAMAP

GENCFG_API = 'http://api.gencfg.yandex-team.ru'
REPLICAMAP_FILENAME = 'replicamap.json'
BASE_NIDX_GROUPS = {
    'MSK_WEB_JUPITER_BASE_PRIEMKA_NIDX':
        ['MSK_WEB_BASE_PRIEMKA', 'MSK_WEB_BASE_PRIEMKA_BACKUP', 'MSK_WEB_BASE_PRIEMKA_ONDISK', ],
    'MSK_WEB_JUPITER_BASE_PRIEMKA_NIDX2':
        ['MSK_WEB_BASE_PRIEMKA', ],
    'MSK_WEB_JUPITER_BASE_R1_NIDX':
        ['MSK_WEB_BASE_R1', 'MSK_WEB_BASE_R1_BACKUP', 'MSK_WEB_BASE_R1_ONDISK', ],
    'MSK_WEB_JUPITER_BASE_R1_NIDX2':
        ['MSK_WEB_BASE_R1', ],
    'SAS_WEB_BASE_NIDX':
        ['SAS_WEB_PLATINUM_JUPITER_BASE', 'SAS_WEB_TIER0_JUPITER_BASE', 'SAS_WEB_TIER1_JUPITER_BASE', ],
    'SAS_WEB_BASE_NIDX2':
        ['SAS_WEB_PLATINUM_JUPITER_BASE'],
    'MAN_WEB_BASE_NIDX':
        ['MAN_WEB_PLATINUM_JUPITER_BASE', 'MAN_WEB_TIER0_JUPITER_BASE', 'MAN_WEB_TIER1_JUPITER_BASE',
         'MAN_WEB_PLATINUM_JUPITER_MMETA', 'MAN_WEB_TIER0_JUPITER_MMETA', 'MAN_WEB_TIER1_JUPITER_MMETA',
         'MAN_WEB_JUD_JUPITER_BASE_NIDX', 'MAN_WEB_CALLISTO_MMETA', 'MAN_WEB_CALLISTO_CAM_MMETA', ],
    'MAN_WEB_BASE_PIP_NIDX':
        ['MAN_WEB_PLATINUM_JUPITER_BASE_PIP', 'MAN_WEB_TIER0_JUPITER_BASE_PIP'],
    'MAN_WEB_MMETA_NIDX':
        ['MAN_WEB_MMETA'],
    'SAS_WEB_MMETA_NIDX':
        ['SAS_WEB_MMETA'],
    'SAS_WEB_SINK_MMETA_NIDX':
        ['SAS_WEB_SINK_MMETA'],
    'SAS_WEB_PLATINUM_BASE_BACKGROUND_NIDX':
        ['SAS_WEB_PLATINUM_BASE_BACKGROUND'],
    'SAS_WEB_MMETA_HAMSTER_NIDX':
        ['SAS_WEB_MMETA_HAMSTER'],
    'MAN_WEB_MMETA_HAMSTER_NIDX':
        ['MAN_WEB_MMETA_HAMSTER'],
    'MSK_WEB_MMETA_HAMSTER_NIDX':
        ['MSK_WEB_MMETA_HAMSTER'],
    'MAN_WEB_GOLDEN1B_MMETA_NIDX':
        ['MAN_WEB_GOLDEN1B_MMETA_TEST1'],
    'ST_TASK_GENCFG_697_BASE_NIDX':
        ['ST_TASK_GENCFG_697_BASE'],
    'VLA_WEB_BASE_NIDX':
        ['VLA_WEB_PLATINUM_JUPITER_BASE', 'VLA_WEB_TIER0_JUPITER_BASE', 'VLA_WEB_TIER1_JUPITER_BASE',
         'VLA_WEB_JUD_JUPITER_BASE_NIDX',
         'VLA_WEB_PLATINUM_JUPITER_TEST1_MMETA', 'VLA_WEB_PLATINUM_JUPITER_TEST2_MMETA',
         'VLA_WEB_GOLDEN1B_MMETA_TEST1', 'VLA_WEB_GOLDEN1B_MMETA_TEST1',
         'VLA_WEB_IO1_MMETA_TEST', 'VLA_WEB_IO2_MMETA_TEST', ],
    'VLA_WEB_MMETA_NIDX':
        ['VLA_WEB_MMETA'],
    'VLA_WEB_MMETA_HAMSTER_NIDX':
        ['VLA_WEB_MMETA_HAMSTER'],
    'VLA_WEB_BASE_PIP_NIDX':
        ['VLA_WEB_PLATINUM_JUPITER_BASE_PIP', 'VLA_WEB_TIER0_JUPITER_BASE_PIP'],
    'VLA_WEB_PLATINUM_BASE_BACKGROUND_NIDX':
        ['VLA_WEB_PLATINUM_BASE_BACKGROUND'],
    'VLA_WEB_MMETA_PIP_NIDX':
        ['VLA_WEB_MMETA_PIP', ],
}


def get_shards_c1(group, tag):
    hosts = {}
    r = requests.get(GENCFG_API + '/' + tag + '/searcherlookup/groups/' + group + '/instances')
    if r.ok:
        json_data = r.json()
        assert len(json_data['instances'])
        for i in json_data['instances']:
            hosts.setdefault(i['hostname'], i['port'])
    else:
        raise RuntimeError("Got {}, request {}".format(r.status_code, r.url))

    shards = {}

    for g in BASE_NIDX_GROUPS[group]:
        r = requests.get(GENCFG_API + '/' + tag + '/searcherlookup/groups/' + g + '/instances')
        if r.ok:
            json_data = r.json()
            assert len(json_data['instances'])
            for i in json_data['instances']:
                shards.setdefault(i['shard_name'], [])
                instance = i['hostname'] + ':' + str(hosts[i['hostname']])
                if instance not in shards[i['shard_name']]:
                    shards[i['shard_name']].append(instance)
        else:
            raise RuntimeError("Got {}, request {}".format(r.status_code, r.url))

    logging.info('Get {0} instances from group {1}'.format(len(shards), group))
    return shards


def get_builders(policy, instances, sample_size=None):
    return [{'instance': i, 'policy': policy}
            for i in sorted(random.sample(instances, sample_size if sample_size else len(instances)))]


def gen_map(group_prefix, tag, build_state, backup1_state, backup2_state, task_id):
    """
        MAGIC_ROLES:
        d - downloader, _NIDX
    """

    shards = {}
    replica_map = {
        'meta': {'sandbox_task_id': task_id},
        'shards': shards,
        'config': {
            'generation': build_state,
            'build_state': build_state,
            'backup1_state': backup1_state,
            'backup2_state': backup2_state,
        }
    }

    def get_int_state(state):
        return str(int(datetime.datetime.strptime(state, '%Y%m%d-%H%M%S').strftime("%s")))

    build_state_int = get_int_state(build_state)
    backup1_state_int = get_int_state(backup1_state)
    backup2_state_int = get_int_state(backup2_state)

    nidx = get_shards_c1(group_prefix, tag)

    d = {}
    for s in set(nidx.keys()):
        d[s] = nidx.get(s, [])

    random.seed(228)
    for s, r in d.iteritems():
        parts = s.split('-')
        shard = parts[2] + '-' + parts[3]

        shards[s.replace('0000000000', build_state_int)] = {
            'shard': shard,
            'instances': get_builders('d', d.get(s, []))
        }

        if backup1_state != build_state:
            shards[s.replace('0000000000', backup1_state_int)] = {
                'shard': shard,
                'instances': get_builders('d', d.get(s, []))
            }
        if backup2_state != build_state and backup2_state != backup1_state:
            shards[s.replace('0000000000', backup2_state_int)] = {
                'shard': shard,
                'instances': get_builders('d', d.get(s, []))
            }

    tier = 'PriemkaTier'
    replica_map['config']['tier'] = tier
    slot_size = 10
    replica_map['config']['slot_size_gb'] = slot_size

    logging.info('Set slot_size_gb {0} for tier {1}'.format(slot_size, tier))
    return json.dumps(replica_map, indent=4, sort_keys=True)


class GroupPrefix(SandboxStringParameter):
    name = 'group_prefix'
    description = 'Gencfg topology group prefix'
    default_value = 'MSK_WEB_JUPITER_BASE_PRIEMKA_NIDX'


class GencfgTag(SandboxStringParameter):
    name = 'gencfg_tag'
    description = 'Gencfg topology version'
    default_value = 'trunk'


class YTState(SandboxStringParameter):
    name = 'yt_state'
    description = 'YT state (timestamp)'
    default_value = '19700101-000001'


class BuildState(YTState):
    name = 'build_state'
    description = 'Build state (timestamp)'
    required = True


class Backup1State(YTState):
    name = 'backup1_state'
    description = 'Backup1 state (timestamp)'
    required = True


class Backup2State(YTState):
    name = 'backup2_state'
    description = 'Backup2 state (timestamp)'
    required = True


class BuildReplicamapC1R1(SandboxTask):
    type = 'BUILD_REPLICAMAPC1R1'
    client_tags = ctc.Tag.Group.LINUX
    input_parameters = [GroupPrefix, GencfgTag, BuildState, Backup1State, Backup2State]

    def on_execute(self):
        replicamap = gen_map(self.ctx.get(GroupPrefix.name),
                             self.ctx.get(GencfgTag.name),
                             self.ctx.get(BuildState.name),
                             self.ctx.get(Backup1State.name),
                             self.ctx.get(Backup2State.name),
                             self.id
                             )

        with open(REPLICAMAP_FILENAME, 'w') as f:
            f.write(replicamap)

        self.create_resource(
            resource_path=REPLICAMAP_FILENAME,
            resource_type=REPLICAMAP,
            description='',
        )


__Task__ = BuildReplicamapC1R1
