# -*- coding: utf-8 -*-

import os
import re
import logging

from sandbox.projects import resource_types
from sandbox.sandboxsdk import environments
from sandbox.projects.common import environments as common_environments
from sandbox.sandboxsdk import paths
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk import parameters as sp
from sandbox.projects.common import commands
from sandbox.projects.common.search import bugbanner

from sandbox.sandboxsdk.errors import SandboxTaskFailureError


class ArcadiaUrl(sp.SandboxArcadiaUrlParameter):
    name = 'checkout_arcadia_from_url'
    description = 'Arcadia url for report:'


class BuildProjects(sp.SandboxStringParameter):
    name = 'build_report_projects'
    description = 'Build report projects:'
    default_value = (
        'web+'
        'web2+'
        'web-touch-phone+'
        'web3+'
        'images+'
        'smart+'
        'touch+'
        'images-smart+'
        'video-touch+'
        'images2+'
        'people+'
        'sitesearch2+'
        'play+'
        'mobile-apps+'
        'mischandlers+'
        'images-touch+'
        'saved_copy'
    )
    required = True


class BuildReport(bugbanner.BugBannerTask, object):
    """
        Сборка общего пакета репорта - и core, и templates
    """
    type = 'BUILD_REPORT'

    environment = (
        environments.SvnEnvironment(),
        common_environments.SandboxJavaEnvironment('1.6.0'),
        common_environments.SandboxNodeNpmEnvironment('0.8.21a'),
        common_environments.SandboxYcssjsEnvironment('2.4.3'),
        common_environments.SandboxRubyEnvironment('1.8.7'),
    )

    input_parameters = (ArcadiaUrl, BuildProjects, )

    path_to_yreport = '/web/report/scripts/dev/yreport'

    additional_path_to_report_url = '/web/report'

    result_report_filename = 'yreport.tar'

    report_type = resource_types.REPORT_PACKAGE  # тип собираемого ресурса

    env_vars = {}

    def on_enqueue(self):
        self.env_vars = {}
        channel.task = self
        report_resource = self.create_resource(
            description=self.descr,
            resource_path=self.result_report_filename,
            resource_type=self.report_type,
            arch='any'
        )
        self.ctx['report_resource_id'] = report_resource.id

    def get_additional_yreport_parameters(self):
        """
            Получить список параметров, с которыми нужно запускать yreport
        """
        return ['-d', ]

    def get_additional_make_parameters(self):
        """
            Получить список параметров, с которыми нужно запускать make
        """
        return []

    def get_build_projects(self):
        """
            Получить список собираемых проектов
            @return: собираемые проекты в виде строки
        """
        return self.ctx.get('build_report_projects', '')

    def init_params(self):
        arcadia_url = self.ctx['checkout_arcadia_from_url']

        if not arcadia_url:
            raise SandboxTaskFailureError('Cannot execute task. "checkout_arcadia_from_url" parameter is not specified.')

        # url без @revision

        if '@' in arcadia_url:
            # fix after RMDEV-523
            arcadia_url = arcadia_url.split('@', 1)[0]

        arcadia_info = Arcadia.info(arcadia_url)
        self.arcadia_url = Arcadia.svn_url(arcadia_url)
        arcadia_revision = arcadia_info['commit_revision']
        self.ctx['arcadia_revision'] = arcadia_revision

        self.report_url = '%s/web/report' % self.arcadia_url

        self.workcopy_dir = paths.make_folder(self.abs_path('workcopy'))

        svn_conf = ''
        with open(os.path.join(os.environ["HOME"], '.subversion/config'), 'r') as f:
            svn_conf = f.read()
        with open(os.path.join(os.environ["HOME"], '.subversion/config'), 'w') as f:
            svn_conf = re.sub(r'^#\s*(use-commit-times\s*=\s*yes)', r'\1', svn_conf, flags=re.M)
            f.write(svn_conf)

    def set_build_report_env_vars(self):
        """
        Ставим перменные окружения перед сборкой репорта
        """
        for key, value in self.env_vars.iteritems():
            os.environ[key] = value

    def on_execute(self):
        self.add_bugbanner(bugbanner.Banners.Report)
        self.init_params()
        self.do_yreport()
        self.build_report()
        self.save_archive()

    def do_yreport(self):
        yreport_script_url = self.arcadia_url + self.path_to_yreport
        logging.info('Get yreport script from %s' % yreport_script_url)
        yreport_script = self.abs_path('yreport')
        Arcadia.export(yreport_script_url, yreport_script)

        build_report_projects = self.get_build_projects()
        logging.info('Build report projects %s with url %s.' % (build_report_projects, self.report_url))

        build_report_cmd = ['bash', yreport_script, ]
        # добавляем дополнительные параметры сборки
        build_report_cmd.extend(self.get_additional_yreport_parameters())
        self.ctx['version'] = self.get_report_version()  # self.arcadia_url
        build_report_cmd.extend([
            '-s', self.report_url,
            '-v', self.ctx['version'],
            build_report_projects,
            self.workcopy_dir
        ])
        # запускаем сборку
        logging.info(
            "VERSION=%s %s %s",
            self.get_report_version(self.arcadia_url),
            self.ctx['version'],
            self.workcopy_dir,
        )
        process.run_process(build_report_cmd, log_prefix='build_report_run_yreport', shell=True)

    def build_report(self):
        os.chdir(self.workcopy_dir)
        make_command = commands.get_make_name(jobs_by_cores=False) + self.get_additional_make_parameters()
        self.set_build_report_env_vars()
        process.run_process(make_command, log_prefix='run_make_report')

    def save_archive(self):
        logging.info('Find and save report archive.')
        file_name = os.listdir(os.path.join(self.workcopy_dir, 'build'))[0]
        report_archive_path = os.path.join(self.workcopy_dir, 'build', file_name)
        # получаем объект созданного ресурса
        report_resource_id = self.ctx.get('report_resource_id')
        if report_resource_id:
            # изменяем название у создаваемого файла, чтобы оно было как в собранном tar-е
            self.change_resource_basename(report_resource_id, os.path.basename(report_archive_path))
            report_resource = channel.sandbox.get_resource(report_resource_id)
        else:
            # ресурс не создан, создаём
            report_resource = self.create_resource(
                description=self.descr,
                resource_path=os.path.basename(report_archive_path),
                resource_type=self.report_type,
                arch='any',
            )
        paths.copy_path(report_archive_path, report_resource.path)
        channel.sandbox.set_resource_attribute(report_resource.id, 'version', self.ctx['version'])

        self.set_runtime_version(report_resource.id)
        self.set_info('Build report process completed successfully.')

    def arcadia_info(self, report_arcadia_url=None):
        if not report_arcadia_url:
            report_arcadia_url = self.ctx.get('report_arcadia_url')
        if report_arcadia_url:
            parsed_url = Arcadia.parse_url(report_arcadia_url)
            return self.ctx.get('arcadia_revision'), parsed_url.tag, parsed_url.branch
        else:
            return None, None, None

    def get_report_version(self, url=None):
        (rev, tag, branch) = self.arcadia_info(url)
        if tag is not None:
            tag = tag.replace('report/', '')
            return tag.replace('/', '.')
        if branch is not None:
            return 'branches.' + branch.replace('/', '.')
        else:
            return 'trunk'

    def set_runtime_version(self, res_id):
        # чтобы вычислить версию данных, нужно притянуть апач
        return


__Task__ = BuildReport
