# -*- coding: utf-8 -*-

import os
import shutil
import logging
import traceback
import time
import tarfile

import sandbox.projects.common.build.parameters as build_params
import sandbox.projects.common.constants as consts
import sandbox.projects.common.search.gdb as gdb
import sandbox.projects.common.utils as utils
# from sandbox import sdk2
from sandbox.common.types.client import Tag
from sandbox.projects import resource_types
from sandbox.projects.BuildSearch import BuildSearch
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.common.build.ArcadiaTask import ArcadiaTask
from sandbox.projects.common.nanny import nanny
from sandbox.projects.saas.common import resources
from sandbox.sandboxsdk import ssh
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError, SandboxTaskUnknownError, SandboxSvnIncorrectUrl
from sandbox.sandboxsdk.parameters import SandboxBoolParameter, SandboxStringParameter
from sandbox.sandboxsdk.paths import make_folder, remove_path
from sandbox.sandboxsdk.svn import Arcadia


class PYmakeParameters(SandboxStringParameter):
    name, description, default_value = 'ymake_parameters', 'ymake parameters (all -D)', ''


class PYmakeFlags(SandboxStringParameter):
    name, description, default_value = 'ymake_flags', 'ymake -W, -f flags', ''


class PYmakeMusl(SandboxBoolParameter):
    name, description, default_value = 'ymake_musl', 'ya make --musl', False


class PSaveMakePaths(SandboxBoolParameter):
    name, description, default_value = 'save_make_paths', 'Save make paths', False


class PWriteDepsTree(SandboxBoolParameter):
    name, description, default_value = 'write_deps_tree', 'Write dependencies tree', False


class PMakeGdb(SandboxBoolParameter):
    name, description, default_value = 'save_gdb_resource', 'Save gdb toolkit for release', False


def GPBuildBin(bin_name):
    class PBuildBin(SandboxBoolParameter):
        name = 'build_' + bin_name
        description = name

    return PBuildBin


def get_build_bins_parameters():
    bin_names = [
        'rtyserver', 'rtyserver_test', 'rtyserver_ut',
        'rtyserver_searchproxy', 'rtyserver_searchproxy_ut',
        'rtyserver_indexerproxy',
        'rtyserver_lf',
        'rtyserver_c',
        'rtyserver_hu',
        'rtyserver_common_proxy',
        'cproxy_avia',
        'cproxy_geosuggest',
        'violetta',
        'rtyserver_multitester',
        'rtyserver_deploy_manager',
        'standalone_indexer',
        'refresh_indexer',
        'distributor'
    ]
    return [GPBuildBin(bin_name) for bin_name in bin_names]


def get_build_rty_server_params():
    return build_params.get_arcadia_params() + [
        build_params.ClearBuild,
        build_params.BuildSystemPlain,
        build_params.BuildType,
        build_params.TargetPlatform
    ]


class BuildRtyServer(nanny.ReleaseToNannyTask, BuildSearch):
    type = 'BUILD_RTYSERVER'
    # Build on CORES24 finished in 1 hour and with CORES32 in 35 minutes
    client_tags = Tag.LINUX_PRECISE & Tag.CORES32
    input_parameters = get_build_rty_server_params() + get_build_bins_parameters() + [
        PYmakeParameters,
        PYmakeFlags,
        PYmakeMusl,
        PSaveMakePaths,
        PWriteDepsTree,
        build_params.UseArcadiaApiFuse,
        build_params.ThinLTO,
        PMakeGdb
    ]
    """
        TARGET_RESOURCE consists of three fields: (resource_type, target, target_path, filename)
        resource_type - type of the target resource
        target - title of the target
        target_path - final path of the built target resource. dirname of the target path is the path to checkout from arcadia
        filename is the name of target resource file.
    """
    TARGET_PATH_TO_NAME_MAP = {
        'saas/rtyserver_test/ut': 'rtyserver_ut',
        'saas/rtyserver_lf': 'rtyserver_lf',
        'saas/rtyserver_c': 'rtyserver_c',
        'saas/rtyserver_hu': 'rtyserver_hu',
        'saas/searchproxy': 'rtyserver_searchproxy',
        'saas/searchproxy/ut': 'rtyserver_searchproxy_ut',
        'saas/common_proxy': 'rtyserver_common_proxy',
        'saas/indexerproxy': 'rtyserver_indexerproxy',
        'saas/tools/mtester': 'rtyserver_multitester',
        'saas/deploy_manager': 'rtyserver_deploy_manager',
        'saas/tools/standalone_indexer': 'standalone_indexer',
        'saas/tools/refresh_indexer': 'refresh_indexer',
        'yweb/realtime/distributor/monolith': 'distributor'
    }

    TARGET_RESOURCES = (
        (resource_types.RTYSERVER_TEST, 'saas/rtyserver_test', 'rtyserver_test'),
        (resource_types.RTYSERVER_UNITTEST, 'saas/rtyserver_test/ut', 'ut_rtyserver'),
        (resource_types.RTYSERVER, 'saas/rtyserver', 'rtyserver'),
        (resource_types.RTYSERVER_LF, 'saas/rtyserver_lf', 'rtyserver'),
        (resources.RTYSERVER_C, 'saas/rtyserver_c', 'rtyserver'),
        (resources.RTYSERVER_HU, 'saas/rtyserver_hu', 'rtyserver'),
        (resource_types.RTYSERVER_SEARCHPROXY, 'saas/searchproxy', 'searchproxy'),
        (resource_types.RTYSERVER_SEARCHPROXY_UT, 'saas/searchproxy/ut', 'ut_searchproxy'),
        (resource_types.RTYSERVER_COMMON_PROXY, 'saas/common_proxy', 'common_proxy'),
        (resource_types.RTYSERVER_COMMON_PROXY_GEOSUGGEST, 'saas/services/geosuggest/cproxy', 'cproxy_geosuggest'),
        (resource_types.RTYSERVER_COMMON_PROXY_VIOLETTA, 'saas/services/violetta/cproxy', 'violetta'),
        (resource_types.RTYSERVER_INDEXER_PROXY, 'saas/indexerproxy', 'indexerproxy'),
        (resource_types.RTYSERVER_MULTITESTER, 'saas/tools/mtester', 'mtester'),
        (resource_types.RTYSERVER_UTILS_DEPLOY_MANAGER, 'saas/deploy_manager', 'deploy_manager'),
        (resources.SAAS_UTIL_STANDALONE_INDEXER, 'saas/tools/standalone_indexer', 'standalone_indexer'),
        (resources.RefreshIndexerBinary, 'saas/tools/refresh_indexer', 'refresh_indexer'),
        (resource_types.DISTRIBUTOR, 'yweb/realtime/distributor/monolith', 'monolith')
    )

    VIRTUAL_TARGETS = (
        (resource_types.RTYSERVER_TEST, 'rtyserver_test', 'saas/rtyserver_test'),
        (resource_types.RTYSERVER_UNITTEST, 'rtyserver_ut', 'saas/rtyserver_test/ut'),
        (resource_types.RTYSERVER, 'rtyserver', 'saas/rtyserver'),
        (resource_types.RTYSERVER_LF, 'rtyserver_lf', 'saas/rtyserver_lf'),
        (resources.RTYSERVER_C, 'rtyserver_c', 'saas/rtyserver_c'),
        (resources.RTYSERVER_HU, 'rtyserver_hu', 'saas/rtyserver_hu'),
        (resource_types.RTYSERVER_SEARCHPROXY, 'rtyserver_searchproxy', 'saas/searchproxy'),
        (resource_types.RTYSERVER_SEARCHPROXY_UT, 'rtyserver_searchproxy_ut', 'saas/searchproxy/ut'),
        (resource_types.RTYSERVER_COMMON_PROXY, 'rtyserver_common_proxy', 'saas/common_proxy'),
        (resource_types.RTYSERVER_COMMON_PROXY_GEOSUGGEST, 'cproxy_geosuggest', 'saas/services/geosuggest/cproxy'),
        (resource_types.RTYSERVER_COMMON_PROXY_VIOLETTA, 'violetta', 'saas/services/violetta/cproxy'),
        (resource_types.RTYSERVER_INDEXER_PROXY, 'rtyserver_indexerproxy', 'saas/indexerproxy'),
        (resource_types.RTYSERVER_MULTITESTER, 'rtyserver_multitester', 'saas/tools/mtester'),
        (resource_types.RTYSERVER_UTILS_DEPLOY_MANAGER, 'rtyserver_deploy_manager', 'saas/deploy_manager'),
        (resources.SAAS_UTIL_STANDALONE_INDEXER, 'standalone_indexer', 'saas/tools/standalone_indexer'),
        (resources.RefreshIndexerBinary, 'refresh_indexer', 'saas/tools/refresh_indexer'),
        (resource_types.DISTRIBUTOR, 'distributor', 'yweb/realtime/distributor/monolith')
    )

    LIB_TARGETS = (
        ('libdatasrc-ftds.so.1', 'ysite/datasrc/ftds'),
        ('libdatasrc-raw_ds.so.1', 'ysite/datasrc/raw_ds'),
        ('libdatasrc-webds.so.1', 'ysite/datasrc/webds'),
    )

    LOCAL_BIN_DIR = 'binaries'
    libs_dir_name = 'rtyserver_client_libs'
    rtyserver_configs_res_dir = 'rtyserver_configs'

    release_dir = ''

    required_ram = 80000

    execution_space = 200000

    def on_enqueue(self):
        ArcadiaTask.on_enqueue(self)
        if self.ctx['build_bundle']:
            return
        self.do_on_enqueue()
        dprio = 0
        try:
            dprio = int(self.ctx.get('dprio', 0))
        except:
            pass
        self.score = dprio

    def mark_stage_start(self, stage):
        self.ctx['__STAGES'][stage + '_started'] = time.time()

    def mark_stage_finish(self, stage):
        self.ctx['__STAGES'][stage + '_finished'] = time.time()

    def do_on_enqueue(self):
        if self.ctx.get('save_make_paths') and self.ctx.get('build_system') in (consts.YMAKE_BUILD_SYSTEM, consts.DISTBUILD_BUILD_SYSTEM):
            return

        for resource_type, target_path, filename in self.TARGET_RESOURCES:
            target = self.get_target_name(target_path)
            if self._target_enabled(target):

                resource_path = os.path.join(self.LOCAL_BIN_DIR, filename)
                resource_name = '%s (%s)' % (self.descr, filename)
                resource = self._create_resource(resource_name,
                                                 resource_path,
                                                 resource_type,
                                                 arch=self.arch)
                self.ctx['%s_resource_id' % target] = resource.id

        if self._target_enabled(self.libs_dir_name):
            libs = self._create_resource('RTYServer client libs. ' + self.descr, self.libs_dir_name,
                                         resource_types.RTYSERVER_CLIENT_LIBS, arch=self.arch)
            self.ctx['%s_resource_id' % self.libs_dir_name] = libs.id

        # create resource with rtyserver configs for tester
        if self.ctx['build_rtyserver_test']:
            configs = self._create_resource('RTYServer configs, ' + self.descr, self.rtyserver_configs_res_dir,
                                            resource_types.RTYSERVER_CONFIGS, arch=self.arch)
            self.ctx['%s_resource_id' % self.rtyserver_configs_res_dir] = configs.id
            if self._need_local_saas():
                bin_tar_gz = self._create_resource('RTYServerTest binary tar.gz, ' + self.descr, os.path.join(self.LOCAL_BIN_DIR, 'rtyserver_test.tar.gz'),
                                                    resources.RTYSERVER_TEST_TAR_GZ, arch=self.arch)
                self.ctx['rtyserver_test_tar_gz_id'] = bin_tar_gz.id

        self.ctx["don_t_release"] = []

        self.set_info('task enqueued for execution')

    def _run_ymake_build(self, enabled_targets, build_type, clear_build):
        self.release_dir = self.abs_path(self.LOCAL_RELEASE_DIR)

        if self.ctx.get('save_make_paths'):
            make_folder(self.release_dir)
            os.chdir(self.release_dir)
            ya_path = os.path.join(self.arcadia_src_dir, 'ya')
            ymake_cmd = [ya_path, 'make',
                         '-j0', '--dump-graph',
                         '--source-root=' + self.arcadia_src_dir,
                         '--results-root=' + self.release_dir,
                         ]
            ymake_cmd.append('--build-dir=' + self.abs_path('tmp'))
            if self.ctx.get('ymake_parameters', '').strip():
                ymake_cmd.extend(self.ctx.get('ymake_parameters').strip().split())
            if self.ctx.get('ymake_musl'):
                ymake_cmd.append('--musl')
            if not self.ctx.get('write_deps_tree'):
                deps = sdk.dump_targets_deps(self.arcadia_src_dir, enabled_targets)
                self.save_dir_list_resource(deps, 'all')
                self.ctx['deps_count_all'] = len(deps)
            else:
                targets = [(os.path.basename(os.path.join(self.arcadia_src_dir, t)),
                            t)
                           for t in enabled_targets]
                for targ in targets:
                    deps = sdk.dump_targets_deps(self.arcadia_src_dir, [targ[1], ])
                    self.save_dir_list_resource(deps, targ[0])
                    self.ctx['deps_count_' + targ[0]] = len(deps)
            self.ctx['task_done'] = True
        else:
            user_flags = sdk.parse_flags(self.ctx.get('ymake_parameters', ''))
            no_d_flags = self.ctx.get('ymake_flags', '')
            musl = self.ctx.get('ymake_musl')
            if no_d_flags:
                try:
                    no_d_flags = 'CFLAGS=' + no_d_flags.strip().strip(';')
                    no_d_flags = no_d_flags.split(';')
                    no_d_flags = dict(
                        (f.split('=', 1)[0], '' + f.split('=', 1)[1] + '') for f in no_d_flags[:] if '=' in f
                    )
                except Exception as e:
                    logging.error('while parsing flags, error: %s' % e)

            flags = {}
            if user_flags:
                flags.update(user_flags)
            if no_d_flags:  # and 'flags' in self.descr:
                flags.update(no_d_flags)
                logging.info('all_flags: %s' % flags)

            platform = self.ctx.get(consts.TARGET_PLATFORM_KEY) or None

            build_system = self.ctx.get('build_system', consts.YMAKE_BUILD_SYSTEM)
            sdk.do_build(
                build_system,
                self.arcadia_src_dir,
                enabled_targets,
                clear_build=clear_build,
                build_type=build_type,
                def_flags=flags,
                results_dir=self.release_dir,
                target_platform=platform,
                timeout=7200,
                thinlto=utils.get_or_default(self.ctx, build_params.ThinLTO),
                musl=musl
            )

    def _build(self, enabled_targets):
        clear_build = self.ctx.get(consts.CLEAR_BUILD_KEY)
        build_type = self.ctx.get(consts.BUILD_TYPE_KEY)
        self._run_ymake_build(enabled_targets, build_type, clear_build)

    def _need_local_saas(self):
        return (
            'saas' in self.ctx[consts.ARCADIA_URL_KEY] and self.arch == 'linux' and
            self.ctx['build_rtyserver_test'] and
            self.ctx['build_type'] == 'release' and
            'builds' in self.descr
        )

    def _get_arcadia_ctx(self):
        # Sorry about that, this task must be rewritten
        try:
            arcadia_ctx = self._get_src_dir()
        except SandboxSvnIncorrectUrl as e:
            logging.error(traceback.format_exc())
            url = self.ctx[consts.ARCADIA_URL_KEY]
            rev = Arcadia.parse_url(url).revision
            if rev:
                url = url.replace('@' + str(rev), '')
                rev1 = Arcadia.get_revision(url)
                if (int(rev) - int(rev1)) < 2000:
                    raise SandboxTaskUnknownError('cannot find revision %s, found %s' % (rev, rev1))
            raise SandboxSvnIncorrectUrl("cannot get svn: " + str(e))
        return arcadia_ctx

    def do_execute(self):
        self.ctx['queue_time_min'] = round((self.timestamp_start - self.timestamp)/60, 1)
        self.ctx['__STAGES'] = self.ctx.get('__STAGES', {})
        os.chdir(self.abs_path())
        binaries_dir = os.path.join(self.abs_path(), self.LOCAL_BIN_DIR)
        make_folder(binaries_dir)

        if self.ctx.get(PMakeGdb.name):
            gdb.append_to_release(self)

        self.mark_stage_start('svn')
        arcadia_ctx = self._get_arcadia_ctx()
        with arcadia_ctx as arcadia_src_dir:

            self.arcadia_src_dir = arcadia_src_dir
            self.prepare_svn()

            self.mark_stage_finish('svn')

            # build dependencies tree
            if self.ctx.get('write_deps_tree'):
                # todo something here
                pass

            # create resource with rtyserver configs for tester
            if self.ctx['build_rtyserver_test']:
                configs_dir = os.path.join(self.abs_path(), self.rtyserver_configs_res_dir)
                if os.path.exists(configs_dir):
                    remove_path(configs_dir)
                configs_arc_dir = os.path.join(self.arcadia_src_dir, 'saas/rtyserver_test/func/configs')
                try:
                    shutil.copytree(configs_arc_dir, configs_dir)
                except:
                    raise SandboxTaskFailureError('Error with copy : file %s' % configs_arc_dir)

            #
            self.fill_system_info()

            enabled_targets = []
            for _, target, target_path in self.VIRTUAL_TARGETS:
                if self._target_enabled(target):
                    enabled_targets.append(target_path)

            #
            self.mark_stage_start('build')
            self._build(enabled_targets)
            self.mark_stage_finish('build')
            if self.ctx.get('task_done', False):
                return

            for _, target_path, filename in self.TARGET_RESOURCES:
                target = self.get_target_name(target_path)
                logging.info('Check target %s, path %s, filename %s' % (target, target_path, filename))
                if self._target_enabled(target):
                    # build resource
                    build_dir = self.abs_path(os.path.join(self.release_dir, target_path))

                    os.chdir(build_dir)  # check abs path
                    logging.info('read_resource: %s, %s', target, self._target_resource_id(target))
                    resource_path = channel.sandbox.get_resource(self._target_resource_id(target)).path

                    target_file = os.path.join(target_path, filename)

                    ready_file = self.abs_path(os.path.join(self.release_dir, target_file))
                    try:
                        shutil.copy(ready_file, resource_path)
                        logging.info('Target file: %s copy to: %s' % (ready_file, resource_path))
                        if target == 'rtyserver_test' and self._need_local_saas():
                            tar_gz_resource_id = self.ctx.get('rtyserver_test_tar_gz_id')
                            path = channel.sandbox.get_resource(tar_gz_resource_id).path
                            logging.info('Create tar.gz: %s' % path)
                            with tarfile.open(name=path, mode='w:gz') as tar:
                                logging.info('Add %s as %s' % (ready_file, filename))
                                tar.add(name=ready_file, arcname=filename)
                            logging.info('Target file: %s added to archive: %s' % (ready_file, path))

                    except IOError as e:
                        raise SandboxTaskFailureError('Error with copy : %s, file %s' % (repr(e), ready_file))

            if self._target_enabled('ut_rtyserver'):
                pass

            if self._target_enabled(self.libs_dir_name):
                libs_dir = os.path.join(self.abs_path(), self.libs_dir_name)
                make_folder(libs_dir)

                for target, target_path in self.LIB_TARGETS:
                    build_dir = self.abs_path(os.path.join(self.release_dir, target_path))
                    os.chdir(build_dir)  # check abs path
                    shutil.copy(self.abs_path(os.path.join(build_dir, target)), libs_dir)

            revision = self.ctx['arcadia_revision']
            svn_url = self.ctx[consts.ARCADIA_URL_KEY]
            if self._need_local_saas():
                self.commit_local_saas()
            if 'saas' in svn_url and self.arch == 'linux' and False:
                if not svn_url.endswith(str(revision)):
                    svn_url = svn_url + '@' + str(revision)
                try:
                    self.do_logs(svn_url, revision)
                except Exception as e:
                    logging.info('cannot commit changelog, error %s' % e)

            # cleanup
            self.clean_release_dir(self.release_dir)

            self.set_info('build completed')
            self.ctx['completed'] = True
            self.ctx['exec_time_min'] = round((int(time.time()) - self.timestamp_start)/60, 1)

            parent_resources = self.ctx.get("parent_resource_ids", {})
            if parent_resources:
                for resource_type, target_path, filename in self.TARGET_RESOURCES:
                    target = self.get_target_name(target_path)
                    if self._target_enabled(target) and target in parent_resources:
                        resource_path = channel.sandbox.get_resource(self._target_resource_id(target)).path
                        self.save_parent_task_resource(resource_path, parent_resources.get(target))

    def save_dir_list_resource(self, lines, targ):
        res_path = self.abs_path("dir_list_" + targ)
        with open(res_path, 'w') as f:
            f.write('\n'.join(lines))
        res = self.create_resource('dir list for ' + targ, res_path, resource_types.DIR_LIST)
        return res

    def commit_local_saas(self):
        try:
            self.mark_all_ready()
            with ssh.Key(self, 'RTYSERVER-ROBOT', 'ssh_key'):
                local_saas_path = Arcadia.checkout('arcadia:/arc/trunk/arcadia/saas/rtyserver_test/local_saas/sandbox-data/', self.path('local_saas'))
                inc_path = os.path.join(local_saas_path, 'rtyserver_test.resource.inc')
                old_id = 0
                with open(inc_path, 'r') as lp:
                    for l in lp.readlines():
                        if 'FROM_SANDBOX(' in l:
                            old_id = l.replace('FROM_SANDBOX(', '').replace('OUT rtyserver_test)', '').strip()
                            logging.info('old resource id=%s' % old_id)
                new_id = self.ctx['rtyserver_test_tar_gz_id']
                if new_id > int(old_id):
                    channel.sandbox.set_resource_attribute(new_id, 'ttl', 100)
                    with open(inc_path, 'r') as f:
                        fcont = f.read()
                    with open(inc_path, 'w') as fo:
                        fo.write(fcont.replace(str(old_id), str(new_id)))
                    Arcadia.commit(inc_path,
                                   'renew rtyserver_test binary from branch, task %s' % self.id,
                                   'saas-robot')
        except Exception as e:
            self.set_info('cannot commit to local_saas: %s' % e)
            logging.error(logging.error(traceback.format_exc()))

    def do_logs(self, svn_url, revision):

        revision_msg, author = None, ''
        revision_msg = self.ctx.get('comment')
        if not revision_msg:
            revision_msg, author = self.get_revision_info(svn_url, revision)

        def should_auto_release(desc, mess):
            if not ('RTYSERVER_MAIN' in desc or 'RTYSERVER_INDEXERPROXY_LINUX' in desc
                    or 'RTYSERVER_SEARCHPROXY_LINUX' in desc):
                return False
            if '[release]' in mess or '[release ]' in mess:
                return True
            return ('rtyserver]' in mess and 'RTYSERVER_MAIN' in desc) or \
                ('iproxy]' in mess and 'INDEXER' in desc) or \
                ('sproxy]' in mess and 'SEARCH' in desc)

        if ('[release' in revision_msg.lower() or self.ctx.get('release_after', False)) and should_auto_release(self.descr.upper(), revision_msg):
            logging.info('doing release...')
            self.mark_all_ready()
            branch = svn_url[svn_url.find('saas/')+5:svn_url.find('saas/')+12]
            # self.ctx['branch'] = branch
            if not author:
                revision_msg, author = self.get_revision_info(svn_url, revision)
            channel.sandbox.set_task_context_value(self.id, 'arcadia_revision', self.ctx['arcadia_revision'])
            # rel = self.create_release(self.id, subject=branch, comments=revision_msg.lower(), changelog=[], addresses_to=[author], addresses_cc=[author])
            try:
                self.on_release({'release_subject': branch, 'release_status': '',
                                 'release_comment': revision_msg.lower()})
            except Exception as e:
                channel.sandbox.send_email(author, '', 'error in release',
                                           'release not created, please do it by hand, task %s, error %s' %
                                           (self.http_url(), e))

    def prepare_svn(self):
        if not self.ctx.get('arcadia_revision', 0):
            revision_info = None
            for i in range(5):
                try:
                    revision_info = Arcadia.info(self.ctx[consts.ARCADIA_URL_KEY])
                    break
                except Exception as e:
                    logging.warning('cannot get svn info, try %s, error %s' % (i, e))
                    if i == 4:
                        raise
            self.ctx['arcadia_revision'] = revision_info['entry_revision']

        arcadia_patch = self.ctx.get('arcadia_patch')
        if arcadia_patch.strip() and arcadia_patch.strip() not in ('None', ):
            Arcadia.apply_patch(self.arcadia_src_dir, self.ctx.get('arcadia_patch'), self.abs_path())

    def get_revision_info(self, svn_url, revision):
        revision_msg, author = '', ''
        if revision > 0:
            try:
                svn_url_info = Arcadia.log(svn_url, revision, revision)
                revision_msg = svn_url_info[0]['msg']
                author = svn_url_info[0]['author']
            except Exception as e:
                logging.info('cannot get revision info, error %s' % e)
        return revision_msg, author

    def mark_all_ready(self):
        for resource in self.list_resources():
            if resource.type.name == 'TASK_LOGS':
                continue
            logging.info('found resource %s' % str(resource))
            if not resource.is_ready():
                try:
                    resource.mark_ready()
                except Exception as e:
                    raise SandboxTaskUnknownError('Problems with resource %s, error %s' % (str(resource.id), e))

    def on_finish(self):
        if not self.ctx.get('completed', False):
            return
        self.mark_stage_start('share')
        self.mark_all_ready()
        self.mark_stage_finish('share')

    def on_release(self, additional_parameters):
        channel.task = self
        for resource in self.list_resources():
            if resource.type.releasable:
                if resource.id in self.ctx.get('don_t_release', []):
                    continue
                if not resource.is_ready():
                    resource.mark_ready()
        svn_url = self.ctx[consts.ARCADIA_URL_KEY]
        svn_branch = 'trunk' if 'trunk' in svn_url else svn_url.split('saas/')[-1]
        additional_parameters['release_subject'] = svn_branch + ' ' + additional_parameters.get('release_subject', '')
        if additional_parameters['release_status'] == 'stable' or True:
            nanny.ReleaseToNannyTask.on_release(self, additional_parameters)
        else:
            logging.info('releasing %s' % additional_parameters)
            nanny_token = self.get_vault_data('RTYSERVER-ROBOT', 'nanny_token')
            logging.info('token len %s' % len(nanny_token))
            client = nanny.NannyClient('http://nanny.yandex-team.ru/', nanny_token)
            info = client.get_service_info_attrs('saas_cloud_base_prestable')
            logging.info('%s' % info)


__Task__ = BuildRtyServer
