# -*- coding: utf-8 -*-

import logging
import os

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.parameters import SandboxStringParameter
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.paths import make_folder

from sandbox.projects import resource_types


class PDataListPath(SandboxStringParameter):
    name, description = 'data_list_path', 'Path to svn file with json'
    default_value = 'svn+ssh://arcadia.yandex.ru/arc/trunk/arcadia/robot/library/oxygen/config/files/RecipeConfig.pb.txt'


class PStopList(SandboxStringParameter):
    name, description = 'stop_list', 'files to not include in resource'
    default_value = 'j-owners-spm.lst, mirrors.trie, hostfactors.dat, hostfactors.idx, geoa.c2p,'\
                    + ' reg_hostfactors.dat, reg_hostfactors.idx'


class BuildRtyserverModels(SandboxTask):
    """
        build rtyserver models folder
    """
    type = 'BUILD_RTYSERVER_MODELS'

    input_parameters = [PDataListPath,
                        PStopList,
                        ]

    def get_files_paths(self, raw_list, stop_paths):
        excluded = {}
        files_paths = {}

        for fsect in raw_list:
            fname, rpath = '', ''
            for line in fsect.split('\n'):
                prop = line.strip().strip('#').split()[0] if line.strip() else ''
                if prop == 'Name:':
                    fname = line.strip().split()[-1].strip('"\'')
                    continue
                if prop == 'RsyncPath:':
                    rpath = line.strip().split()[-1].strip('"\'')
                    continue
                if fname and rpath:
                    break
            if fname and rpath:
                if fname not in stop_paths:
                    files_paths[fname] = rpath
                else:
                    excluded[fname] = rpath
            else:
                logging.error('cannot parse section %s' % fsect)

        logging.info('excluded files: %s' % excluded.keys())
        return files_paths

    def on_execute(self):
        data_list_path = self.ctx.get('data_list_path')
        if 'svn+ssh' not in data_list_path and 'arcadia' not in data_list_path:
            logging.error('incorrect or unsupported path: %s' % data_list_path)
            raise SandboxTaskFailureError('incorrect or unsupported path: %s' % data_list_path)

        Arcadia.export(self.ctx.get('data_list_path'), self.log_path('files_list'))

        with open(self.log_path('files_list'), 'r') as f:
            raw_list = f.read().split('\nFiles')
        stop_paths = [p.strip() for p in self.ctx.get('stop_list').split(',')]

        files_paths = self.get_files_paths(raw_list, stop_paths)
        logging.info('files to download: %s' % files_paths)

        models_path = self.abs_path('models')
        make_folder(models_path)
        for fname, frpath in files_paths.items():
            dest_path = os.path.join(models_path, fname)
            self.remote_copy(frpath, dest_path, create_resource=False)
        mod_res = self.create_resource('models folder: ' + self.descr, models_path, resource_types.RTYSERVER_MODELS)
        self.mark_resource_ready(mod_res.id)

    def arcadia_info(self):
        return None, 'nobranch', None


__Task__ = BuildRtyserverModels
