# -*- coding: utf-8 -*-

import os
import logging
import traceback

import sandbox.common.types.client as ctc

from sandbox.sandboxsdk.parameters import SandboxBoolParameter
from sandbox.projects.common.build.CommonBuildTask import CommonBuildTask
import sandbox.projects.common.build.parameters as build_params
import sandbox.projects.common.constants as consts
from sandbox.projects import resource_types
from sandbox.sandboxsdk.paths import make_folder, copy_path
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.errors import SandboxTaskFailureError, SandboxTaskUnknownError, SandboxSvnIncorrectUrl
from sandbox.projects.common.nanny import nanny
from sandbox.projects.BuildRtyserverUtils import ferryman_resources
from sandbox.projects.saas.common import resources


def GPBuildBin(bin_name):
    class PBuildBin(SandboxBoolParameter):
        name = 'build_' + bin_name
        description = name

    return PBuildBin


def get_build_bins_parameters():
    bin_names = ['dsclient', 'printkeys', 'rtyserver_emulator', 'evlogdump',
                 'config_patcher', 'deploy_manager',
                 'ferryman_server', 'ferryman_worker', 'ferryman_json_worker', 'ferryman_geo_worker',
                 'ferryman_cooker', 'standalone_indexer', 'get_standalone_indexer',
                 'distributor', 'roads_graph', 'common_proxy',
                 'cproxy_avia', 'cproxy_geosuggest', 'violetta', 'saas_push']
    return [GPBuildBin(bin_name) for bin_name in bin_names]


class BuildRtyserverUtils(nanny.ReleaseToNannyTask, CommonBuildTask):
    """
        build rtyserver small utils
    """

    type = 'BUILD_RTYSERVER_UTILS'

    client_tags = ctc.Tag.Group.LINUX

    input_parameters = build_params.get_common_build_params() +\
                       get_build_bins_parameters()

    execution_space = 60000

    TARGET_PATH_TO_NAME_MAP = {
        'yweb/crawlrank/buildall/': 'distributor'
    }

    TARGET_RESOURCES = (
        (resource_types.RTYSERVER_UTILS_PRINTKEYS, 'tools/printkeys/printkeys'),
        (resource_types.RTYSERVER_UTILS_RTYSERVER_EMULATOR, 'saas/tools/rtyserver_emulator/rtyserver_emulator'),
        (resource_types.RTYSERVER_UTILS_EVLOGDUMP, 'search/tools/evlogdump/evlogdump'),
        (resource_types.RTYSERVER_UTILS_CONFIG_PATCHER, 'search/tools/config_patcher/config_patcher'),
        (resource_types.RTYSERVER_UTILS_DEPLOY_MANAGER, 'saas/deploy_manager/deploy_manager'),
        (resource_types.RTYSERVER_UTILS_DISTRIBUTOR_CLIENT, 'saas/tools/distcl/distcl'),
        (resource_types.RTYSERVER_UTILS_ROADS_GRAPH, 'saas/tools/roads_graph/roads_graph'),
        (resource_types.RTYSERVER_UTILS_TEST_MODEL, 'saas/tools/test_model/test_model'),
        (resource_types.SAAS_FERRYMAN_SERVER, 'yweb/querydata/querydata_indexer_saas/ferryman/server/ferryman_server'),
        (resource_types.SAAS_FERRYMAN_WORKER, 'yweb/querydata/querydata_indexer_saas/ferryman/worker/ferryman_worker'),
        (ferryman_resources.SAAS_FERRYMAN_JSON_WORKER, 'yweb/querydata/querydata_indexer_saas/ferryman/json_worker/ferryman_json_worker'),
        (ferryman_resources.SAAS_FERRYMAN_COOKER, 'yweb/querydata/querydata_indexer_saas/ferryman/cooker/ferryman_cooker'),
        (ferryman_resources.SAAS_FERRYMAN_GEO_WORKER, 'yweb/querydata/querydata_indexer_saas/ferryman/geo_worker/ferryman_geo_worker'),
        (resources.SAAS_UTIL_STANDALONE_INDEXER, 'saas/tools/standalone_indexer/standalone_indexer'),
        (resources.SAAS_UTIL_GET_STANDALONE_INDEXER, 'saas/tools/devops/get_standalone_indexer/get_standalone_indexer'),
        (resource_types.RTYSERVER_COMMON_PROXY, 'saas/common_proxy/common_proxy'),
        (resource_types.RTYSERVER_COMMON_PROXY_GEOSUGGEST, 'saas/services/geosuggest/cproxy/cproxy_geosuggest'),
        (resource_types.RTYSERVER_COMMON_PROXY_VIOLETTA, 'saas/services/violetta/cproxy/violetta'),
        (resources.SaasPushClient, 'saas/tools/saas_push/saas_push'),
        (None, 'yweb/crawlrank/buildall/'),
    )

    TARGET_LIBS = {
    }

    TARGETS_MULTIPLE = {
        'distributor': ['yweb/crawlrank/datastore/src/bdblds',
                        'yweb/crawlrank/controller/server/controller',
                        'yweb/realtime/distributor/server/distributor'],
    }

    def initCtx(self):
        pass

    def on_enqueue(self):
        CommonBuildTask.on_enqueue(self)

    def on_execute(self):
        try:
            CommonBuildTask.on_execute(self)
        except OSError as e:
            logging.info(traceback.format_exc())
            raise SandboxTaskFailureError('%s' % e)
        except SandboxSvnIncorrectUrl as e:
            logging.error(traceback.format_exc())
            url = self.ctx[consts.ARCADIA_URL_KEY]
            rev = Arcadia.parse_url(url).revision
            if rev:
                url = url.replace('@'+str(rev), '')
                rev1 = Arcadia.get_revision(url)
                if (int(rev) - int(rev1)) < 2000:
                    raise SandboxTaskUnknownError('cannot find revision %s, found %s' % (rev, rev1))
            raise SandboxSvnIncorrectUrl("cannot get svn: "+str(e))

        if self._target_enabled('distributor'):
            binaries_dir = self.abs_path(self.LOCAL_BIN_DIR)
            distr_dir = os.path.join(binaries_dir, 'distributor')
            make_folder(distr_dir)
            for p in self.TARGETS_MULTIPLE['distributor']:
                copy_path(self.abs_path(self.LOCAL_RELEASE_DIR, p), os.path.join(distr_dir, os.path.basename(p)))
            self.create_resource(self.descr + ' ('+'distributor'+')', distr_dir, resource_types.DISTRIBUTOR, self.arch)

        self.ctx['libs_res_id'] = {}
        if not self.ctx.get('build_without_libs', False):
            rel_dir = self.abs_path(self.LOCAL_RELEASE_DIR)
            bin_dir = self.abs_path(self.LOCAL_BIN_DIR)
            for _, targ, p in self.get_target_resources():
                if self._target_enabled(targ) and targ in self.TARGET_LIBS:
                    libs_dir = os.path.join(bin_dir, targ + '_libs')
                    make_folder(libs_dir)

                    for lib in self.TARGET_LIBS[targ]:
                        lib_src = os.path.join(rel_dir, lib[1])
                        copy_path(os.path.join(lib_src, lib[0]), os.path.join(libs_dir, lib[0]))
                    if len(self.TARGET_LIBS[targ]) == 1 and targ == 'dsclient':
                        resource_path = os.path.join(libs_dir, self.TARGET_LIBS[targ][0][0])
                        self.ctx['libs_res_id'][targ] = self._create_resource('libs for %s' % targ, resource_path, resource_types.RTYSERVER_CLIENT_LIBS, True).id
                    else:
                        self.ctx['libs_res_id'][targ] = self._create_resource('libs for %s' % targ, libs_dir, resource_types.RTYSERVER_CLIENT_LIBS, True).id

        def ext_branch(url):
            if 'trunk' in url:
                return 'trunk'
            bb = url.find('saas/')
            if bb == -1:
                return 'nobranch'
            return url[bb:bb+12]

        for resource in self.list_resources():
            if resource.type.name == 'TASK_LOGS':
                continue
            channel.sandbox.set_resource_attribute(resource.id, 'branch',
                                                   ext_branch(self.ctx[consts.ARCADIA_URL_KEY]))

    def on_release(self, additional_parameters):
        channel.task = self
        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)


__Task__ = BuildRtyserverUtils
