import os

import sandbox.common.types.client as ctc

import sandbox.projects.common.gnupg as gnupg
import sandbox.projects.common.utils as utils
import sandbox.projects.CreatePackage as debian

from sandbox.sandboxsdk import ssh
from sandbox.sandboxsdk import svn
from sandbox.sandboxsdk import task
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import environments


SALMON = os.path.join('solomon', 'j')


class BuildSalmon(task.SandboxTask):
    type = 'BUILD_SALMON'

    client_tags = ctc.Tag.LINUX_PRECISE

    environment = [environments.SvnEnvironment()]

    class Revision(parameters.SandboxIntegerParameter):
        name = 'revision'
        description = 'Svn revision'

    class Version(parameters.SandboxStringParameter):
        name = 'pkg_version'
        description = 'Package version'
        default_value = None

    class GpgPrivate(parameters.SandboxStringParameter):
        name = 'gpg_private_key'
        description = 'Gpg private key vault'
        default_value = 'robot-test-kiwi-gpg-private'

    class GpgPublic(parameters.SandboxStringParameter):
        name = 'gpg_public_key'
        description = 'Gpg public key vault'
        default_value = 'robot-test-kiwi-gpg-public'

    class SshPrivate(parameters.SandboxStringParameter):
        name = 'ssh_private_key'
        description = 'Ssh private key'
        default_value = 'robot-test-kiwi-ssh'

    class RepoKeyId(parameters.SandboxStringParameter):
        name = 'repo_key_id'
        description = 'Repo key id'
        default_value = 'Kiwi Project (Functional testing) <robot-test-kiwi@yandex-team.ru>'

    class RepoName(parameters.SandboxStringParameter):
        name = 'repo_name'
        description = 'Repo name'
        default_value = 'search'

    input_parameters = [
        Revision,
        Version,
        GpgPrivate,
        GpgPublic,
        SshPrivate,
        RepoKeyId,
        RepoName,
    ]

    def on_execute(self):
        url = svn.Arcadia.trunk_url(revision=self.ctx['revision'] or None)

        if self.ctx['pkg_version']:
            ver = self.ctx['pkg_version']

        else:
            ver = '.'.join(map(str, [svn.Arcadia.get_revision(url), utils.get_short_branch_name(url), self.id]))

        self.set_info('building deb packages yandex-solomon-*=' + ver)

        source_root = svn.Arcadia.get_arcadia_src_dir(url)

        pkg = debian.deb.Package(os.path.join(source_root, SALMON))

        pkg.set_names()
        pkg.main_name = 'yandex-solomon'
        pkg.set_version(ver)
        pkg.set_install_files()

        with gnupg.GpgKey(self, self.owner, self.ctx['gpg_private_key'], self.ctx['gpg_public_key']):
            pkg.mk_changelog(self.ctx['repo_key_id'])

            pkg.build(self.ctx['repo_key_id'], preserve_env=True)

            with ssh.Key(self, self.owner, self.ctx['ssh_private_key']):
                pkg.upload(self.ctx['repo_name'], debian.util.get_login_from_email(self.ctx['repo_key_id']))


__Task__ = BuildSalmon
