# -*- coding: utf-8 -*-

import os
import time
import json
import shutil
import logging
import datetime

from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.svn import Arcadia
from sandbox.sandboxsdk.paths import make_folder
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.errors import SandboxTaskFailureError

from sandbox.projects import resource_types

import sandbox.sandboxsdk.parameters as sb_params
from sandbox.projects.common import error_handlers as eh
from sandbox.projects.common import utils
from sandbox.projects.common import context_managers
from sandbox.projects.common.build import ArcadiaTask as ArcTask
from sandbox.projects.common.build import dependency_banner
import sandbox.projects.common.build.parameters as build_params
import sandbox.projects.common.constants as consts
from sandbox.projects.logs.resources import USER_SESSIONS_BINARIES
import sandbox.projects.balancer.resources as balancer_resources
import sandbox.projects.websearch.middlesearch.resources as ms_resources
import sandbox.projects.websearch.begemot.resources as bg_resources
import sandbox.projects.websearch.rpslimiter.resources as rpslimiter_resources
import sandbox.projects.websearch.upper.resources as upper_resources
import sandbox.projects.app_host.resources as app_host_resources
import sandbox.projects.younglings.tutor.resources as younglings_resources
import sandbox.projects.ydo.backend.YdoBackendApphostSource.resource_types as ydo_resources
from sandbox.projects.ydo import YdoDatabaseConfig, YdoDatabaseTestConfig
import sandbox.projects.OrgCollections.resources as org_collections_resources
import sandbox.projects.BuildIznankaCompanyUrlsTrie.resources as iznanka_resources
import sandbox.projects.VideoSearch.video_resource_types as vid_base_resources
import sandbox.projects.woland.resources as woland_resources
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.common.arcadia import sdk_tested_parts as stp
from sandbox.projects.common.nanny import nanny
from sandbox.projects.images.idx_ops import resources as images_idx_ops_resources
from sandbox.projects.geosearch import resource_types as geo_types
from sandbox.projects.geosuggest import resources as geosuggest_types
from sandbox.projects.barnavig import resources as barnavig_resources


def get_common_build_params(group_name='Common build params'):

    class BuildParams(object):

        class KeepReleaseDir(parameters.SandboxBoolParameter):
            name = 'keep_release_dir'
            description = 'Keep releases directory'
            group = group_name
            default_value = False

        class ExtraBuildOptions(parameters.SandboxStringParameter):
            name = 'extra_build_options'
            description = 'Extra ymake build options (json)'
            group = group_name
            default_value = ''

        class PGOProfile(parameters.ResourceSelector):
            name = 'pgo_profile_resource_id'
            description = 'PGO profile (optional)'
            group = group_name

        params = [
            ArcTask.UseObjCache,
            KeepReleaseDir,
            ExtraBuildOptions,
            dependency_banner.BannedDependencies,
            dependency_banner.WhitelistedDependencies,
            PGOProfile,
        ]

    return BuildParams


_BUILD_PARAMS = get_common_build_params()
BASE_BUILD_GROUP_NAME = 'Base build params'


class ClearBuild(sb_params.SandboxBoolParameter):
    name = consts.CLEAR_BUILD_KEY
    description = 'Clear build'
    default_value = True
    group = BASE_BUILD_GROUP_NAME


class BuildSearch(nanny.ReleaseToNannyTask, ArcTask.ArcadiaTask):
    """
        **Описание**

            Общий таск для сборки поисковых и околопоисковых програм и библиотек.

        **Параметры**

        * ``{} projects``:  Проекты для сборки. Можно выбрать опции `build_all`
            (собрать весь проект) или `choose` (выбрать)
        * ``---> build {}``: Подполя, которые появляются при выборе опции `choose`.
            Позволяют собирать проект не целиком, а только то, что необходимо.
    """
    type = 'BUILD_SEARCH'

    build_search_params = [
        k for k in build_params.get_common_build_params()
        if k.name != consts.CLEAR_BUILD_KEY
    ] + _BUILD_PARAMS.params + [
        build_params.UseArcadiaApiFuseAsDefault,
        build_params.UseArcInsteadOfArcadiaApiAsDefault,
        build_params.AllowArcadiaApiFallback,
        ClearBuild,
    ]

    execution_space = 70000  # 70 Gb

    build_fields = {
        'search_projects': [
            'app_host_ops',
            'balancer',
            'basesearch',
            'basesearch_fuzzy',
            'video_basesearch_fuzzy',
            'begemot',
            'begemot_fuzzy',
            'blender_fuzzy',
            'compressed_factors_decoder',
            'directmetasearch',
            'dump_to_indexerproxy',
            'entitysearch',
            'entitysearch_converter',
            'entitysearch_dbsubset',
            'evlog_cachehit',
            'evlog_sample',
            'evlogdump',
            'factsbasesearch',
            'factsbasesearch_ua',
            'fresh_ranking_middlesearch',
            'yt',
            'http_cache',
            'idx_ops',
            'idx_ops_images',
            'idx_ops_video',
            'idx_qdump',
            'idx_qloss',
            'idx_query',
            'intsearch',
            'log_analyze',
            'matrixnet',
            'middlesearch',
            'misc_middlesearch',
            'music_wizard_data_converter',
            'nanny_tidiness',
            'news_basesearch',
            'news_basesearch_old',
            'peoplesearch',
            'peopless',
            'pool_merge_filter',
            'printwzrd',
            'proto_pool_reader',
            'pool_comparer',
            'pool_converter',
            'ranking_itditp_middlesearch',
            'ranking_middlesearch',
            'ranking_middlesearch_fuzzy',
            'realsearch',
            'request_sampler',
            'rpslimiter',
            'resinfodaemon',
            'rtyserver',
            'search_meta_ut',
            'sport_proxy',
            'stat_collector',
            'stat_fetcher',
            'upperapache',
            'video_middlesearch',
            'videosearch',
            'videortyserver',
            'vitrina_basesearch',
            'vitrina_base.cfg',
            'vitrina_metasearch',
            'wizard',
            'wizard_fuzzy',
            'wizard_unittest',
            'woland',
            'grpc_client',
        ],
        'miscellaneous_projects': [
            'appsearch_cleandescr_dll',
            'archiver',
            'clickdaemon',
            'clickdaemon.keys',
            'cproxy',
            'ddumper',
            'dexecutor',
            'dneher',
            'dolb_dumper',
            'dplanner',
            'dsindexer',
            'dumpwork',
            'encyc_report',
            'firehose_feeder',
            'firehose_proxy',
            'gztcompiler',
            'hashem',
            'hilitedaemon',
            'infotest',
            'lboost_ops',
            'logmux_collector',
            'microform_tool',
            'middle_userdown_mockup',
            'mrcollector',
            'mx_ops',
            'news_dehtmld',
            'news_feed_finder',
            'news_undup_server',
            'poetrylover',
            'printkeys',
            'printurls',
            'check_factor_usage',
            'remorphc',
            'rus_lang',
            'slave_newsd',
            'thumbdaemon',
            'triecompiler',
            'userspy',
            'verdict_basesearch',
            'verdict_report',
            'y_local_env',
            'yabar_word_create_index',
            'xfactor-yt',
        ],
        'robot_projects': [
            'fut',
            'geturl',
            'spider',
            'zora',
        ],
        'report_projects': [
            'comm_create',
            'report_bundle',
            'report_package',
        ],
        'antirobot_projects': [
            'antirobot_daemon',
            'antirobot_evlogdump',
        ],
        'mapreduce_projects': [
            'mapreduce',
            'mapreduce-yt',
            'mr_bounds',
            'mr_cat',
            'mr_cp',
            'mr_data_manip',
            'mr_diff',
            'mr_du',
            'mr_find',
            'mr_get_keys',
            'mr_grep',
            'mr_head',
            'mr_hist',
            'mr_ls',
            'mr_mv',
            'mr_rm',
            'mr_sample',
            'mr_set_ops',
            'mr_sort',
            'mr_touch',
            'mr_uniq',
            'mr_wc',
            'querydata_indexer',
            'quality_user_profiles',
        ],
        'younglings_projects': [
            'education',
            'tutor_html_adminka',
            'tutor_apphost_adminka',
            'tutor_morda_cards'
        ],
        'images_projects': [
            'images_middlesearch',
        ],
        'music_projects': [
            'audio_middlesearch',
            'audiomatch_middlesearch',
            'audiosearch',
            'audiomatchsearch',
            'festsearch',
            'music_middlesearch',
            'musicsearch',
            'musicsearch_stable',
            'yamrecsearch',
            'yamrec_middlesearch',
        ],
        'addressnip_projects': [
            'addressnip_index',
            'addressnip_search',
        ],
        'geosearch_projects': [
            'extevlogproc',
            'geoadvertindexer',
            'geofastfeaturesindexer',
            'geobasesearch',
            'geobusinessindexer',
            'geobusinessdatafilters',
            'geometasearch',
            'geopoiindexer',
            'svdgenerator',
            'geodaemon',
            'maps_shlib_relevance_minimal',
            'geourlsextractor',
            'geowebannmrfilter',
            'geowebannindexer',
            'hypotheses_from_backa',
            'process_preparat',
            'calc_similar_orgs',
            'calc_rating',
            'main_rubric_classifier',
            'prepare_for_upload',
            'printmms',
            'qaas',
            'iznanka_urls_extractor',
            'social_profiles_merger',
            'social_facts_extractor',
            'social_actual_profiles_builder',
            'social_events_snippet_builder',
            'geobasesearch_config',
            'count_factors',
            'static_factors_downloader',
            'geosearch_standalone_indexer',
            'geosearch_index_downloader',
            'geosearch_sitedata_downloader',
            'geosearch_yt_url_parser',
            'geosearch_yt_indexer_configs',
            'geosearch_yt_data_preparer',
            'geocollectionsindexer',
            'geosearch_objects_indexer',
            'geosearch_panther_sq_converter',
            'geosearch_panther_attr_sq_converter',
            'geosearch_panther_idx_converter',
            'geosearch_dynamic_formula',
            'fast_export_collect_changes',
            'fast_export_fresh_states_logger',
            'geosnippets_monitoring',
            'balancer_builder',
            'bad_url_preparer',
            'geosuggestd',
        ],
        'related_search_projects': [
            'relsearch_basesearch',
            'relsearch_libreport',
        ],
        'serpapi_projects': [
            'serpapi_index',
            'serpapi_search',
        ],
        'userdata_projects': [
            'mr_sample_mirrors',
            'mr_sample_pool',
            'querydata_viewer',
            'upper_logs_tests',
            'userdata_view',
        ],
        'user_sessions_projects': sorted(USER_SESSIONS_BINARIES.keys()),
        'ydo_projects': [
            'ydo_backend',
            'ydo_backend_request_extractor',
            'ydo_db_prod_config',
            'ydo_db_test_config',
            'ydo_schemas',
        ]
    }

    input_parameters = build_search_params + ArcTask.gen_input_params(build_fields)

    TARGET_PATH_TO_NAME_MAP = {
        'extsearch/geo/indexer/business/yandex-geosearch-indexer-business': 'geobusinessindexer',
        'extsearch/geo/saas/data_filters': 'geobusinessdatafilters',
        'extsearch/geo/indexer/advert_menu_tags/yandex-geosearch-indexer-advert': 'geoadvertindexer',
        'extsearch/geo/indexer/fast_features/yandex-geosearch-indexer-fast-features': 'geofastfeaturesindexer',
        'extsearch/geo/indexer/poi/yandex-geosearch-indexer-poi': 'geopoiindexer',
        'extsearch/geo/kernel/config/business': 'geobasesearch_config',
        'extsearch/geo/kernel/static_factors_yt/downloader/static_factors_downloder': 'static_factors_downloder',
        'extsearch/geo/indexer/collections/yandex-geosearch-indexer-collections': 'geocollectionsindexer',
    }

    """
        TARGET_RESOURCES consists of three fields: resource_type, target, target_path
        resource_type - type of the target resource
        target_path - final path of the built target resource.
        basename of the target path is the name of target resource file.
        dirname of the target path is the path to checkout from arcadia
    """
    TARGET_RESOURCES = [
        (resource_types.STAT_COLLECTOR_BINARY, 'quality/ab_testing/stat_collector/stat_collector'),
        (resource_types.STAT_FETCHER_BINARY, 'quality/ab_testing/stat_fetcher/stat_fetcher'),
        (resource_types.CREATE_SESSIONS_YT_PYTHON_EXECUTABLE, 'yt/python/yt/wrapper/bin/yt_make/yt'),
        (resource_types.RTYSERVER_EXECUTABLE, 'saas/rtyserver/rtyserver'),
        (resource_types.NEWS_BASESEARCH_EXECUTABLE, 'news_basesearch', 'extsearch/news/base/newssearch/newssearch'),
        (resource_types.NEWS_BASESEARCH_EXECUTABLE, 'news_basesearch_old', 'search/daemons/newssearch/newssearch'),
        (resource_types.SEARCH_META_UT_EXECUTABLE, 'search_meta_ut', 'search/meta/ut/search-meta-ut'),
        (ms_resources.IntsearchExecutable, 'search/daemons/intsearch/intsearch'),
        (resource_types.MISC_MIDDLESEARCH_EXECUTABLE, 'search/daemons/misc_middlesearch/misc_middlesearch'),
        (resource_types.FRESH_RANKING_MIDDLESEARCH_EXECUTABLE,
         'extsearch/fresh/meta/fresh_ranking_middlesearch/fresh_ranking_middlesearch'),
        (resource_types.IMAGES_MIDDLESEARCH_EXECUTABLE, 'images_middlesearch'),
        (resource_types.YAMRECSEARCH_EXECUTABLE, 'yamrecsearch', 'extsearch/audio/yamrec/basesearch/yamrecsearch/yamrecsearch'),
        (resource_types.YAMRECMIDDLESEARCH_EXECUTABLE, 'yamrec_middlesearch', 'extsearch/audio/yamrec/middlesearch/daemon/yamrecmiddle'),
        (resource_types.MUSICSEARCH_EXECUTABLE, 'audiomatchsearch', 'extsearch/audio/match/base/audiomatchsearch/audiomatchsearch'),
        (resource_types.MUSIC_MIDDLESEARCH_EXECUTABLE, 'audiomatch_middlesearch', 'extsearch/audio/meta/daemon/audiomatchmeta'),
        (resource_types.DIRECTMETASEARCH_EXECUTABLE, 'specsearches/directtier/directmetasearch/directmetasearch'),
        (resource_types.YSERVER_2011_03_EXECUTABLE, 'yserver_2011_03', 'search/daemons/yserver/yserver'),
        (resource_types.VIDEOSEARCH_EXECUTABLE, 'extsearch/video/base/videosearch/videosearch'),
        (resource_types.VIDEORTYSERVER_EXECUTABLE, 'extsearch/video/base/videortyserver/videortyserver'),
        (resource_types.PEOPLESS_EXECUTABLE, 'extsearch/people/daemon/peopless'),
        (younglings_resources.EDUCATION_EXECUTABLE, 'education', 'extsearch/younglings/education/education'),
        (younglings_resources.TUTOR_HTML_ADMINKA_EXECUTABLE, 'tutor_html_adminka', 'extsearch/younglings/tutor_admin/html_server/tutor_html_adminka'),
        (younglings_resources.TUTOR_APPHOST_ADMINKA_EXECUTABLE, 'tutor_apphost_adminka', 'extsearch/younglings/tutor_admin/apphost_server/tutor_apphost_adminka'),
        (younglings_resources.TUTOR_MORDA_CARDS_EXECUTABLE, 'tutor_morda_cards', 'extsearch/younglings/tutor_morda_cards/tutor_morda_cards'),
        (resource_types.HTTPCACHE_EXECUTABLE, 'extsearch/people/daemon/http_cache/http_cache'),
        (resource_types.LOGANALYZE_EXECUTABLE, 'log_analyze', 'search/tools/log-analyze/log-analyze'),
        (resource_types.EVLOGDUMP_EXECUTABLE, 'search/tools/evlogdump/evlogdump'),
        (resource_types.EVLOGSAMPLE_EXECUTABLE, 'evlog_sample', 'search/tools/evlog-sample/evlog-sample'),
        (resource_types.EVLOGCACHEHIT_EXECUTABLE, 'evlog_cachehit', 'search/tools/evlog-cachehit/evlog-cachehit'),
        (upper_resources.REQUEST_SAMPLER_EXECUTABLE, 'request_sampler', 'search/tools/request_sampler/request_sampler'),
        (rpslimiter_resources.RPS_LIMITER_EXECUTABLE, 'rpslimiter', 'search/priemka/rpslimiter/bin/rpslimiter'),
        (upper_resources.APP_HOST_OPS_EXECUTABLE, 'app_host_ops', 'search/tools/app_host_ops/app_host_ops'),
        (balancer_resources.BALANCER_EXECUTABLE, 'balancer/daemons/balancer/balancer'),
        (resource_types.RESINFO_DAEMON_EXECUTABLE, 'resinfodaemon', 'yweb/webdaemons/resinfod/resinfod'),
        (resource_types.DBG_DOLBILKA_EXECUTABLE, 'quality/query_pool/dbg_dolbilka/dbg_dolbilka'),
        (resource_types.REMOTE_WIZARD, 'web/daemons/wizard/wizard'),
        (resource_types.ENTITY_SEARCH_EXECUTABLE, 'web/daemons/entitysearch/entitysearch'),
        (
            resource_types.ENTITY_SEARCH_DBSUBSET_EXECUTABLE,
            'entitysearch_dbsubset',
            'search/wizard/entitysearch/tools/dbsubset/dbsubset'
        ),
        (
            resource_types.ENTITY_SEARCH_CONVERTER,
            'entitysearch_converter',
            'search/wizard/entitysearch/tools/convertor/convertor'
        ),
        (resource_types.MUSIC_WIZARD_DATA_CONVERTER, 'music_wizard_data_converter',
         'yweb/music/convert_wizard_data/convert_wizard_data'),
        (resource_types.PRINTWZRD, 'tools/printwzrd/printwzrd'),
        (resource_types.WIZARD_UNITTEST, 'unittest/wizard/wizard_ut'),
        (resource_types.MATRIXNET_EXECUTABLE, 'quality/relev_tools/matrixnet/matrixnet'),
        (resource_types.MX_OPS_EXECUTABLE, 'quality/relev_tools/mx_ops/mx_ops'),
        (resource_types.INFOTEST_EXECUTABLE, 'quality/relev_tools/tests/infotest/infotest'),
        (resource_types.LBOOST_OPS_EXECUTABLE, 'quality/relev_tools/lboost_ops/main/lboost_ops'),
        (resource_types.XFACTOR_YT_EXECUTABLE, 'quality/relev_tools/xfactor-yt/bin/xfactor-yt'),
        (resource_types.MIDDLE_USERDOWN_MOCKUP_EXECUTABLE,
         'quality/query_pool/middle_userdown_mockup/middle_userdown_mockup'),
        (resource_types.MRCOLLECTOR_EXECUTABLE, 'yweb/pumpkin/mrcollector/mrcollector'),
        (resource_types.KIDSEARCH_DESCR_CLEANER, 'appsearch_cleandescr_dll',
            'yweb/kidsearch/appsearch_cleandescr/processor_dll/libkidsearch-appsearch_cleandescr-processor_dll.so'),
        (resource_types.WIZARD_DATA_COMPILER, 'gztcompiler', ['tools/gztcompiler/gztcompiler',
                                                              'dict/gazetteer/compiler/gztcompiler']),
        (resource_types.REMORPH_DATA_COMPILER, 'tools/remorphc/remorphc'),
        (resource_types.POETRYLOVER_EXECUTABLE, 'poetrylover', 'specsearches/poetry/searchdaemon/poetry-daemon'),
        (resource_types.HILITEDAEMON_EXECUTABLE, 'yweb/webdaemons/hilitedaemon/hilitedaemon'),
        (resource_types.CLICKDAEMON_EXECUTABLE, 'yweb/webdaemons/clickdaemon/clickdaemon'),
        (resource_types.CLICKDAEMON_KEYS, 'clickdaemon.keys', 'web/report/data/clickdaemon.keys'),
        (barnavig_resources.USERSPY_EXECUTABLE, 'web/daemons/userspy/collector/userspy'),
        (resource_types.ENCYC_REPORT, 'encyc_report', 'specsearches/slovari/report/libencyc_report_shared.so'),
        (resource_types.RUSLANG_SLOVAR, 'rus_lang', 'specsearches/slovari/rus_lang/librus_lang_shared.so'),
        (resource_types.VERDICT_REPORT, 'verdict_report', 'specsearches/verdict/report/cpp/libverdictreport.so'),
        (resource_types.MICROFORM_TOOL_EXECUTABLE, 'yweb/structhtml/microform/tool/microform_tool'),

        (resource_types.ROBOT_SPIDER, 'yweb/robot/spider/spider'),
        (resource_types.ROBOT_ZORRO, 'yweb/robot/zorro/zora'),
        (resource_types.ROBOT_FUT, 'yweb/robot/fut/fut'),
        (resource_types.ROBOT_GETURL, 'yweb/webutil/geturl/geturl'),

        (resource_types.M2NSORT_EXECUTABLE, 'yweb/robot/m2nsort/main/m2nsort'),

        (resource_types.REPORT_COMM_CREATE, 'tools/comm_create/comm_create'),
        (resource_types.REPORT_BUNDLE, 'report_bundle', 'bindings/perl/report-bundle/libReportBundle_shared.so'),

        (resource_types.DUMPWORK_EXECUTABLE, 'yweb/realtime/indexer/tools/dumpwork/dumpwork'),
        (resource_types.CPROXY_EXECUTABLE, 'tools/cproxy/cproxy'),

        (resource_types.ANTIROBOT_DAEMON_EXECUTABLE, 'antirobot/daemon/antirobot_daemon'),
        (resource_types.ANTIROBOT_EVLOGDUMP_EXECUTABLE, 'antirobot/tools/evlogdump/antirobot_evlogdump'),
        (resource_types.DOLB_DUMPER_EXECUTABLE, 'dolb_dumper', 'yweb/querydata/tools/dolb_dumper/dolb_dumper'),
        (resource_types.DSINDEXER_EXECUTABLE, 'dsindexer', 'tools/dsindexer/dsindexer'),

        (resource_types.MAPREDUCE_EXECUTABLE, 'quality/mapreduce/mapreduce'),
        (resource_types.MAPREDUCE_YT_EXECUTABLE, 'yt/python/yt/wrapper/bin/mapreduce-yt_make/mapreduce-yt'),

        (resource_types.QUERYDATA_INDEXER, 'yweb/querydata/querydata_indexer/querydata_indexer'),
        (resource_types.QUERYDATA_VIEWER_EXECUTABLE, 'yweb/querydata/querydata_viewer/querydata_viewer'),

        (resource_types.QUALITY_USER_PROFILES, 'quality_user_profiles',
         'quality/pers/user_profiles/get_user_profiles'),

        (resource_types.Y_LOCAL_ENV, 'y_local_env', 'search/tools/y-local-env/y-local-env'),

        (resource_types.PRINTURLS_EXECUTABLE, 'tools/printurls/printurls'),
        (resource_types.IDX_OPS_VIDEO_EXECUTABLE, 'extsearch/video/tools/idx_ops/idx_ops_video'),

        (resource_types.SLAVE_NEWSD_EXECUTABLE, 'yweb/news/daemon/slave/slave_newsd'),
        (resource_types.NEWS_UNDUP_SERVER_EXECUTABLE, 'news_undup_server', 'yweb/news/undup/server/undup-server'),
        (resource_types.NEWS_DEHTMLD_EXECUTABLE, 'news_dehtmld', 'yweb/news/dehtmld/dehtmld'),
        (resource_types.NEWS_FEED_FINDER_EXECUTABLE, 'news_feed_finder',
         'yweb/news/skybit/modules/feed_finder/feed_finder'),
        (resource_types.HASHEM_EXECUTABLE, 'search/tools/hashem/hashem'),
        (resource_types.FIREHOSE_PROXY_EXECUTABLE, 'firehose_proxy', 'yweb/delivery/firehose/proxy/proxy'),
        (resource_types.FIREHOSE_FEEDER, 'firehose_feeder', 'yweb/delivery/daemons/feeder/feeder_daemon'),
        (resource_types.ARCHIVER_TOOL_EXECUTABLE, 'tools/archiver/archiver'),
        (resource_types.RELEV_FML_UNUSED_EXECUTABLE, 'search/tools/check_factor_usage/check_factor_usage'),

        (resource_types.LOGMUX_COLLECTOR, 'logmux_collector', 'quality/logs/logmux/collector/logmux-collector'),
        (resource_types.ADDRESSNIP_SEARCH, 'addressnip_search', 'extsearch/addressnip/base/searcher/addr_search'),
        (resource_types.ADDRESSNIP_INDEX, 'addressnip_index', 'extsearch/addressnip/indexer/addr_index'),

        (resource_types.GEOBASESEARCH_EXECUTABLE, 'extsearch/geo/base/geobasesearch/geobasesearch'),
        (resource_types.GEOBUSINESSDATAFILTER_EXECUTABLE, 'extsearch/geo/saas/data_filters'),
        (resource_types.GEOADVERTINDEXER_EXECUTABLE, 'extsearch/geo/indexer/advert_menu_tags/yandex-geosearch-indexer-advert'),
        (resource_types.GEO_FAST_FEATURES_INDEXER_EXECUTABLE,
         'extsearch/geo/indexer/fast_features/yandex-geosearch-indexer-fast-features'),
        (resource_types.GEOPOIINDEXER_EXECUTABLE, 'extsearch/geo/indexer/poi/yandex-geosearch-indexer-poi'),
        (resource_types.GEOSEARCH_STATIC_FACTORS_EXECUTABLE, 'count_factors', 'extsearch/geo/tools/count_factors/count_factors'),
        (resource_types.GEOSEARCH_STATIC_DOWNLOADER_EXECUTABLE, 'static_factors_downloader', 'extsearch/geo/kernel/static_factors_yt/downloader/static_factors_downloder'),
        (resource_types.GEOSEARCH_GOLOVAN_EXTEVLOGPROC_EXECUTABLE, 'extevlogproc',
         'search/geo/tools/golovan/extevlogproc/extevlogproc'),
        (resource_types.MAPS_SVD_GENERATOR_EXECUTABLE, 'svdgenerator',
         'extsearch/geo/tools/generatesvd/generatesvd'),
        (resource_types.MAPS_RELEVANT_URLS_EXTRACTOR_EXECUTABLE, 'geourlsextractor',
         'extsearch/geo/tools/extracturls/extracturls'),
        (resource_types.GEOBASESEARCH_CONFIG, 'geobasesearch_config', 'extsearch/geo/kernel/config/business'),
        (resource_types.PRINTMMS_EXECUTABLE, 'search/geo/tools/org1_wizard_by_aggregators/printmms/printmms'),
        (resource_types.GEOWEBANNMRFILTER_EXECUTABLE, 'geowebannmrfilter',
         'extsearch/geo/tools/web_annotation_ops/mr_filter/mr_filter'),
        (resource_types.GEOWEBANNINDEXER_EXECUTABLE, 'geowebannindexer',
         'extsearch/geo/indexer/web_annotation/indexer-web-annotation'),

        (iznanka_resources.IznankaUrlsExtractorExecutable, 'iznanka_urls_extractor', 'search/geo/tools/iznanka/get_company_urls/get_company_urls'),
        (resource_types.SOCIAL_PROFILES_MERGER_FROM_AURORA_EXECUTABLE, 'social_profiles_merger', 'search/geo/tools/social_links/merge_from_aurora/merge_from_aurora'),
        (resource_types.SOCIAL_PROFILES_FACTS_EXTRACTOR_EXECUTABLE, 'social_facts_extractor', 'search/geo/tools/social_links/extract_facts/extract_facts'),
        (resource_types.SOCIAL_ACTUAL_PROFILES_BUILDER_EXECUTABLE, 'social_actual_profiles_builder', 'search/geo/tools/social_links/build_actual_profiles/build_actual_profiles'),
        (resource_types.SOCIAL_EVENTS_SNIPPET_BUILDER_EXECUTABLE, 'social_events_snippet_builder', 'search/geo/tools/social_links/build_saas_snippets/build_saas_snippets'),
        (geo_types.GEOSEARCH_STANDALONE_INDEXER, 'geosearch_standalone_indexer', 'extsearch/geo/indexer/business_indexer_yt/standalone_indexer/standalone_indexer'),
        (geo_types.GEOSEARCH_INDEX_DOWNLOADER, 'geosearch_index_downloader', 'extsearch/geo/indexer/business_indexer_yt/download_index/download_index'),
        (geo_types.GEOSEARCH_YT_URL_PARSER, 'geosearch_yt_url_parser', 'extsearch/geo/indexer/business_indexer_yt/parse_urls_from_export_yt/parse_urls_from_export_yt'),
        (geo_types.GEOSEARCH_YT_INDEXER_CONFIGS, 'geosearch_yt_indexer_configs', 'extsearch/geo/indexer/business_indexer_yt/indexer_config/indexer_config.tar'),
        (geo_types.GEOSEARCH_YT_DATA_PREPARER, 'geosearch_yt_data_preparer', 'extsearch/geo/indexer/business_indexer_yt/prepare_data/prepare_data'),
        (geo_types.GEOSEARCH_OBJECTS_INDEXER, 'geosearch_objects_indexer', 'extsearch/geo/indexer/objects/yandex-geosearch-indexer-objects'),
        (geo_types.GEOSEARCH_PANTHER_SQ_CONVERTER, 'geosearch_panther_sq_converter', 'extsearch/geo/indexer/panther_sq_converter/panther_sq_conv'),
        (geo_types.GEOSEARCH_PANTHER_ATTR_SQ_CONVERTER, 'geosearch_panther_attr_sq_converter', 'extsearch/geo/indexer/panther_attr_sq_converter/panther_attr_sq_conv'),
        (geo_types.GEOSEARCH_PANTHER_IDX_CONVERTER, 'geosearch_panther_idx_converter', 'search/panther/tools/idx_convert/idx_convert'),
        (geo_types.GEOSEARCH_DYNAMIC_FORMULA, 'geosearch_dynamic_formula', 'extsearch/geo/base/formula/dynamic_mxnet_formulas/dynamic_mxnet_formulas'),
        (geo_types.FAST_EXPORT_COLLECT_CHANGES, 'fast_export_collect_changes', 'extsearch/geo/fast_export/collect_changes/collect_changes'),
        (geo_types.FAST_EXPORT_FRESH_STATES_LOGGER, 'fast_export_fresh_states_logger', 'extsearch/geo/fast_export/fresh_states_logger/fresh_states_logger'),
        (geo_types.FAST_EXPORT_SPEED_MEASURE, 'fast_export_speed_measure', 'extsearch/geo/fast_export/measure_speed/measure_speed'),
        (geo_types.GEOSEARCH_SNIPPETS_MONITORING, 'geosnippets_monitoring', 'search/geo/tools/task_manager/monitoring/geosnippets_monitoring'),
        (geo_types.GEOSEARCH_BAD_URLS_PREPARER, 'bad_urls_preparer', 'arcadia/extsearch/geo/tools/bad_urls_preparer/bad_urls_preparer'),
        (geo_types.ADDRS_TESTING_BALANCER_GENERATOR, 'balancer_builder', 'search/geo/tools/production/balancer/bin/balancer_builder'),
        (geosuggest_types.GEO_SUGGEST_WEBDAEMON, 'geosuggestd', 'quality/trailer/suggest/services/maps/SuggestFcgiD/geosuggestd'),

        (org_collections_resources.OrgCollectionsIndexerExecutable, 'extsearch/geo/indexer/collections/yandex-geosearch-indexer-collections'),

        (resource_types.SERPAPI_SEARCH, 'yweb/serpapi/core/searcher/sapi_search'),
        (resource_types.SERPAPI_INDEX, 'yweb/serpapi/core/indexer/sapi_index'),

        (resource_types.BIATHLON_PROXY_EXECUTABLE, 'extsearch/wizards/sport_proxy/bin/sport_proxy'),

        (resource_types.UPPER_LOGS_TESTS_EXECUTABLE,
         'quality/userdata/browser_tests/upper_logs_tests/upper_logs_tests'),

        (bg_resources.BEGEMOT_EXECUTABLE, 'web/daemons/begemot/begemot'),
        (woland_resources.WOLAND_EXECUTABLE, 'woland', 'search/tools/woland/woland'),
        (ms_resources.ItditpMiddlesearchExecutable, 'search/daemons/ranking_itditp_middlesearch/ranking_itditp_middlesearch'),
        (app_host_resources.AppHostGrpcClientExecutable, 'grpc_client', app_host_resources.AppHostGrpcClientExecutable.arcadia_build_path),

        (ydo_resources.YdoBackend, 'ydo_backend', 'ydo/server/apphost_source/bin/apphost_source'),
        (ydo_resources.YdoBackendRequestExtractor, 'ydo_backend_request_extractor', 'ydo/tools/extract_backend_request_source/extract_backend_request_source'),
        (YdoDatabaseConfig, 'ydo_db_prod_config', 'ydo/database/configs/prod_conf.json'),
        (YdoDatabaseTestConfig, 'ydo_db_test_config', 'ydo/database/configs/test_conf.json'),
        (ydo_resources.YdoSchemas, 'ydo_schemas', 'ydo/schema/schemas'),
    ]

    TARGET_RESOURCE_TYPES = [
        resource_types.VIDEO_RANKING_MIDDLESEARCH_EXECUTABLE,
        resource_types.TRIECOMPILER_EXECUTABLE,
        resource_types.PRINTKEYS_EXECUTABLE,
        resource_types.MR_BOUNDS,
        resource_types.MR_CAT,
        resource_types.MR_CP,
        resource_types.MR_DATA_MANIP,
        resource_types.MR_DIFF,
        resource_types.MR_DU,
        resource_types.MR_FIND,
        resource_types.MR_GET_KEYS,
        resource_types.MR_GREP,
        resource_types.MR_HEAD,
        resource_types.MR_HIST,
        resource_types.MR_LS,
        resource_types.MR_MV,
        resource_types.MR_RM,
        resource_types.MR_SAMPLE,
        resource_types.MR_SET_OPS,
        resource_types.MR_SORT,
        resource_types.MR_TOUCH,
        resource_types.MR_UNIQ,
        resource_types.MR_WC,
        images_idx_ops_resources.IDX_OPS_IMAGES_EXECUTABLE,
        resource_types.POOL_MERGE_FILTER_EXECUTABLE,
        resource_types.POOL_CONVERTER_EXECUTABLE,
        resource_types.PROTO_POOL_READER_EXECUTABLE,
        resource_types.POOL_COMPARER_EXECUTABLE,
        resource_types.IDX_QUERY_EXECUTABLE,
        resource_types.IDX_QDUMP_EXECUTABLE,
        resource_types.MAPS_SHLIB_RELEVANCE_MINIMAL,
        resource_types.GEOMETASEARCH_EXECUTABLE,
        resource_types.GEODAEMON_EXECUTABLE,
        resource_types.GEOQAAS_EXECUTABLE,
        resource_types.GEOBASESEARCH_CONFIG,
        resource_types.RELSEARCH_LIBREPORT,
        resource_types.RELSEARCH_BASE_EXECUTABLE,
        resource_types.VERDICT_BASESEARCH_EXECUTABLE,
        resource_types.FACTS_BASESEARCH_EXECUTABLE,
        resource_types.FACTS_BASESEARCH_EXECUTABLE_UA,
        resource_types.MR_SAMPLE_POOL_EXECUTABLE,
        resource_types.MR_FILTER_BY_POOL_EXECUTABLE,
        resource_types.MR_SAMPLE_MIRRORS_EXECUTABLE,
        resource_types.USERDATA_VIEW_EXECUTABLE,
        resource_types.COMPRESSED_FACTORS_DUMP_EXECUTABLE,
        ms_resources.RankingMiddlesearchExecutable,
        resource_types.BASESEARCH_EXECUTABLE,
        resource_types.BASESEARCH_EXECUTABLE_FUZZY,
        vid_base_resources.VIDEO_BASESEARCH_EXECUTABLE_FUZZY,
        vid_base_resources.VIDEO_VITRINA_BASESEARCH_EXECUTABLE,
        vid_base_resources.VIDEO_VITRINA_BASESEARCH_CONFIG,
        vid_base_resources.VIDEO_VITRINA_METASEARCH_EXECUTABLE,
        upper_resources.WebBlenderExecutableFuzzy,
        ms_resources.WebMiddlesearchExecutableFuzzy,
        bg_resources.BEGEMOT_EXECUTABLE_FUZZY,
        bg_resources.BEGEMOT_NANNY_TIDINESS,
        resource_types.WIZARD_EXECUTABLE_FUZZY,
        resource_types.IDX_OPS_EXECUTABLE,
        resource_types.IDX_QLOSS_EXECUTABLE,
        resource_types.SIMILAR_ORGS_HYPOS_SPRAV_EXECUTABLE,
        resource_types.SIMILAR_ORGS_PROCESS_PREPARAT_EXECUTABLE,
        resource_types.CALC_SIMILAR_ORGS_EXECUTABLE,
        resource_types.CALC_BUSINESS_RATING_EXECUTABLE,
        resource_types.CALC_MAIN_RUBRIC_EXECUTABLE,
        resource_types.CREATE_SIMILAR_ORGS_EXPORT_EXECUTABLE,
        resource_types.DUMP_TO_INDEXERPROXY_EXECUTABLE,
        resource_types.GEOSEARCH_STATIC_DOWNLOADER_EXECUTABLE,
        upper_resources.REQUEST_SAMPLER_EXECUTABLE,
        rpslimiter_resources.RPS_LIMITER_EXECUTABLE,
        upper_resources.APP_HOST_OPS_EXECUTABLE,
        ms_resources.ItditpMiddlesearchExecutable,
        app_host_resources.AppHostGrpcClientExecutable,
    ] + USER_SESSIONS_BINARIES.values()

    LOCAL_BIN_DIR = 'binaries'

    def on_enqueue(self):
        ArcTask.on_enqueue_input_params(self.ctx, self.build_fields)
        ArcTask.ArcadiaTask.on_enqueue(self)
        if self.ctx.get('build_bundle'):
            return
        self.do_on_enqueue()

    @staticmethod
    def __check_target_path(target_paths, arcadia_folder):
        """
            Проверить, существует ли путь у цели сборки.
            Если цель сборки указана как список или кортеж, выбрать первый существующий.

            :param target_paths: пути для сборки цели
            :param arcadia_folder: путь до аркадии
            :return: путь цели сборки в виде строки
            :rtype: str
        """
        if not (isinstance(target_paths, list) or isinstance(target_paths, tuple)):
            # приводим пути, не являющиеся списками, к спискам
            target_paths = [target_paths]
        for target_path in target_paths:
            logging.debug('Check build target path {0} in arcadia folder {1}'.format(target_path, arcadia_folder))
            full_target_path = os.path.join(arcadia_folder, os.path.dirname(target_path))
            if os.path.exists(full_target_path):
                return target_path
        raise SandboxTaskFailureError('Build target paths {0} do not exist in arcadia folder {1}'.format(
            target_paths, arcadia_folder
        ))

    def do_on_enqueue(self):
        dependency_banner.create_resources(self)

        for resource_type, target, target_path in self.get_target_resources():
            if self._target_enabled(target):
                if isinstance(target_path, list) or isinstance(target_path, tuple):
                    filename = os.path.basename(target_path[0])
                else:
                    filename = os.path.basename(target_path)
                resource_path = os.path.join(self.LOCAL_BIN_DIR, filename)
                resource_name = u'{} ({})'.format(self.descr, filename)
                attrs = None
                if 'testenv_database' in self.ctx:
                    attrs = {'testenv_database': self.ctx.get('testenv_database')}
                resource = self._create_resource(
                    resource_name, resource_path, resource_type,
                    arch=self.arch, attrs=attrs
                )
                self.ctx['{}_resource_id'.format(target)] = resource.id

    def _build_with_ya(self, make_only_list):
        self.release_dir = self.abs_path(self.LOCAL_RELEASE_DIR)

        def_flags_all = utils.get_or_default(self.ctx, build_params.DefinitionFlags)
        target_platform_flags = utils.get_or_default(self.ctx, build_params.TargetPlatformFlags)
        extra_build_options_str = utils.get_or_default(self.ctx, _BUILD_PARAMS.ExtraBuildOptions)
        extra_build_options = {}
        try:
            extra_build_options = json.loads(extra_build_options_str)
        except Exception as e:
            logging.error("Cannot parse extra build options '%s', got error: %s", extra_build_options_str, e)

        pgo_profile_resource_id = utils.get_or_default(self.ctx, _BUILD_PARAMS.PGOProfile)
        if pgo_profile_resource_id:
            extra_build_options["use_pgo"] = self.sync_resource(pgo_profile_resource_id)

        def_flags = stp.parse_flags(def_flags_all)
        def_flags['SANDBOX_TASK_ID'] = self.id
        build_type = utils.get_or_default(self.ctx, build_params.BuildType)
        build_system = self.ctx.get(consts.BUILD_SYSTEM_KEY)
        clear_build = self.ctx.get(consts.CLEAR_BUILD_KEY)
        sdk.do_build(
            build_system, self.arcadia_src_dir, make_only_list,
            build_type=build_type,
            def_flags=def_flags,
            results_dir=self.release_dir,
            clear_build=clear_build,
            target_platform_flags=target_platform_flags,
            **extra_build_options
        )

    def _build(self):
        enabled_targets = {}
        for _, target, target_path in self.get_target_resources():
            if self._target_enabled(target):
                enabled_targets[target] = self.__check_target_path(target_path, self.arcadia_src_dir)
        eh.ensure(enabled_targets, 'No targets for build are specified.')

        make_only_list = [os.path.dirname(target_path) for target_path in enabled_targets.values()]

        dependency_banner.check_dependencies(self, self.arcadia_src_dir, make_only_list)

        build_start_ts = int(time.time())
        build_start_time = datetime.datetime.now()

        self._build_with_ya(make_only_list)

        cpu_model = self.client_info['cpu'].lower()
        coeff = 1.0
        if '5645' in cpu_model:
            # old slow CPUs
            coeff = 0.55

        self.ctx['seconds_wasted_on_build'] = int((time.time() - build_start_ts) * coeff)
        self.ctx['time_wasted_on_build'] = str(datetime.datetime.now() - build_start_time)

        for target, target_path in enabled_targets.items():
            if self._target_enabled(target):
                self.copy_built_target(target, self.abs_path(os.path.join(self.release_dir, target_path)))

    def _get_src_dir(self):
        if utils.get_or_default(self.ctx, build_params.UseArcadiaApiFuseAsDefault):
            arcadia_url = self.ctx.get(consts.ARCADIA_URL_KEY, "")
            use_arc_instead_of_aapi = utils.get_or_default(self.ctx, build_params.UseArcInsteadOfArcadiaApiAsDefault)
            return sdk.mount_arc_path(arcadia_url, use_arc_instead_of_aapi=use_arc_instead_of_aapi)
        return context_managers.nullcontext(self.get_arcadia_src_dir())

    def do_execute(self):
        os.chdir(self.abs_path())
        binaries_dir = self.abs_path(self.LOCAL_BIN_DIR)
        make_folder(binaries_dir)
        arcadia_ctx = self._get_src_dir()
        with arcadia_ctx as arcadia_src_dir:
            self.arcadia_src_dir = arcadia_src_dir
            Arcadia.apply_patch(self.arcadia_src_dir, self.ctx.get('arcadia_patch'), self.abs_path())

            self.ctx['arcadia_revision'] = (
                utils.svn_revision(self.arcadia_src_dir) or utils.svn_revision(self.ctx.get(consts.ARCADIA_URL_KEY, ""))
            )
            self.ctx['last_change'] = utils.svn_last_change(self.arcadia_src_dir)
            self.fill_system_info()

            self._build()

            # cleanup
            if not utils.get_or_default(self.ctx, _BUILD_PARAMS.KeepReleaseDir):
                self.clean_release_dir(self.release_dir)

            self.set_info('build completed')

    def copy_built_target(self, target, source_path):
        logging.info('PATH %s', source_path)
        shutil.copy(source_path, channel.sandbox.get_resource(self._target_resource_id(target)).path)

    def get_short_task_result(self):
        return self.ctx.get('time_wasted_on_build', None)


__Task__ = BuildSearch
